/*
 * Projrct F-11 - Web SCADA for Java
 * Copyright (C) 2002 Freedom, Inc. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

package org.F11.scada.server.converter;

import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;

import org.F11.scada.WifeException;
import org.F11.scada.WifeUtilities;
import org.F11.scada.server.communicater.Environment;
import org.F11.scada.server.event.WifeCommand;

/**
 * WifeʐMW[MC3ER}h̎NXB
 */
public class MC3E implements Converter {
    /** WFINSR}hMpPbgőTCY\萔ł */
    public static final int DEFAULT_MC3E_PACKET_MAX_SIZE = 960 * 2;

    /** MC3ER}h wb_ */
    public static final int MC3E_COMMAND_HEADER_LENGTH = 11;

    /** MC3ER}h vf[^ */
    public static final int MC3E_COMMAND_REQDATA_LENGTH = 12;

    /** MC3ER}h IR[h */
    public static final int MC3E_COMMAND_ENDCODE_LENGTH = 2;

    /**  D f[^WX^(word)\܂B */
    public static final Integer D_AREA = new Integer(0);

    /**  M [GA(bit)\܂B */
    public static final Integer M_AREA = new Integer(1);

    /**  X ̓[GA(bit)\܂B */
    public static final Integer X_AREA = new Integer(2);

    /**  Y o̓[GA(bit)\܂B */
    public static final Integer Y_AREA = new Integer(3);

    /**  B N[GA(bit)\܂B */
    public static final Integer B_AREA = new Integer(4);

    /**  L b`[GA(bit)\܂B */
    public static final Integer L_AREA = new Integer(5);

    /**  ZR t@CWX^(word)\܂B */
    public static final Integer R_AREA = new Integer(10);

    /**  W NWX^(word)\܂B */
    public static final Integer W_AREA = new Integer(11);

    /**  ꃊ[GA(bit)\܂B */
    public static final Integer SM_AREA = new Integer(90);

    /** R}hʂ̃}bv */
    private static final Map memoryModeMap;
    static {
        memoryModeMap = new HashMap();
        memoryModeMap.put(D_AREA, new ComndKindWorddev(0xA8));
        memoryModeMap.put(M_AREA, new ComndKindBitdev(0x90));
        memoryModeMap.put(X_AREA, new ComndKindBitdev(0x9C));
        memoryModeMap.put(Y_AREA, new ComndKindBitdev(0x9D));
        memoryModeMap.put(B_AREA, new ComndKindBitdev(0xA0));
        memoryModeMap.put(L_AREA, new ComndKindBitdev(0x92));
        memoryModeMap.put(R_AREA, new ComndKindWorddev(0xB0));
        memoryModeMap.put(W_AREA, new ComndKindWorddev(0xB4));
        memoryModeMap.put(SM_AREA, new ComndKindBitdev(0x91));
    }

    /** wb_f[^ */
    private byte[] head = { (byte) 0x50, (byte) 0x00, (byte) 0x00, (byte) 0x00,
            (byte) 0xff, (byte) 0x03, (byte) 0x00 };

    /** CPUĎ^C} */
    private byte[] cpu = { (byte) 0x00, (byte) 0x00 };

    /**
     * ftHgRXgN^
     */
    public MC3E() {
    }

    /** ݒ肵AX|Xwb_ԂB */
    public byte[] setEnvironment(Environment device) {
        head[2] = (byte) device.getPlcNetNo();
        head[3] = (byte) device.getPlcNodeNo();
        cpu[0] = (byte) (device.getPlcWatchWait() % 0x100);
        cpu[1] = (byte) (device.getPlcWatchWait() / 0x100);

        byte[] resp = new byte[head.length];
        resp[0] = (byte) 0xd0;
        resp[1] = (byte) 0x00;
        resp[2] = head[2];
        resp[3] = head[3];
        resp[4] = head[4];
        resp[5] = head[5];
        resp[6] = head[6];
        return resp;
    }

    /** Ǎ݃R}hݒ肷B */
    public void setReadCommand(WifeCommand commdef) throws WifeException {
        mc3eCommand = READ_MC3ECOMMAND;
        mc3eCommand.setCommand(commdef, cpu, null);
    }

    /** ݃R}hݒ肷B */
    public void setWriteCommand(WifeCommand commdef, byte[] data)
            throws WifeException {
        mc3eCommand = WRITE_MC3ECOMMAND;
        mc3eCommand.setCommand(commdef, cpu, data);
    }

    /** R}h擾\H */
    public boolean hasCommand() {
        return mc3eCommand.hasCommand();
    }

    /** R}h쐬ÃR}h܂B */
    public void nextCommand(ByteBuffer sendBuffer) {
        sendBuffer.put(head);
        mc3eCommand.nextCommand(sendBuffer);
    }

    /** OsR}h쐬܂B */
    public void retryCommand(ByteBuffer sendBuffer) {
        sendBuffer.put(head);
        mc3eCommand.retryCommand(sendBuffer);
    }

    /** Mf[^ƎMf[^̐܂B */
    public WifeException checkCommandResponce(ByteBuffer recvBuffer)
            throws WifeException {
        byte[] err = { 0, 0 };
        if (recvBuffer.remaining() < MC3E_COMMAND_HEADER_LENGTH) {
            StringBuffer sb = new StringBuffer();
            sb.append("RecvData (");
            sb.append(WifeUtilities.toString(recvBuffer));
            sb.append(") is short!");
            return new WifeException(WifeException.WIFE_ERROR,
                    WifeException.WIFE_NET_RESPONCE_ERROR, sb.toString()
                            + mc3eCommand.toString());
        }
        // IR[h
        if (recvBuffer.get(9) != (byte) 0x00
                || recvBuffer.get(10) != (byte) 0x00) {
            StringBuffer sb = new StringBuffer();
            sb.append("End code error: ");
            sb.append(" RecvData (");
            sb.append(WifeUtilities.toString(recvBuffer));
            sb.append(")");
            return new WifeException(WifeException.WIFE_ERROR,
                    WifeException.WIFE_NET_RESPONCE_ENDCODE_ERROR, err, sb
                            .toString()
                            + mc3eCommand.toString());
        }
        // f[^
        int actualDataLen = recvBuffer.remaining() - MC3E_COMMAND_HEADER_LENGTH
                + MC3E_COMMAND_ENDCODE_LENGTH;
        int headerDataLen = ((int) recvBuffer.get(7) & 0x00ff)
                + (((int) recvBuffer.get(8) * 0x100) & 0x00ff00);
        if (actualDataLen != headerDataLen) {
            StringBuffer sb = new StringBuffer();
            sb.append("Header data length ").append(headerDataLen);
            sb.append(" != Actual data length ").append(actualDataLen);
            sb.append(" RecvData (");
            sb.append(WifeUtilities.toString(recvBuffer));
            sb.append(")");
            return new WifeException(WifeException.WIFE_ERROR,
                    WifeException.WIFE_NET_RESPONCE_CMND_ERROR, sb.toString());
        }
        int reqDataLen = mc3eCommand.getReqDataLength();
        if (actualDataLen - 2 != reqDataLen) {
            StringBuffer sb = new StringBuffer();
            sb.append("Request data length ").append(reqDataLen);
            sb.append(" != Actual data length ").append(actualDataLen);
            sb.append(" RecvData (");
            sb.append(WifeUtilities.toString(recvBuffer));
            sb.append(")");
            return new WifeException(WifeException.WIFE_ERROR,
                    WifeException.WIFE_NET_RESPONCE_CMND_ERROR, sb.toString());
        }
        return null;
    }

    /** Mf[^f[^擾܂B */
    public void getResponceData(ByteBuffer recvBuffer, ByteBuffer recvData) {
        mc3eCommand.getResponceData(recvBuffer, recvData);
    }

    /** ʐM̍ő咷Ԃ܂B */
    public int getPacketMaxSize(WifeCommand commdef) {
        return DEFAULT_MC3E_PACKET_MAX_SIZE / 2;
    }

    /**
     * \Ԃ܂B
     */
    public String toString() {
        return mc3eCommand.toString();
    }

    /**
     * R}hϊwp[NX̃C^[tFCXłB
     */
    private interface Mc3eCommand {
        /** R}hݒ肷B */
        public void setCommand(WifeCommand commdef, byte[] cpu, byte[] data)
                throws WifeException;

        /** R}h擾\H */
        public boolean hasCommand();

        /** R}h쐬ÃR}h܂B */
        public void nextCommand(ByteBuffer sendBuffer);

        /** OsR}h쐬܂B */
        public void retryCommand(ByteBuffer sendBuffer);

        /** R}h̍v擾܂B */
        public int getReqDataLength();

        /** Mf[^f[^擾܂B */
        public void getResponceData(ByteBuffer recvBuffer, ByteBuffer recvData);
    }

    /**
     * Ǎ݃R}hϊwp[NX̃CX^XłB
     */
    private final Mc3eCommand READ_MC3ECOMMAND = new Mc3eCommand() {
        /** CPUĎ^C} */
        private byte[] cpu;

        /** PLC */
        private byte memoryMode;

        /** PLCAhX */
        private long memoryAddress;

        /** ׂPLCf[^̎coCg */
        private int restByteLength = 0;

        /** PLCf[^oCg */
        private int thisByteLength = 0;

        /** foCXʃwp[NXւ̎Q */
        private ComndKind kind;

        /** R}h̃l^ݒ肷B */
        public void setCommand(WifeCommand commdef, byte[] cpu, byte[] data)
                throws WifeException {
            this.cpu = cpu;
            // foCXʃwp[NX
            kind = (ComndKind) memoryModeMap.get(new Integer(commdef
                    .getMemoryMode()));
            if (kind == null) {
                throw new WifeException(WifeException.WIFE_ERROR,
                        WifeException.WIFE_NET_COMMAND_ERROR,
                        "Not supported memory mode " + commdef.getMemoryMode());
            }
            memoryMode = kind.getMemKind();

            memoryAddress = commdef.getMemoryAddress();
            restByteLength = commdef.getWordLength() * 2;
            thisByteLength = 0;
        }

        /** R}h擾\H */
        public boolean hasCommand() {
            return 0 < restByteLength;
        }

        /** R}h쐬ÃR}h܂B */
        public void nextCommand(ByteBuffer sendBuffer) {
            long addr = kind.getAddress(memoryAddress);

            // foCXɈˑ钷𒴂Ă
            if (DEFAULT_MC3E_PACKET_MAX_SIZE < restByteLength) {
                thisByteLength = DEFAULT_MC3E_PACKET_MAX_SIZE;
                restByteLength -= DEFAULT_MC3E_PACKET_MAX_SIZE;
            } else {
                thisByteLength = restByteLength;
                restByteLength = 0;
            }

            // MC3Ewb_̍쐬
            int datalen = MC3E_COMMAND_REQDATA_LENGTH;
            sendBuffer.put((byte) (datalen % 0x100));
            sendBuffer.put((byte) (datalen / 0x100));
            sendBuffer.put(cpu);
            // R}hATuR}h
            sendBuffer.put((byte) 0x01);
            sendBuffer.put((byte) 0x04);
            sendBuffer.put((byte) 0x00);
            sendBuffer.put((byte) 0x00);
            // foCXAhX
            sendBuffer.put((byte) (addr % 0x100));
            sendBuffer.put((byte) ((addr / 0x100) % 0x100));
            sendBuffer.put((byte) ((addr / 0x10000) % 0x100));
            // foCXR[h
            sendBuffer.put(memoryMode);
            // 
            sendBuffer.put((byte) ((thisByteLength / 2) % 0x100));
            sendBuffer.put((byte) ((thisByteLength / 2) / 0x100));

            // ̃AhX
            memoryAddress += (thisByteLength / 2);
        }

        /** OsR}h쐬܂B */
        public void retryCommand(ByteBuffer sendBuffer) {
            long addr = kind.getAddress(memoryAddress - (thisByteLength / 2));

            // MC3Ewb_̍쐬
            int datalen = MC3E_COMMAND_REQDATA_LENGTH;
            sendBuffer.put((byte) (datalen % 0x100));
            sendBuffer.put((byte) (datalen / 0x100));
            sendBuffer.put(cpu);
            // R}hATuR}h
            sendBuffer.put((byte) 0x01);
            sendBuffer.put((byte) 0x04);
            sendBuffer.put((byte) 0x00);
            sendBuffer.put((byte) 0x00);
            // foCXAhX
            sendBuffer.put((byte) (addr % 0x100));
            sendBuffer.put((byte) ((addr / 0x100) % 0x100));
            sendBuffer.put((byte) ((addr / 0x10000) % 0x100));
            // foCXR[h
            sendBuffer.put(memoryMode);
            // 
            sendBuffer.put((byte) ((thisByteLength / 2) % 0x100));
            sendBuffer.put((byte) ((thisByteLength / 2) / 0x100));
        }

        /** R}h̍v擾܂B */
        public int getReqDataLength() {
            return thisByteLength;
        }

        /** Mf[^f[^擾܂B */
        public void getResponceData(ByteBuffer recvBuffer, ByteBuffer recvData) {
            for (int i = MC3E_COMMAND_HEADER_LENGTH; i < recvBuffer.remaining(); i += 2) {
                recvData.put(recvBuffer.get(i + 1));
                recvData.put(recvBuffer.get(i + 0));
            }
        }

        /**
         * ̃CX^X̕\Ԃ܂B
         */
        public String toString() {
            StringBuffer s = new StringBuffer();
            s.append("ReadMode:").append("\n");
            s.append("memoryMode:").append(memoryMode).append("\n");
            s.append("memoryAddress:").append(memoryAddress).append("\n");
            s.append("restByteLength:").append(restByteLength).append("\n");
            s.append("thisByteLength:").append(thisByteLength).append("\n");
            return s.toString();
        }
    };

    /**
     * ݃R}hϊwp[NX̃CX^XłB
     */
    private final Mc3eCommand WRITE_MC3ECOMMAND = new Mc3eCommand() {
        /** CPUĎ^C} */
        private byte[] cpu;

        /** PLC */
        private byte memoryMode;

        /** PLCAhX */
        private long memoryAddress;

        /** ݃f[^ */
        private byte[] writeData;

        /** ݃f[^ʒu */
        private int writePos;

        /** ׂPLCf[^̎coCg */
        private int restByteLength = 0;

        /** PLCf[^oCg */
        private int thisByteLength = 0;

        /** foCXʃwp[NXւ̎Q */
        private ComndKind kind;

        /** R}hݒ肷B */
        public void setCommand(WifeCommand commdef, byte[] cpu, byte[] data)
                throws WifeException {
            this.cpu = cpu;
            // foCXʃwp[NX
            kind = (ComndKind) memoryModeMap.get(new Integer(commdef
                    .getMemoryMode()));
            if (kind == null) {
                throw new WifeException(WifeException.WIFE_ERROR,
                        WifeException.WIFE_NET_COMMAND_ERROR,
                        "Not supported memory mode " + commdef.getMemoryMode());
            }
            memoryMode = kind.getMemKind();
            memoryAddress = commdef.getMemoryAddress();
            writeData = data;
            writePos = 0;
            restByteLength = commdef.getWordLength() * 2;
            thisByteLength = 0;
        }

        /** R}h擾\H */
        public boolean hasCommand() {
            return 0 < restByteLength;
        }

        /** R}h쐬ÃR}h܂B */
        public void nextCommand(ByteBuffer sendBuffer) {
            long addr = kind.getAddress(memoryAddress);

            // foCXɈˑ钷𒴂Ă
            if (DEFAULT_MC3E_PACKET_MAX_SIZE < restByteLength) {
                thisByteLength = DEFAULT_MC3E_PACKET_MAX_SIZE;
                restByteLength -= DEFAULT_MC3E_PACKET_MAX_SIZE;
            } else {
                thisByteLength = restByteLength;
                restByteLength = 0;
            }

            // MC3Ewb_̍쐬
            int datalen = MC3E_COMMAND_REQDATA_LENGTH + thisByteLength;
            sendBuffer.put((byte) (datalen % 0x100));
            sendBuffer.put((byte) (datalen / 0x100));
            sendBuffer.put(cpu);
            // R}hATuR}h
            sendBuffer.put((byte) 0x01);
            sendBuffer.put((byte) 0x14);
            sendBuffer.put((byte) 0x00);
            sendBuffer.put((byte) 0x00);
            // foCXAhX
            sendBuffer.put((byte) (addr % 0x100));
            sendBuffer.put((byte) ((addr / 0x100) % 0x100));
            sendBuffer.put((byte) ((addr / 0x10000) % 0x100));
            // foCXR[h
            sendBuffer.put(memoryMode);
            // 
            sendBuffer.put((byte) ((thisByteLength / 2) % 0x100));
            sendBuffer.put((byte) ((thisByteLength / 2) / 0x100));
            // ݃f[^
            for (int i = writePos; i < (writePos + thisByteLength); i += 2) {
                sendBuffer.put(writeData[i + 1]);
                sendBuffer.put(writeData[i + 0]);
            }

            // ̃AhX
            memoryAddress += (thisByteLength / 2);
            writePos += thisByteLength;
        }

        /** OsR}h쐬܂B */
        public void retryCommand(ByteBuffer sendBuffer) {
            long addr = kind.getAddress(memoryAddress - (thisByteLength / 2));
            int pos = writePos - thisByteLength;

            // MC3Ewb_̍쐬
            int datalen = MC3E_COMMAND_REQDATA_LENGTH + thisByteLength;
            sendBuffer.put((byte) (datalen % 0x100));
            sendBuffer.put((byte) (datalen / 0x100));
            sendBuffer.put(cpu);
            // R}hATuR}h
            sendBuffer.put((byte) 0x01);
            sendBuffer.put((byte) 0x14);
            sendBuffer.put((byte) 0x00);
            sendBuffer.put((byte) 0x00);
            // foCXAhX
            sendBuffer.put((byte) (addr % 0x100));
            sendBuffer.put((byte) ((addr / 0x100) % 0x100));
            sendBuffer.put((byte) ((addr / 0x10000) % 0x100));
            // foCXR[h
            sendBuffer.put(memoryMode);
            // 
            sendBuffer.put((byte) ((thisByteLength / 2) % 0x100));
            sendBuffer.put((byte) ((thisByteLength / 2) / 0x100));
            // ݃f[^
            for (int i = pos; i < (pos + thisByteLength); i += 2) {
                sendBuffer.put(writeData[i + 1]);
                sendBuffer.put(writeData[i + 0]);
            }
        }

        /** R}h̍v擾܂B */
        public int getReqDataLength() {
            return 0;
        }

        /** Mf[^f[^擾܂B */
        public void getResponceData(ByteBuffer recvBuffer, ByteBuffer recvData) {
        }

        /**
         * ̃CX^X̕\Ԃ܂B
         */
        public String toString() {
            StringBuffer s = new StringBuffer();
            s.append("WriteMode:").append("\n");
            s.append("memoryMode:").append(memoryMode).append("\n");
            s.append("memoryAddress:").append(memoryAddress).append("\n");
            s.append("restByteLength:").append(restByteLength).append("\n");
            s.append("thisByteLength:").append(thisByteLength).append("\n");
            return s.toString();
        }
    };

    /** R}hϊNX */
    private Mc3eCommand mc3eCommand = READ_MC3ECOMMAND;

    /**
     * foCXʃwp[NX
     */
    private abstract static class ComndKind {
        private byte memKind;

        public ComndKind(int kd) {
            this.memKind = (byte) (kd & 0xff);
        }

        public byte getMemKind() {
            return memKind;
        }

        abstract public long getAddress(long addr);
    }

    /**
     * rbgfoCX
     */
    private static class ComndKindBitdev extends ComndKind {
        public ComndKindBitdev(int kd) {
            super(kd);
        }

        public long getAddress(long addr) {
            return addr * 16;
        }
    }

    /**
     * [hfoCX
     */
    private static class ComndKindWorddev extends ComndKind {
        public ComndKindWorddev(int kd) {
            super(kd);
        }

        public long getAddress(long addr) {
            return addr;
        }
    }

}
