/*
 * Projrct F-11 - Web SCADA for Java Copyright (C) 2002 Freedom, Inc. All Rights
 * Reserved. This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version. This program is distributed in the hope that it
 * will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details. You should have received a copy of the GNU
 * General Public License along with this program; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

package org.F11.scada.server.communicater;

import java.io.IOException;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.spi.AbstractSelector;
import java.nio.channels.spi.SelectorProvider;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

/**
 * ZN^ǗNXłB
 */
public final class PortSelector implements Runnable {
	private final static Logger log = Logger.getLogger(PortSelector.class);
	private static final int WAIT_TIME = 1000;
	/** ێő吔 */
	private static final int COMMAND_MAX = 2000;

	/** ZN^ĎXbh */
	private Thread thread;
	/** ZN^ */
	private AbstractSelector selector;
	/** ZN^o^ςkeỹ}bv */
	private Map<PortListener, SelectionKey> enterKeys = new HashMap<PortListener, SelectionKey>();

	/** ̏vێL[ */
	private LinkedList<SelectCommand> commands = new LinkedList<SelectCommand>();

	/**
	 * RXgN^ ĎXbhJn܂B
	 */
	public PortSelector() throws IOException {
		startup();
	}

	/*
	 * ZN^쐬AĎXbhJn܂B
	 */
	private void startup() throws IOException {
		log.debug("startup.");
		selector = SelectorProvider.provider().openSelector();
		thread = new Thread(this);
		thread.setName("PortSelector");
		thread.start();
	}

	/**
	 * Xi[o^Ă True
	 */
	public synchronized boolean isActive() {
		return !enterKeys.isEmpty();
	}

	/**
	 * `lZN^ɓo^AXi[o^܂B o^܂ŕԂ܂B
	 * @param listener o^郊Xi[
	 * @return o^̃ZNVL[
	 */
	public void addListener(PortListener listener) throws InterruptedException {
		log.debug("addListener.");

		// o^v
		enterCommand(new OpenChannel(listener));
		// o^܂őҋ@
		synchronized (this) {
			while (!enterKeys.containsKey(listener)) {
				selector.wakeup();
				wait(WAIT_TIME);
			}
		}
	}

	/**
	 * wZNVL[̃`lAZN^폜܂B 폜܂ŕԂ܂B
	 * @param selectionKey ZNVL[
	 * @return SẴ`lꍇATrue
	 */
	public void removeListener(PortListener listener)
			throws InterruptedException {
		log.debug("removeListener.");
		// 폜v
		enterCommand(new CloseChannel(listener));
		// ZNVL[폜܂őҋ@
		synchronized (this) {
			while (enterKeys.containsKey(listener)) {
				selector.wakeup();
				wait(WAIT_TIME);
			}
			if (selector.keys().isEmpty()) {
				selector.wakeup();
				thread.interrupt();
				thread = null;
				log.debug("shutdown.");
			}
		}
	}

	/**
	 * wZNVL[̑ΏۃZbgύX܂B ύX҂ɕԂ܂B
	 * @param selectionKey ZNVL[
	 * @param ops ΏۃZbg
	 */
	public void setInterestOps(PortListener listener, int ops)
			throws InterruptedException {
		log.debug("setInterestOps.");
		// ύXv
		enterCommand(new SetInterestOpsChannel(listener, ops));
	}
	public void resetInterestOps(PortListener listener, int ops)
			throws InterruptedException {
		log.debug("resetInterestOps.");
		// ύXv
		enterCommand(new ResetInterestOpsChannel(listener, ops));
	}

	/**
	 * w̃`l֍ăI[vv܂BύX҂ɕԂ܂B
	 * @param listener
	 * @throws InterruptedException
	 */
	public void reopenChannel(PortListener listener)
			throws InterruptedException {
		log.debug("reopenChannel.");
		// ăI[vv
		enterCommand(new ReopenChannel(listener));
	}

	/**
	 * ZN^̊Ď[vłB
	 */
	public void run() {
		log.debug("start.");
		Thread th = thread;
		try {
			while (th != null) {
				int n = 0;
				try {
					n = selector.select();
				} catch (Exception e) {
					if (!selector.isOpen()) {
						log.fatal("Exception caught: ", e);
						break;
					}
				}
				if (n > 0) {
					Iterator<SelectionKey> keyIterator = selector.selectedKeys().iterator();

					while (keyIterator.hasNext()) {
						SelectionKey key = keyIterator.next();
						keyIterator.remove();
						if (!key.isValid()) {
							continue;
						}
						PortListener listener = (PortListener) key.attachment();
						if (listener == null) {
							continue;
						}
						try {
							if (key.isAcceptable()) {
								listener.onAccept();
							} else if (key.isConnectable()) {
								listener.onConnect();
							} else if (key.isReadable()) {
								listener.onRead();
							} else if (key.isWritable()) {
								listener.onWrite();
							}
						} catch (Exception e) {
							log.warn("Exception caught: ", e);

							synchronized (this) {
								commands.addLast(new ReopenChannel(listener));
							}
						}
					}
				}
				// v
				executeCommands();
				th = thread;
			}
		} catch (Exception e) {
			log.fatal("Exception caught: ", e);
		} finally {
			th = null;
			try {
				selector.close();
			} catch (IOException e) {
				log.fatal("Exception caught: ", e);
			}
		}
		log.debug("stop.");
	}

	private synchronized void enterCommand(SelectCommand command)
			throws InterruptedException {
		// L[󂭂܂őҋ@
		while (COMMAND_MAX <= commands.size()) {
			wait(WAIT_TIME);
		}
		commands.addLast(command);
		selector.wakeup();
	}

	/*
	 * vL[̐擪s܂B
	 */
	private synchronized void executeCommands() throws Exception {
		List<SelectCommand> nextCommands = new ArrayList<SelectCommand>();
		while (!commands.isEmpty()) {
			SelectCommand command = commands.removeFirst();
			SelectCommand leftComm = command.execute();
			if (leftComm != null) {
				nextCommands.add(leftComm);
			}
		}
		commands.addAll(nextCommands);
		notifyAll();
	}

	/*
	 * ṽC^[tF[XłB
	 */
	private interface SelectCommand {
		public SelectCommand execute() throws Exception;
	}

	/*
	 * I[vvsNXłB
	 */
	private final class OpenChannel implements SelectCommand {
		private PortListener listener;

		public OpenChannel(PortListener listener) {
			this.listener = listener;
		}
		public SelectCommand execute() throws IOException, InterruptedException {
			if (log.isDebugEnabled()) {
				log.debug("OpenChannel. " + listener.toString());
			}
			SelectableChannel channel = listener.open();
			SelectionKey key = channel.register(selector, 0);
			key.attach(listener);
			enterKeys.put(listener, key);
			return null;
		}

	}

	/*
	 * N[YvsNXłB
	 */
	private final class CloseChannel implements SelectCommand {
		private final PortListener listener;

		public CloseChannel(PortListener listener) {
			this.listener = listener;
		}

		public SelectCommand execute() throws IOException {
			if (log.isDebugEnabled()) {
				log.debug("CloseChannel. " + listener.toString());
			}
			SelectionKey key = enterKeys.remove(listener);
			if (key != null) {
				key.attach(null);
				key.cancel();
				listener.close();
				return null;
			} else {
				enterKeys.put(listener, key);
				return this;
			}
		}

	}

	/*
	 * ăI[vvsNXłB
	 */
	private final class ReopenChannel implements SelectCommand {
		private final PortListener listener;

		public ReopenChannel(PortListener listener) {
			this.listener = listener;
		}

		public SelectCommand execute() throws IOException, InterruptedException {
			log.debug("ReopenChannel. " + listener.toString());
			SelectionKey key = enterKeys.remove(listener);
			if (key != null) {
				key.attach(null);
				key.cancel();
				listener.close();
				return new OpenChannel(listener);
			} else {
				enterKeys.put(listener, key);
				return this;
			}
		}

	}

	/*
	 * ύXvsNXłB
	 */
	private final class SetInterestOpsChannel implements SelectCommand {
		private final PortListener listener;
		private final int ops;

		public SetInterestOpsChannel(PortListener listener, int ops) {
			this.listener = listener;
			this.ops = ops;
		}
		public SelectCommand execute() {
			log.debug("SetInterestOpsChannel. " + listener.toString() + " Ops=" + ops);
			SelectionKey key = enterKeys.get(listener);
			if (key != null && key.isValid()) {
				int old = key.interestOps();
				key.interestOps(old | ops);
			}
			return null;
		}

	}

	/*
	 * ύXvsNXłB
	 */
	private final class ResetInterestOpsChannel implements SelectCommand {
		private final PortListener listener;
		private final int ops;

		public ResetInterestOpsChannel(PortListener listener, int ops) {
			this.listener = listener;
			this.ops = ops;
		}
		public SelectCommand execute() {
			log.debug("ResetInterestOpsChannel. " + listener.toString() + " Ops=" + ops);
			SelectionKey key = enterKeys.get(listener);
			if (key != null && key.isValid()) {
				int old = key.interestOps();
				key.interestOps(old & (~ops));
			}
			return null;
		}

	}

}