{
  Feeding Origin 2 is software that supports calculation of nutrient requirement of livestock
  and feed design based on Japanese feeding standard.
  Copyright (C) 2017,  Syuichiro Fukagawa.

  This program is free software; you can redistribute it and/or modify it under the terms of
  the GNU General Public License as published by the Free Software Foundation; either version 3
  of the License, or (at your option) any later version.
  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU General Public License for more details.
  You should have received a copy of the GNU General Public License along with this program.
  If not, see <http://www.gnu.org/licenses/>.


  Feeding Origin 2 は、日本飼養標準に基づいた家畜の養分要求量の計算および飼料設計を支援するソフ
  トウェアです.
  Copyright (C) 2017,　深川修一郎.

  　このプログラムはフリーソフトウェアです。あなたはこれを、フリーソフトウェア財団によって
  発行されたGNU一般公衆利用許諾書(バージョン3か、それ以降のバージョンのうちどれか)が定める
  条件の下で再頒布または改変することができます。
  このプログラムは有用であることを願って頒布されますが、*全くの無保証* です。商業可能性の保証
  や特定目的への適合性は、言外に示されたものも含め、全く存在しません。詳しくはGNU一般公衆利用
  許諾書をご覧ください。
  　あなたはこのプログラムと共に、GNU一般公衆利用許諾書のコピーを一部 受け取っているはずです。
  もし受け取っていなければ、<http://www.gnu.org/licenses/> をご覧ください。
  
  *本日本語訳は、FSFより2007年6月29日に発行されたGNU GPL v3の原文から、八田真行氏が翻訳した
   ものを基にしています。 ＜ https://mag.osdn.jp/07/09/02/130237 ＞
}

unit u_feedingcalc2;

{$mode objfpc}{$H+}

interface

Uses Math, SysUtils;

type reNUTRIENT = record     // 養分要求量のレコード型宣言
 	dFCM  :  Double;          // FCM
 	dDMI  :  Double;          // DMI
 	dCP   :  Double;          // CP
 	dDCP  :  Double;          // DCP
 	dTDN  :  Double;          // TDN
 	dDE   :  Double;          // DE
 	dMEC  :  Double;          // ME(Mcal)
 	dMEJ  :  Double;          // ME(MJ)
 	dCa   :  Double;          // Ca
 	dP    :  Double;          // P
 	dVA   :  Double;          // VA
 	dVD   :  Double;          // VD
end;

type reMILK = record         // 搾乳牛用のレコード型宣言
	dYld1  :    Double;	  // 乳量
	dFat   :    Double;	  // 乳脂率
	dWt1   :    Double;	  // 体重
	dWkbrt :    Double;	  // 分娩後週
	cFed   :    String;	  // 給与方法
	dKion  :    Double;	  // 平均気温
	cBrfq  :    String;	  // 産次
	cLVCP  :    String;	  // CPの最低値
	Nutrient :  reNUTRIENT;   // 養分要求量のレコード
end;

type reDRY = record          // 乾乳牛用のレコード型宣言
	dWt1   :    Double;	  // 体重
	cPre   :    String;	  // 分娩前
	cWkb   :    String;	  // 分娩1週間前
	cLVCP  :    String;	  // CPの最低値
	cKind  :    String;	  // 胎子の品種
	Nutrient :  reNUTRIENT;   // 養分要求量のレコード
end;

type reGROW = record         // 育成牛用のレコード型宣言
	dWt1	 :  Double ;      // 体重
	dInWt1	 :  Double ;      // 日増体重
	cLVCP	 :  String ;      // CPの最低値
	cKset	 :  String ;      // 季節の考慮
	cWean	 :  String ;      // 育成牛の離乳(離乳前、離乳後、体重が66kg＜X＜75kg)
	Nutrient :  reNUTRIENT;   // 養分要求量のレコード
end;

var
   gdDMI	 : 	Double ;		// 乾物摂取量DMI（ｋｇ）

   gdTDN	 : 	Double ;		// 可消化養分総量
   gdDE	         : 	Double ;		// 可消化エネルギー
   gdME	         : 	Double ;		// 代謝エネルギーの総要求量(Mcal)
   gdMJ	         : 	Double ;		// 代謝エネルギーの総要求量(MJ)

   gdCPR	 : 	Double ;		// 粗蛋白質要求量
   gdDCPR	 : 	Double ;		// 可消化蛋白質要求量
   gdMCP	 : 	Double ;		// 維持に要する粗蛋白質要求量
   gdMDCP	 : 	Double ;		// 維持に要する粗蛋白質要求量
   gdPCP	 : 	Double ;		// 妊娠に要する粗蛋白質要求量
   gdPDCP	 : 	Double ;		// 妊娠に要する粗蛋白質要求量

   gdCa	         : 	Double ;		// カルシウム
   gdP	         : 	Double ;		// リン
   gdVA	         : 	Double ;		// ビタミンA
   gdVD	         : 	Double ;		// ビタミンD

   gdFCM	 : 	Double ;		// FCM

   gdMEm	 : 	Double ;		// 代謝エネルギーの維持要求量
   gdMEp	 : 	Double ;		// 代謝エネルギーの生産要求量

   gdYCP	 : 	Double ;		// 維持に要する粗蛋白質要求量
   gdYDCP	 : 	Double ;		// 維持に要する粗蛋白質要求量

function  MilkingCow( stM : reMILK ) : reNUTRIENT ;      // 搾乳牛の養分要求量計算
procedure Milking_Calc_DMI( stM : reMILK ) ;             // 　〃　　乾物摂取量計算
procedure Milking_Calc_ME( stM : reMILK ) ;              // 　〃　　エネルギーの計算
procedure Milking_Calc_DCP( stM : reMILK ) ;             // 　〃　　蛋白質の計算
procedure Milking_Calc_CAPAD( stM : reMILK ) ;           // 　〃　　ミネラル＆ビタミンの計算

function  DryingCow_A( stD : reDRY ) : reNUTRIENT ;      // 乾乳牛の養分要求量計算
procedure Drying_Calc_DMI_A( stD : reDRY ) ;             // 　〃　　乾物摂取量計算
procedure Drying_Calc_ME_A( stD : reDRY ) ;              // 　〃　　エネルギーの計算
procedure Drying_Calc_DCP_A( stD : reDRY ) ;             // 　〃　　蛋白質の計算
procedure Drying_Calc_CAPAD_A( stD : reDRY ) ;           // 　〃　　ミネラル＆ビタミンの計算

function  GrowingCow_A( stG : reGROW ) : reNUTRIENT ;    // 育成牛の養分要求量計算
procedure Growing_Calc_DMI_A( stG : reGROW ) ;           // 　〃　　乾物摂取量計算
procedure Growing_Calc_ME_A( stG : reGROW ) ;            // 　〃　　エネルギーの計算
procedure Growing_Calc_DCP_A( stG : reGROW ) ;           // 　〃　　蛋白質の計算
procedure Growing_Calc_CAPAD_A( stG : reGROW ) ;         // 　〃　　ミネラル＆ビタミンの計算

function StandardDailyWeightGain(var dWt : Double) : Double;  // １日の増体重の計算

implementation

//*** 搾乳牛の養分要求量計算 ***
function MilkingCow( stM : reMILK ) : reNUTRIENT ;
var
   stRet : reNUTRIENT ;
begin
  //計算結果出力変数の初期化
	gdFCM  := 0;            // 4%脂肪補正乳量
	gdDMI  := 0;		// 乾物摂取量DMI（ｋｇ）
	gdTDN  := 0;  	        // 可消化養分総量
	gdDE   := 0;   	        // 可消化エネルギー
	gdME   := 0;   	        // 代謝エネルギーの総要求量(Mcal)
	gdMJ   := 0;   	        // 代謝エネルギーの総要求量(MJ)
	gdCPR  := 0;		// 粗蛋白質要求量
	gdDCPR := 0; 	        // 可消化蛋白質要求量
	gdMCP  := 0;		// 維持に要する粗蛋白質要求量
	gdMDCP := 0;		// 維持に要する粗蛋白質要求量
	gdPCP  := 0;		// 妊娠に要する粗蛋白質要求量
	gdPDCP := 0;		// 妊娠に要する粗蛋白質要求量
	gdCa   := 0;		// カルシウム
	gdP    := 0;		// リン
	gdVA   := 0;		// ビタミンA
	gdVD   := 0;		// ビタミンD

  //養分要求量の計算処理
	Milking_Calc_DMI( stM );    //乾物摂取量の計算
	Milking_Calc_ME( stM );     //エネルギーの計算
	Milking_Calc_DCP( stM );    //蛋白質の計算
	Milking_Calc_CAPAD( stM );  //ミネラル＆ビタミンの計算

  //養分要求量計算結果の代入
	stRet.dFCM	:=	gdFCM;	    // 4%脂肪補正乳量
	stRet.dDMI	:=	gdDMI;      // 乾物摂取量DMI（ｋｇ
	stRet.dCP	:=	gdCPR;      // 粗蛋白質要求量
	stRet.dDCP	:=	gdDCPR;     // 可消化蛋白質要求量
	stRet.dTDN	:=	gdTDN;      // 可消化養分総量
	stRet.dDE	:=	gdDE;       // 可消化エネルギー
	stRet.dMEC	:=	gdME;       // 代謝エネルギーの総要求量(Mcal)
	stRet.dMEJ	:=	gdMJ;       // 代謝エネルギーの総要求量(MJ)
	stRet.dCa	:=	gdCa;       // カルシウム
	stRet.dP	:=	gdP;        // リン
	stRet.dVA	:=	gdVA;       // ビタミンA
	stRet.dVD	:=	gdVD;       // ビタミンD

	MilkingCow := stRet ;
end;

//*** 搾乳牛の乾物摂取量計算 ***
procedure Milking_Calc_DMI( stM : reMILK ) ;
var
    iBnbn : Integer ;            //分娩後週
    dDMI  : Double  ;
    dKSU  : Double  ;

begin
    iBnbn := Trunc(stM.dWkbrt);        //分娩後週
    gdFCM  := stM.dYld1 * ( 0.15 * stM.dFat + 0.4 ) ; //4%脂肪補正乳量

    if( iBnbn >= 11 ) then     //分娩後11週以降
      begin
        if( stM.cBrfq = '初産分娩後' )  then  //初産泌乳牛
            dDMI := 1.9120 + 0.07031 *  power(stM.dWt1, 0.75) + 0.34923 * gdFCM   // JFS2006 p6, 1.1-2
        else                      //2産以上の泌乳牛
            dDMI := 1.3922 + 0.05839 *  power(stM.dWt1, 0.75) + 0.40497 * gdFCM ; // JFS2006 p6, 1.1-1
      end
    else                //泌乳初期
      begin
        if( stM.cBrfq = '初産分娩後' )  then  //初産泌乳牛
          begin
            dDMI := 1.9120 + 0.07031 *  power(stM.dWt1, 0.75) + 0.34923 * gdFCM ;  // JFS2006 p6, 1.1-2
            dKSU := 1.3671 - 0.6558 * Exp((-0.0498 * iBnbn)) ;                // JFS2006 p7, 1.1-4
          end
        else                                      //2産以上の泌乳牛
          begin
            dDMI := 1.3922 + 0.05839 *  power(stM.dWt1, 0.75) + 0.40497 * gdFCM ;  // JFS2006 p6, 1.1-1
            dKSU := 1.0 - 0.3531 * Exp((-0.3247 * iBnbn)) ;                   // JFS2006 p7, 1.1-3
          end ;
        dDMI := dDMI * dKSU ;
      end ;

    //平均気温による補正
    if( stM.cBrfq = '初産分娩後' ) then   //初産泌乳牛
      begin
        if(stM.dKion < 20)  then           //平均気温による補正 JFS2006 p65、表4.9.1.2　
            dDMI := dDMI * 1.00
        else if((stM.dKion <= 20) and (stM.dKion < 22)) then
            dDMI := dDMI * 1.00
        else if((stM.dKion <= 22) and (stM.dKion < 24)) then
            dDMI := dDMI * 1.00
        else if((stM.dKion <= 24) and (stM.dKion < 26)) then
            dDMI := dDMI * 0.99
        else if((stM.dKion <= 26) and (stM.dKion < 28)) then
            dDMI := dDMI * 0.96
        else if((stM.dKion <= 28) and (stM.dKion < 30)) then
            dDMI := dDMI * 0.92
        else
            dDMI := dDMI * 0.87 ;
      end
    else                        //2産以上の泌乳牛
      begin
        if(stM.dKion < 20) then                      //平均気温による補正 JFS2006 p65、表4.9.1.2
            dDMI := dDMI * 1.00
        else if((stM.dKion <= 20) and (stM.dKion < 22)) then
            dDMI := dDMI * 1.00
        else if((stM.dKion <= 22) and (stM.dKion < 24)) then
            dDMI := dDMI * 0.99
        else if((stM.dKion <= 24) and (stM.dKion < 26)) then
            dDMI := dDMI * 0.96
        else if((stM.dKion <= 26) and (stM.dKion < 28)) then
            dDMI := dDMI * 0.90
        else if((stM.dKion <= 28) and (stM.dKion < 30)) then
            dDMI := dDMI * 0.82
        else
            dDMI := dDMI * 0.70 ;
      end ;

      gdDMI := dDMI ;
end;

//*** 搾乳牛のエネルギーの計算 ***
procedure Milking_Calc_ME( stM : reMILK ) ;
var
    dMTDN, dMDE, dMMJ : Double  ;
    dPTDN, dPDE, dPMJ : Double  ;
    dMERC : Double  ;
    dCFA  : Double  ;

begin
    gdMEm := 0.1163 * power(stM.dWt1, 0.75);    //維持要求量 JFS2006 p129、8.3.2-1
    dMDE  := gdMEm / 0.82 ;           // JFS2006 p130、8.3.2-2
    dMTDN := dMDE  / 4.41 ;           // JFS2006 p130、8.3.2-3
    dMMJ  := gdMEm * 4.184 ;          // JFS2006 p129、L36

    //産次
    if ( stM.cBrfq = '初産分娩後' ) then    //初産分娩時から２産分娩前まで
        begin
            gdMEm := gdMEm * 1.3 ;    // JFS2006 p129、右段、L17
            dMDE  := dMDE  * 1.3 ;
            dMTDN := dMTDN * 1.3 ;
            dMMJ  := dMMJ  * 1.3 ;
        end
    else if ( stM.cBrfq = '2産分娩から3産分娩まで' ) then //２産分娩時から３産分娩前まで
        begin
            gdMEm := gdMEm * 1.15 ;    // JFS2006 p129、右段、L19
            dMDE  := dMDE  * 1.15 ;
            dMTDN := dMTDN * 1.15 ;
            dMMJ  := dMMJ  * 1.15 ;
        end ;

    //飼料給与方式
    if ( stM.cFed = '分離' ) then
        dCFA := 1 + (stM.dYld1 / 15) * 0.04      //分離給与 JFS2006 p133、8.6.1
    else if( stM.cFed = 'TMR' ) then
        dCFA := 1 + (stM.dYld1 / 15) * 0.035 ;   //ＴＭＲ   JFS2006 p133、8.6.2

    //乳生産にかかる要求量
    dMERC := ( 0.0913 * stM.dFat + 0.3678 ) / 0.62 ; // JFS2006 p10、1.2-1 & p133、8.5.2-1 & 8.5.2-2
    gdMEp := dMERC * stM.dYld1 ;		// 代謝エネルギーの生産要求量

    dPDE  := gdMEp / 0.82 ;   // JFS2006 p130、8.3.2-2
    dPTDN := dPDE  / 4.41 ;   // JFS2006 p130、8.3.2-3
    dPMJ  := gdMEp * 4.184 ;  // JFS2006 p129、L36

    //維持と乳生産の合計と給与方式の補正
    gdME  := ( gdMEm + gdMEp ) * dCFA ;   // 代謝エネルギーの総要求量(Mcal)
    gdTDN := ( dMTDN + dPTDN ) * dCFA ;   // 可消化養分総量
    gdDE  := ( dMDE  + dPDE  ) * dCFA ;   // 可消化エネルギー
    gdMJ  := ( dMMJ  + dPMJ  ) * dCFA ;   // 代謝エネルギーの総要求量(MJ)
end;

//*** 搾乳牛の蛋白質の計算 ***
procedure Milking_Calc_DCP( stM : reMILK ) ;
var
    ldKM    :  Double  ;    //飼料給与方法による補正値
    ldHS    :  Double  ;    //産次による補正値
    ldWrk1  :  Double  ;

begin
    gdMDCP   := 2.71 * power(stM.dWt1, 0.75);    //維持可消化蛋白質要求量　JFS2006 p129、右段、8.3.1-1
    gdYDCP   := (26.6 + 5.3 * stM.dFat) * stM.dYld1 ;    //産乳に要する可消化蛋白質要求量　JFS2006 p133、左段、8.5.1-1
    gdMCP    := gdMDCP / 0.60 ;            //維持粗蛋白質要求量　JFS2006 p129、右段、8.3.1-2
    gdYCP    := gdYDCP / 0.65 ;            //産乳に要する粗蛋白質要求量　JFS2006 p133、左段、8.5.1-2

    //産次による条件
    if( stM.cBrfq = '初産分娩後' ) then
        ldHS := 1.30 ;    //初産分娩から二産分娩まで  JFS2006 p28～p29

    if( stM.cBrfq = '2産分娩から3産分娩まで' ) then
         ldHS := 1.15 ;   //二産分娩から三産分娩まで

    if( stM.cBrfq = '3産分娩以後' ) then
        ldHS := 1.00 ;   //三産分娩以降

    gdMDCP := gdMDCP * ldHS;
    gdMCP  := gdMCP  * ldHS;

    //飼料給与方式による条件
    if( stM.cFed = '分離' ) then
        ldKM := 0.04  ;    //分離給与

    if( stM.cFed = 'TMR' )  then
        ldKM := 0.035 ;    //ＴＭＲ

    gdCPR  := (gdMCP  + gdYCP)  * (1 + (stM.dYld1 / 15) * ldKM ) ; // 粗蛋白質要求量     JFS2006 p32, p133 8.6.1
    gdDCPR := (gdMDCP + gdYDCP) * (1 + (stM.dYld1 / 15) * ldKM ) ; // 可消化蛋白質要求量 JFS2006 p32, p133 8.6.1

    //CPの最低値をDMの12%に設定する
    if( stM.cLVCP = 'ON' ) then
      begin
        ldWrk1 := gdDMI * 120 ;  //DMの12%(g)
        if( gdCPR < ldWrk1 ) then
          begin
            gdCPR  := ldWrk1 ;         // 粗蛋白質要求量
            gdDCPR := gdCPR * 0.60;    // 可消化蛋白質要求量
          end;
      end;
end;

//*** 搾乳牛のミネラル＆ビタミンの計算 ***
procedure Milking_Calc_CAPAD( stM : reMILK ) ;
var
    ldCFA :  Double  ;    //飼料給与方法による補正値
    ldHS  :  Double  ;    //産次による補正値
    dMCa  :  Double  ;
    dMP   :  Double  ;
    dMVA  :  Double  ;
    dMVD  :  Double  ;
    dYCa  :  Double  ;
    dYP   :  Double  ;
    dYVA  :  Double  ;
    dYVD  :  Double  ;

begin
    dMCa := 0.0154 * stM.dWt1 / 0.38 ;   //維持に要するカルシウム JFS2006 p130, 8.3.3-1
    dMP  := 0.0143 * stM.dWt1 / 0.5  ;   //維持に要するリン       JFS2006 p130, 8.3.3-2
    dMVA := 0.0424 * stM.dWt1 ;          //維持に要するビタミンA  JFS2006 p130, 8.3.3-3
    dMVD := 0.006  * stM.dWt1 ;          //維持に要するビタミンD  JFS2006 p130, 8.3.3-4

    dYCa := ( 1.20 * gdFCM ) / 0.38 ;    //産乳に要するカルシウム JFS2006 p133、8.5.3-1
    dYP  := ( 0.90 * gdFCM ) / 0.5  ;    //産乳に要するリン       JFS2006 p133、8.5.3-2
    dYVA := 1.3 * stM.dYld1 ;            //産乳に要するビタミンA  JFS2006 p133、8.5.3-3
    dYVD := 0.004 * stM.dWt1 ;           //産乳に要するビタミンD  JFS2006 p133、8.5.3-4

    //産次による補正
    if( stM.cBrfq = '初産分娩後' ) then
        ldHS := 1.30 ;   //初産分娩から二産分娩まで  JFS2006 p28～ｐ29 & p129,右段,L16～L19

    if( stM.cBrfq = '2産分娩から3産分娩まで' ) then
        ldHS := 1.15 ;   //二産分娩から三産分娩まで

    if( stM.cBrfq = '3産分娩以後' ) then
        ldHS := 1.00 ;   //三産分娩以降

    dMCa := dMCa * ldHS;
    dMP  := dMP  * ldHS;

    //要求量の合計
    gdCa :=  dMCa + dYCa  ;        // カルシウム
    gdP  :=  dMP  + dYP   ;        // リン
    gdVA :=  dMVA + dYVA  ;        // ビタミンA
    gdVD :=  dMVD + dYVD  ;        // ビタミンD

    //飼料給与方式による補正
    if( stM.cFed = '分離' ) then
        ldCFA := 1 + (stM.dYld1 / 15) * 0.04  ;    //分離給与 JFS2006 p133, 8.6.1

    if( stM.cFed = 'TMR' )  then
        ldCFA := 1 + (stM.dYld1 / 15) * 0.035 ;    //ＴＭＲ JFS2006 p133, 8.6.2

    gdCa := gdCa * ldCFA ;        // カルシウム
    gdP  := gdP  * ldCFA ;        // リン

    //平均気温による補正
    if( ( stM.dKion >= 22.0) and (stM.dKion < 26.0) ) then
    begin
        gdCa := gdCa * 1.10 ;    // カルシウム   JFS2006 p66、表4.9.2
        gdP  := gdP  * 1.10 ;    // リン         JFS2006 p66、表4.9.2
    end;

    if( stM.dKion >= 26.0 ) then
    begin
        gdCa := gdCa * 1.20 ;    // カルシウム   JFS2006 p66、表4.9.2
        gdP  := gdP  * 1.20 ;    // リン         JFS2006 p66、表4.9.2
    end;
end;

//*** 乾乳牛の養分要求量計算 ***
function  DryingCow_A( stD : reDRY ) : reNUTRIENT ;
var
    stRet : reNUTRIENT ;
begin
    //計算結果出力変数の初期化
      	gdFCM  := 0;        // 4%脂肪補正乳量
      	gdDMI  := 0;		// 乾物摂取量DMI（ｋｇ）
      	gdTDN  := 0;  	    // 可消化養分総量
      	gdDE   := 0;   	    // 可消化エネルギー
      	gdME   := 0;   	    // 代謝エネルギーの総要求量(Mcal)
      	gdMJ   := 0;   	    // 代謝エネルギーの総要求量(MJ)
      	gdCPR  := 0;		// 粗蛋白質要求量
      	gdDCPR := 0; 	    // 可消化蛋白質要求量
      	gdMCP  := 0;		// 維持に要する粗蛋白質要求量
      	gdMDCP := 0;		// 維持に要する粗蛋白質要求量
      	gdPCP  := 0;		// 妊娠に要する粗蛋白質要求量
      	gdPDCP := 0;		// 妊娠に要する粗蛋白質要求量
      	gdCa   := 0;		// カルシウム
      	gdP    := 0;		// リン
      	gdVA   := 0;		// ビタミンA
      	gdVD   := 0;		// ビタミンD

    //養分要求量の計算処理
      	Drying_Calc_DMI_A( stD );    //乾物摂取量の計算
      	Drying_Calc_ME_A( stD );     //エネルギーの計算
      	Drying_Calc_DCP_A( stD );    //蛋白質の計算
      	Drying_Calc_CAPAD_A( stD );  //ミネラル＆ビタミンの計算

    //養分要求量計算結果の代入
      	stRet.dFCM	:=	gdFCM;	    // 4%脂肪補正乳量
      	stRet.dDMI	:=	gdDMI;      // 乾物摂取量DMI（ｋｇ
      	stRet.dCP	:=	gdCPR;      // 粗蛋白質要求量
      	stRet.dDCP	:=	gdDCPR;     // 可消化蛋白質要求量
      	stRet.dTDN	:=	gdTDN;      // 可消化養分総量
      	stRet.dDE	:=	gdDE;       // 可消化エネルギー
      	stRet.dMEC	:=	gdME;       // 代謝エネルギーの総要求量(Mcal)
      	stRet.dMEJ	:=	gdMJ;       // 代謝エネルギーの総要求量(MJ)
      	stRet.dCa	:=	gdCa;       // カルシウム
      	stRet.dP	:=	gdP;        // リン
      	stRet.dVA	:=	gdVA;       // ビタミンA
      	stRet.dVD	:=	gdVD;       // ビタミンD

      	DryingCow_A := stRet ;
end;

//*** 乾乳牛の乾物摂取量計算 ***
procedure Drying_Calc_DMI_A( stD : reDRY ) ;
begin
    if( stD.cKind = 'ﾎﾙｽﾀｲﾝ(初妊)' )  then          // ﾎﾙｽﾀｲﾝ(初妊)
        if ( stD.cPre <> 'それ以外' )   then        // 分娩前
            gdDMI := 0.49137 + 0.01768 * stD.dWt1 + 0.91754 * 0.35  // JFS2006 p7, 1.1-4 (日増体重0.35kg)
        else
            gdDMI := 0
    else
        if( stD.cWkb = 'ON' ) then  // 分娩１週間前
            gdDMI := 0.016 * stD.dWt1                                // JFS2006 p7、1.1-6
        else                        // それ以外
            gdDMI := 0.017 * stD.dWt1 ;                             // JFS2006 p7、1.1-5
end;

//*** 乾乳牛のエネルギーの計算 ***
procedure Drying_Calc_ME_A( stD : reDRY ) ;
var
    dMME   : Double ;  // 維持代謝エネルギー要求量
    dMTDN  : Double ;  // 維持可消化養分総量
    dMDE   : Double ;  // 維持可消化エネルギー
    dMMJ   : Double ;  // 維持代謝エネルギー要求量(MJ)
    dMERC  : Double ;  // 妊娠時のME増給量
    dTDNRC : Double ;  //    〃   可消化養分増給量
    dDERC  : Double ;  //    〃   可消化エネルギー増給量
    dMJRC  : Double ;  //    〃   代謝エネルギー増給量(MJ)
    dE     : Double ;  // 妊娠ｔ日目における胎子のエネルギー蓄積総量(Mcal) JFS2006 p131, 8.4.2-1

begin
    if ( stD.cKind = 'ﾎﾙｽﾀｲﾝ(初妊)') then   //ホルスタイン種（初妊）
      begin
        // 維持＆増体の代謝エネルギー要求量
        dMME  := 0.1163 * power(stD.dWt1, 0.75)
               + 0.1355 * 0.35 * power(stD.dWt1, 0.75) ;  // 維持代謝エネルギー要求量  JFS2006 p129, 8.2.2-3 (日増体重0.35kg)

        dMDE  := dMME / 0.82 ;      // 維持可消化エネルギー  JFS2006 p130、8.3.2-2
        dMTDN := dMDE / 4.41 ;      // 維持可消化養分総量    JFS2006 p130、8.3.2-3
        dMMJ  := dMME * 4.184 ;     // 維持代謝エネルギー要求量(MJ)  p129、L36

        dE := ((0.00159 * power(280, 2) - 0.0352 * 280 - 35.4) / 46 * 42)
            - ((0.00159 * power(217, 2) - 0.0352 * 217 - 35.4) / 46 * 42) ;  // JFS2006 p131, 8.4.2-1

        if ( stD.cPre = '分娩前9週から4週まで' ) then    //分娩前9週から4週まで
          begin
            dMERC  := dE / 63 / 0.14 * 0.9 ;      // JFS2006 p132, 8.4.2-2

            dDERC  := dMERC / 0.82 ;     // JFS2006 p130、8.3.2-2
            dTDNRC := dDERC / 4.41 ;     // JFS2006 p130、8.3.2-3
            dMJRC  := dMERC * 4.184 ;    // JFS2006 p129、L36

            gdME   := dMME  + dMERC ;    // 代謝エネルギーの総要求量(Mcal)
            gdTDN  := dMTDN + dTDNRC ;   // 可消化養分総量
            gdDE   := dMDE  + dDERC ;    // 可消化エネルギー
            gdMJ   := dMMJ  + dMJRC ;    // 代謝エネルギーの総要求量(MJ)
          end
        else if ( stD.cPre = '分娩前3週から分娩まで' ) then     //分娩前3週から分娩まで
          begin
            dMERC  := dE / 63 / 0.14 * 1.2 ;      // JFS2006 p132, 8.4.2-3

            dDERC  := dMERC / 0.82 ;     // JFS2006 p130、8.3.2-2
            dTDNRC := dDERC / 4.41 ;     // JFS2006 p130、8.3.2-3
            dMJRC  := dMERC * 4.184 ;    // JFS2006 p129、L36

            gdME   := dMME  + dMERC ;    // 代謝エネルギーの総要求量(Mcal)
            gdTDN  := dMTDN + dTDNRC ;   // 可消化養分総量
            gdDE   := dMDE  + dDERC ;    // 可消化エネルギー
            gdMJ   := dMMJ  + dMJRC ;    // 代謝エネルギーの総要求量(MJ)
          end
        else                             //それ以外
          begin   //そのまま
            gdME   := dMME  * 1.1 ;      // 代謝エネルギーの総要求量(Mcal)  JFS2006 p130、左段、L14
            gdTDN  := dMTDN * 1.1 ;      // 可消化養分総量
            gdDE   := dMDE  * 1.1 ;      // 可消化エネルギー
            gdMJ   := dMMJ  * 1.1 ;      // 代謝エネルギーの総要求量(MJ)
          end
      end

    else if ( stD.cKind = 'ﾎﾙｽﾀｲﾝ(経産)') then   //ホルスタイン種（経産）
      begin
        // 維持の代謝エネルギー要求量
        dMME  := 0.1163 * power(stD.dWt1, 0.75) ;      // JFS2006 p129、8.3.2-1

        dMDE  := dMME / 0.82 ;      // JFS2006 p130、8.3.2-2
        dMTDN := dMDE / 4.41 ;      // JFS2006 p130、8.3.2-3
        dMMJ  := dMME * 4.184 ;     // JFS2006 p129、L36

        dE := ((0.00159 * power(280, 2) - 0.0352 * 280 - 35.4) / 46 * 46)
            - ((0.00159 * power(217, 2) - 0.0352 * 217 - 35.4) / 46 * 46) ;  // JFS2006 p131, 8.4.1-1

        if( stD.cPre = '分娩前9週から4週まで' )  then    //分娩前9週から4週まで
          begin
            dMERC  := dE / 63 / 0.14 * 0.9 ;         // JFS2006 p132, 8.4.2-2

            dDERC  := dMERC / 0.82 ;     // JFS2006 p130、8.3.2-2
            dTDNRC := dDERC / 4.41 ;     // JFS2006 p130、8.3.2-3
            dMJRC  := dMERC * 4.184 ;    // JFS2006 p129、L36

            gdME   := dMME  + dMERC ;    // 代謝エネルギーの総要求量(Mcal)
            gdTDN  := dMTDN + dTDNRC ;   // 可消化養分総量
            gdDE   := dMDE  + dDERC ;    // 可消化エネルギー
            gdMJ   := dMMJ  + dMJRC ;    // 代謝エネルギーの総要求量(MJ)
          end
        else if ( stD.cPre = '分娩前3週から分娩まで' )  then    //分娩前3週から分娩まで
          begin
            dMERC  := dE / 63 / 0.14 * 1.2 ;         // JFS2006 p132, 8.4.2-3

            dDERC  := dMERC / 0.82 ;     // JFS2006 p130、8.3.2-2
            dTDNRC := dDERC / 4.41 ;     // JFS2006 p130、8.3.2-3
            dMJRC  := dMERC * 4.184 ;    // JFS2006 p129、L36

            gdME   := dMME  + dMERC ;    // 代謝エネルギーの総要求量(Mcal)
            gdTDN  := dMTDN + dTDNRC ;   // 可消化養分総量
            gdDE   := dMDE  + dDERC ;    // 可消化エネルギー
            gdMJ   := dMMJ  + dMJRC ;    // 代謝エネルギーの総要求量(MJ)
          end
        else                             //それ以外
          begin   //そのまま
            gdME   := dMME  * 1.1 ;      // 代謝エネルギーの総要求量(Mcal)    // JFS2006 p130、左段、L14
            gdTDN  := dMTDN * 1.1 ;      // 可消化養分総量
            gdDE   := dMDE  * 1.1 ;      // 可消化エネルギー
            gdMJ   := dMMJ  * 1.1 ;      // 代謝エネルギーの総要求量(MJ)
          end
      end

    else if ( stD.cKind = '黒毛和種単子') then  //黒毛和種（単子）
      begin
        // 維持の代謝エネルギー要求量
        dMME  := 0.1163 * power(stD.dWt1,  0.75);        // JFS2006 p129、8.3.2-1

        dMDE  := dMME / 0.82 ;      // JFS2006 p130、8.3.2-2
        dMTDN := dMDE / 4.41 ;      // JFS2006 p130、8.3.2-3
        dMMJ  := dMME * 4.184 ;     // JFS2006 p129、L36

        dE := (1.542 * (1 / power(10, 12)) * power(285, 5.45601))
            - (1.542 * (1 / power(10, 12)) * power(222, 5.45601)) ;  // JFS2006 p132, 8.4.2-5

        if( stD.cPre = '分娩前9週から4週まで' )  then    //分娩前9週から4週まで
          begin
            dMERC  := dE / 63 / 0.123 * 0.9 ;        // JFS2006 p132、8.4.2-8

            dDERC  := dMERC / 0.82 ;     // JFS2006 p130、8.3.2-2
            dTDNRC := dDERC / 4.41 ;     // JFS2006 p130、8.3.2-3
            dMJRC  := dMERC * 4.184 ;    // JFS2006 p129、L36

            gdME   := dMME  + dMERC ;    // 代謝エネルギーの総要求量(Mcal)
            gdTDN  := dMTDN + dTDNRC ;   // 可消化養分総量
            gdDE   := dMDE  + dDERC ;    // 可消化エネルギー
            gdMJ   := dMMJ  + dMJRC ;    // 代謝エネルギーの総要求量(MJ)
          end
        else if ( stD.cPre = '分娩前3週から分娩まで' )  then    //分娩前3週から分娩まで
          begin
            dMERC  := dE / 63 / 0.123 * 1.2 ;        // JFS2006 p132、8.4.2-11

            dDERC  := dMERC / 0.82 ;     // JFS2006 p130、8.3.2-2
            dTDNRC := dDERC / 4.41 ;     // JFS2006 p130、8.3.2-3
            dMJRC  := dMERC * 4.184 ;    // JFS2006 p129、L36

            gdME   := dMME  + dMERC ;    // 代謝エネルギーの総要求量(Mcal)
            gdTDN  := dMTDN + dTDNRC ;   // 可消化養分総量
            gdDE   := dMDE  + dDERC ;    // 可消化エネルギー
            gdMJ   := dMMJ  + dMJRC ;    // 代謝エネルギーの総要求量(MJ)
          end
        else                             //それ以外
          begin   //そのまま
            gdME   := dMME  * 1.1 ;      // 代謝エネルギーの総要求量(Mcal)    // JFS2006 p130、左段、L14
            gdTDN  := dMTDN * 1.1 ;      // 可消化養分総量
            gdDE   := dMDE  * 1.1 ;      // 可消化エネルギー
            gdMJ   := dMMJ  * 1.1 ;      // 代謝エネルギーの総要求量(MJ)
          end
      end

    else if ( stD.cKind = '黒毛和種双子') then   //黒毛和種（双子）
      begin
        // 維持の代謝エネルギー要求量
        dMME  := 0.1163 * power(stD.dWt1,  0.75);        // JFS2006 p129、8.3.2-1

        dMDE  := dMME / 0.82 ;      // JFS2006 p130、8.3.2-2
        dMTDN := dMDE / 4.41 ;      // JFS2006 p130、8.3.2-3
        dMMJ  := dMME * 4.184 ;     // JFS2006 p129、L36

        dE := (2.712 * (1 / power(10, 12)) * power(279, 5.45601))
            - (2.712 * (1 / power(10, 12)) * power(216, 5.45601)) ;  // JFS2006 p132, 8.4.2-6

        if( stD.cPre = '分娩前9週から4週まで' )  then    //分娩前9週から4週まで
          begin
            dMERC := dE / 63 / 0.123 * 0.9 ;        // JFS2006 p132、8.4.2-9

            dDERC  := dMERC / 0.82 ;     // JFS2006 p130、8.3.2-2
            dTDNRC := dDERC / 4.41 ;     // JFS2006 p130、8.3.2-3
            dMJRC  := dMERC * 4.184 ;    // JFS2006 p129、L36

            gdME   := dMME  + dMERC ;    // 代謝エネルギーの総要求量(Mcal)
            gdTDN  := dMTDN + dTDNRC ;   // 可消化養分総量
            gdDE   := dMDE  + dDERC ;    // 可消化エネルギー
            gdMJ   := dMMJ  + dMJRC ;    // 代謝エネルギーの総要求量(MJ)
          end
        else if ( stD.cPre = '分娩前3週から分娩まで' )  then    //分娩前3週から分娩まで
          begin
            dMERC  := dE / 63 / 0.123 * 1.2 ;        // JFS2006 p132、8.4.2-12

            dDERC  := dMERC / 0.82 ;     // JFS2006 p130、8.3.2-2
            dTDNRC := dDERC / 4.41 ;     // JFS2006 p130、8.3.2-3
            dMJRC  := dMERC * 4.184 ;    // JFS2006 p129、L36

            gdME   := dMME  + dMERC ;    // 代謝エネルギーの総要求量(Mcal)
            gdTDN  := dMTDN + dTDNRC ;   // 可消化養分総量
            gdDE   := dMDE  + dDERC ;    // 可消化エネルギー
            gdMJ   := dMMJ  + dMJRC ;    // 代謝エネルギーの総要求量(MJ)
          end
        else                             //それ以外
          begin   //そのまま
            gdME   := dMME  * 1.1 ;      // 代謝エネルギーの総要求量(Mcal)    // JFS2006 p130、左段、L14
            gdTDN  := dMTDN * 1.1 ;      // 可消化養分総量
            gdDE   := dMDE  * 1.1 ;      // 可消化エネルギー
            gdMJ   := dMMJ  * 1.1 ;      // 代謝エネルギーの総要求量(MJ)
          end
      end

    else if ( stD.cKind = '交雑種')  then   //交雑種
      begin
        // 維持の代謝エネルギー要求量
        dMME  := 0.1163 * power(stD.dWt1,  0.75);        // JFS2006 p129、8.3.2-1

        dMDE  := dMME / 0.82 ;      // JFS2006 p130、8.3.2-2
        dMTDN := dMDE / 4.41 ;      // JFS2006 p130、8.3.2-3
        dMMJ  := dMME * 4.184 ;     // JFS2006 p129、L36

        dE := (1.979 * (1 / power(10, 12)) * power(280, 5.45601))
            - (1.979 * (1 / power(10, 12)) * power(217, 5.45601)) ;  // JFS2006 p132, 8.4.2-7

        if( stD.cPre = '分娩前9週から4週まで' )  then    //分娩前9週から4週まで
          begin
            dMERC  := dE / 63 / 0.123 * 0.9 ;        // JFS2006 p132、8.4.2-10

            dDERC  := dMERC / 0.82 ;     // JFS2006 p130、8.3.2-2
            dTDNRC := dDERC / 4.41 ;     // JFS2006 p130、8.3.2-3
            dMJRC  := dMERC * 4.184 ;    // JFS2006 p129、L36

            gdME   := dMME  + dMERC ;    // 代謝エネルギーの総要求量(Mcal)
            gdTDN  := dMTDN + dTDNRC ;   // 可消化養分総量
            gdDE   := dMDE  + dDERC ;    // 可消化エネルギー
            gdMJ   := dMMJ  + dMJRC ;    // 代謝エネルギーの総要求量(MJ)
          end
        else if( stD.cPre = '分娩前3週から分娩まで' ) then      //分娩前3週から分娩まで
          begin
            dMERC  := dE / 63 / 0.123 * 1.2 ;        // JFS2006 p132、8.4.2-13

            dDERC  := dMERC / 0.82 ;     // JFS2006 p130、8.3.2-2
            dTDNRC := dDERC / 4.41 ;     // JFS2006 p130、8.3.2-3
            dMJRC  := dMERC * 4.184 ;    // JFS2006 p129、L36

            gdME   := dMME  + dMERC ;    // 代謝エネルギーの総要求量(Mcal)
            gdTDN  := dMTDN + dTDNRC ;   // 可消化養分総量
            gdDE   := dMDE  + dDERC ;    // 可消化エネルギー
            gdMJ   := dMMJ  + dMJRC ;    // 代謝エネルギーの総要求量(MJ)
          end
        else                             //それ以外
          begin   //そのまま
            gdME   := dMME  * 1.1 ;      // 代謝エネルギーの総要求量(Mcal)    // JFS2006 p130、左段、L14
            gdTDN  := dMTDN * 1.1 ;      // 可消化養分総量
            gdDE   := dMDE  * 1.1 ;      // 可消化エネルギー
            gdMJ   := dMMJ  * 1.1 ;      // 代謝エネルギーの総要求量(MJ)
          end
      end;
end;

//*** 乾乳牛の蛋白質の計算 ***
procedure Drying_Calc_DCP_A( stD : reDRY ) ;
var
    ldPP_0   :  Double ;    //妊娠子宮への蛋白質蓄積量（妊娠期間）
    ldPP_63  :  Double ;    //妊娠子宮への蛋白質蓄積量（妊娠期間-63）
    ldPP     :  Double ;    //妊娠子宮への蛋白質蓄積量（ｇ）
    ldT      :  Double ;    //妊娠期間（日）
    ldT63    :  Double ;    //妊娠期間-63（日）
    ldPPDCPR :  Double ;    //妊娠に必要な可消化蛋白質要求量
    ldFP     :  Double ;    //代謝ふん中蛋白質（g/日）
    ldIDM    :  Double ;    //乾物摂取量増加分（kg/日）
    ldFMEC   :  Double ;    //飼料ME含量（Mcal/kg DM）
    ldIME    :  Double ;    //ME増給分（Mcal/日）
    dWork1   :  Double ;
    dFN      :  Double ;
    dUN      :  Double ;
    dSP      :  Double ;
    dRPC     :  Double ;
    dRP      :  Double ;
    dNP      :  Double ;
    dRD      :  Double ;
    dEP      :  Double ;

begin
    //維持＆増体要求量
    if ( stD.cKind = 'ﾎﾙｽﾀｲﾝ(初妊)' )  then     //ホルスタイン種（初妊）
      begin
        dFN  := 30 * gdDMI / 6.25 ;                       // JFS2006 p-128、8.2.1-2
        dUN  := 2.75 * (power( stD.dWt1, 0.5 ) / 6.25);   // 8.2.1-4
        dSP  := 0.2 * power( stD.dWt1, 0.6 );             // 8.2.1-5
        dRPC := 23.5505 * (1 / power(stD.dWt1, 0.0645));  // 8.2.1-6
        dRP  := 10 * 0.35 * dRPC ;                        // 8.2.1-7（日増体重）
        dNP  := dFN * 6.25 + dUN * 6.25 + dSP + dRP;      // 8.2.1-8
        dRD  := 0.85 ;                                    // JFS2006 p128 表8.2.1
        dEP  := 0.51 ;                                    // JFS2006 p128 表8.2.1
        gdMDCP  := dNP / dEP ;                            // 粗蛋白質維持要求量     JFS2006 p128、8.2.1-1
        gdMCP := gdMDCP * dRD - dFN * 6.25;               // 維持に要する粗蛋白質要求量   8.2.1-9
      end
    else     //ホルスタイン種（初妊）以外
      begin
        gdMCP  := 2.71 * power( stD.dWt1, 0.75 );         // 維持に要する粗蛋白質要求量   JFS2006 p129、8.3.1-1
        gdMDCP := gdMCP / 0.60 ;                          // 粗蛋白質維持要求量           JFS2006 p129、8.3.1-2
      end;

    //妊娠要求量
    if ( stD.cKind = 'ﾎﾙｽﾀｲﾝ(初妊)' )  then
      begin    //ホルスタイン種（初妊）
        ldT     := 280 ;                 //妊娠期間、JFS2006 p130、表8.4
        ldT63   := ldT - 63 ;            //妊娠期間-63
        ldPP_0  := 0.345 * power(ldT,2) - 69.2 * ldT + 2818 ;        // JFS2006 p131、8.4.1-1
        ldPP_63 := 0.345 * power(ldT63,2) - 69.2 * ldT63 + 2818 ;    // JFS2006 p131、8.4.1-1
        ldPP    := ldPP_0 - ldPP_63 ;
        if ( stD.cPre = '分娩前9週から4週まで' )  then         //分娩前9～4週間
            gdPCP := ldPP * 42 / 46 / 63 / 0.33 / 0.7 * 0.9        //妊娠に必要な粗蛋白質要求量   JFS2006 p131、8.4.1-2 & p130、表8.4
        else if ( stD.cPre = '分娩前3週から分娩まで' )  then   //分娩前3週間
            gdPCP := ldPP * 42 / 46 / 63 / 0.33 / 0.7 * 1.2        //妊娠に必要な粗蛋白質要求量   JFS2006 p131、8.4.1-3 & p130、表8.4
        else                                                   //それ以外
            gdPCP := 0 ;                                           //妊娠に必要な粗蛋白質要求量

        ldPPDCPR := gdPCP * 0.60 ;                                 //妊娠に必要な可消化蛋白質要求量
      end;

    if ( stD.cKind = 'ﾎﾙｽﾀｲﾝ(経産)' )  then
      begin    //ホルスタイン種（経産）
        ldT     := 280 ;                     //妊娠期間、JFS2006 p130、表8.4
        ldT63   := ldT - 63 ;                //妊娠期間-63
        ldPP_0  := 0.345 * power(ldT,2) - 69.2 * ldT + 2818 ;        // JFS2006 p131、8.4.1-1
        ldPP_63 := 0.345 * power(ldT63,2) - 69.2 * ldT63 + 2818 ;    // JFS2006 p131、8.4.1-1
        ldPP    := ldPP_0 - ldPP_63 ;
        if ( stD.cPre = '分娩前9週から4週まで' )  then               //分娩前9週から4週まで
            gdPCP := ldPP * 46 / 46 / 63 / 0.33 / 0.7 * 0.9        //妊娠に必要な粗蛋白質要求量  JFS2006 p131、8.4.1-2 & p130、表8.4
        else if ( stD.cPre = '分娩前3週から分娩まで' )  then         //分娩前3週から分娩まで
            gdPCP := ldPP * 46 / 46 / 63 / 0.33 / 0.7 * 1.2        //妊娠に必要な粗蛋白質要求量  JFS2006 p131、8.4.1-3 & p130、表8.4
        else                                                         //それ以外
            gdPCP := 0 ;                                           //妊娠に必要な粗蛋白質要求量

        ldPPDCPR := gdPCP * 0.60 ;                                 //妊娠に必要な可消化蛋白質要求量
      end;

    if ( stD.cKind = '黒毛和種単子' )  then
      begin    //黒毛和種（単子）
        ldT      := 285 ;                     //妊娠期間、JFS2006 p130、表8.4
        ldT63    := ldT - 63 ;                //妊娠期間-63

        ldPP_0   := (1.486 * 0.0001 * power(ldT,3) - 4.247
                 *  0.01 * power(ldT,2) + 3.173 * ldT - 0.328)
                 *  (-0.323 * 0.000001 * power(ldT,3) + 3.000
                 *  0.0001 * power(ldT,2) - 9.430 * 0.01 * ldT
                 +  11.263) * 6.25 ;                               // JFS2006 p131、8.4.1-4
        ldPP_63  := (1.486 * 0.0001 * power(ldT63,3) - 4.247
                 *  0.01 * power(ldT63,2) + 3.173 * ldT63 - 0.328)
                 *  (-0.323 * 0.000001 * power(ldT63,3) + 3.000
                 *  0.0001 * power(ldT63,2) - 9.430 * 0.01 * ldT63
                 +  11.263) * 6.25 ;                               // JFS2006 p131、8.4.1-4
        ldPP     := ldPP_0 - ldPP_63 ;
        if ( stD.cPre = '分娩前9週から4週まで' )  then   //分娩前9週から4週まで
          begin
            ldFMEC := 2.28 ;                  //飼料ME含量（Mcal/kg DM）JFS2006 p131
            ldIME  := 3.30 ;                  //ME増給分（Mcal/日）JFS2006 p31、表2.1.3.1
            ldIDM  := ldIME / ldFMEC ;        //乾物摂取量増加分（kg/日）JFS2006 p131、8.4.1-8
          //ldFP   := 6.25 * 4.8 * ldIDM ;    //代謝ふん中蛋白質（g/日）JFS2006 p131、8.4.1-7
            ldFP   := 30.00 * ldIDM ;         //代謝ふん中蛋白質（g/日）JFS2006 p131、8.4.1-7
          //ldPPDCPR := ldPP * 30 / 38.5 / 63 / 0.6 * 0.9 + ldFP ;      //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-5 & p130、表8.4
            ldPPDCPR := ldPP * 29.996 / 38.5 / 63 / 0.6 * 0.9 + ldFP ;  //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-5 & p130、表8.4
          end
        else if ( stD.cPre = '分娩前3週から分娩まで' )  then  //分娩前3週から分娩まで
          begin
            ldFMEC := 2.42 ;                   //飼料ME含量（Mcal/kg DM）JFS2006 p131
            ldIME  := 4.40 ;                   //ME増給分（Mcal/日）JFS2006 p31、表2.1.3.2
            ldIDM  := ldIME / ldFMEC ;         //乾物摂取量増加分（kg/日）JFS2006 p131、8.4.1-8
          //ldFP   := 6.25 * 4.8 * ldIDM ;     //代謝ふん中蛋白質（g/日）JFS2006 p131、8.4.1-7
            ldFP   := 30.00 * ldIDM ;          //代謝ふん中蛋白質（g/日）JFS2006 p131、8.4.1-7
          //ldPPDCPR := ldPP * 30 / 38.5 / 63 / 0.6 * 1.2 + ldFP ;        //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-6 & p130、表8.4
            ldPPDCPR := ldPP * 29.99625 / 38.5 / 63 / 0.6 * 1.2 + ldFP ;  //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-6 & p130、表8.4
          end
        else                            //それ以外
            ldPPDCPR := 0 ;                    //妊娠に必要な可消化蛋白質要求量

        gdPCP := ldPPDCPR / 0.60 ;             //妊娠に要する粗蛋白質要求量
      end;

    if ( stD.cKind = '黒毛和種双子' )  then
      begin    //黒毛和種（双子）
        ldT      := 279 ;                     //妊娠期間、JFS2006 p130、表8.4
        ldT63    := ldT - 63 ;                //妊娠期間-63
        ldPP_0   := (1.486 * 0.0001 * power(ldT,3) - 4.247
                 *  0.01 * power(ldT,2) + 3.173 * ldT - 0.328)
                 *  (-0.323 * 0.000001 * power(ldT,3) + 3.000
                 *  0.0001 * power(ldT,2) - 9.430 * 0.01 * ldT
                 +  11.263) * 6.25 ;                               // JFS2006 p131、8.4.1-4
        ldPP_63  := (1.486 * 0.0001 * power(ldT63,3) - 4.247
                 *  0.01 * power(ldT63,2) + 3.173 * ldT63 - 0.328)
                 *  (-0.323 * 0.000001 * power(ldT63,3) + 3.000
                 *  0.0001 * power(ldT63,2) - 9.430 * 0.01 * ldT63
                 +  11.263) * 6.25 ;                               // JFS2006 p131、8.4.1-4
        ldPP     := ldPP_0 - ldPP_63 ;
        if ( stD.cPre = '分娩前9週から4週まで' )  then     //分娩前9週から4週まで
          begin
            ldFMEC := 2.28 ;                //飼料ME含量（Mcal/kg DM）JFS2006 p131
            ldIME  := 5.22 ;                //ME増給分（Mcal/日）JFS2006 p31、表2.1.3.1
            ldIDM  := ldIME / ldFMEC ;      //乾物摂取量増加分（kg/日）JFS2006 p131、8.4.1-8
            ldFP   := 6.25 * 4.8 * ldIDM ;  //代謝ふん中蛋白質（g/日）JFS2006 p131、8.4.1-7
          //ldPPDCPR := ldPP * 48 / 38.5 / 63 / 0.6 * 0.9 + ldFP ;         //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-5 & p130、表8.4
            ldPPDCPR := ldPP * 48.02125 / 38.5 / 63 / 0.6 * 0.9 + ldFP ;   //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-5 & p130、表8.4
          end
        else if ( stD.cPre = '分娩前3週から分娩まで' )  then     //分娩前3週から分娩まで
          begin
            ldFMEC := 2.42 ;                //飼料ME含量（Mcal/kg DM）JFS2006 p131
            ldIME  := 6.97 ;                //ME増給分（Mcal/日）JFS2006 p31、表2.1.3.2
            ldIDM  := ldIME / ldFMEC ;      //乾物摂取量増加分（kg/日）JFS2006 p131、8.4.1-8
            ldFP   := 6.25 * 4.8 * ldIDM ;  //代謝ふん中蛋白質（g/日）JFS2006 p131、8.4.1-7
          //ldPPDCPR := ldPP * 48 / 38.5 / 63 / 0.6 * 1.2 + ldFP ;        //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-6 & p130、表8.4
            ldPPDCPR := ldPP * 47.98622 / 38.5 / 63 / 0.6 * 1.2 + ldFP ;  //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-6 & p130、表8.4
          end
        else                                //それ以外
            ldPPDCPR := 0 ;                 //妊娠に必要な可消化蛋白質要求量

        gdPCP := ldPPDCPR / 0.60 ;        //妊娠に必要な粗蛋白質要求量
      end;

    if ( stD.cKind = '交雑種' )  then
      begin    //交雑種（Ｆ１）
        ldT     := 280 ;                    //妊娠期間、JFS2006 p130、表8.4
        ldT63   := ldT - 63 ;               //妊娠期間-63
        ldPP_0  := (1.486 * 0.0001 * power(ldT,3) - 4.247
                *  0.01 * power(ldT,2) + 3.173 * ldT - 0.328)
                *  (-0.323 * 0.000001 * power(ldT,3) + 3.000
                *  0.0001 * power(ldT,2) - 9.430 * 0.01 * ldT
                +  11.263) * 6.25 ;                               // JFS2006 p131、8.4.1-4
        ldPP_63 := (1.486 * 0.0001 * power(ldT63,3) - 4.247
                *  0.01 * power(ldT63,2) + 3.173 * ldT63 - 0.328)
                *  (-0.323 * 0.000001 * power(ldT63,3) + 3.000
                *  0.0001 * power(ldT63,2) - 9.430 * 0.01 * ldT63
                +  11.263) * 6.25 ;                               // JFS2006 p131、8.4.1-4
        ldPP    := ldPP_0 - ldPP_63 ;
        if ( stD.cPre = '分娩前9週から4週まで' )  then     //分娩前9週から4週まで
          begin
            ldFMEC := 2.28 ;                //飼料ME含量（Mcal/kg DM）JFS2006 p131
            ldIME  := 3.88 ;                //ME増給分（Mcal/日）JFS2006 p31、表2.1.3.1
            ldIDM  := ldIME / ldFMEC ;      //乾物摂取量増加分（kg/日）JFS2006 p131、8.4.1-8
            ldFP   := 6.25 * 4.8 * ldIDM ;  //代謝ふん中蛋白質（g/日）JFS2006 p131、8.4.1-7
          //ldPPDCPR := ldPP * 35.6 / 38.5 / 63 / 0.6 * 0.9 + ldFP ;     //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-5 & p130、表8.4
            ldPPDCPR := ldPP * 35.60141 / 38.5 / 63 / 0.6 * 0.9 + ldFP ; //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-5 & p130、表8.4
          end
        else if ( stD.cPre = '分娩前3週から分娩まで' )  then     //分娩前3週から分娩まで
          begin
            ldFMEC := 2.42 ;                //飼料ME含量（Mcal/kg DM）JFS2006 p131
            ldIME  := 5.17 ;                //ME増給分（Mcal/日）JFS2006 p31、表2.1.3.2
            ldIDM  := ldIME / ldFMEC ;      //乾物摂取量増加分（kg/日）JFS2006 p131、8.4.1-8
            ldFP   := 6.25 * 4.8 * ldIDM ;  //代謝ふん中蛋白質（g/日）JFS2006 p131、8.4.1-7
          //ldPPDCPR := ldPP * 35.6 / 38.5 / 63 / 0.6 * 1.2 + ldFP ;     //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-6 & p130、表8.4
            ldPPDCPR := ldPP * 35.61246 / 38.5 / 63 / 0.6 * 1.2 + ldFP ; //妊娠に必要な可消化蛋白質要求量  JFS2006 p131、8.4.1-6 & p130、表8.4
          end
        else                                //それ以外
            ldPPDCPR := 0 ;                 //妊娠に必要な可消化蛋白質要求量

        gdPCP := ldPPDCPR / 0.60 ;          //妊娠に必要な粗蛋白質要求量
    end;

    gdDCPR := gdMCP + ldPPDCPR ;     //可消化粗蛋白質総要求量
    gdCPR  := gdMDCP  + gdPCP  ;     //粗蛋白質総要求量

    if ( stD.cLVCP = 'ON' )  then    // CPの最低値をDMの12%に設定する
      begin
        dWork1 := (gdDMI * 1000) * 0.12 ;
        if (gdCPR < dWork1)  then
          begin
            gdCPR  := dWork1 ;
            gdDCPR := dWork1 * 0.60 ; //可消化粗蛋白質総要求量
          end;
      end;

    gdPDCP := gdPCP * 0.60 ;   // 妊娠に要する可消化蛋白質要求量
end;

//*** 乾乳牛のミネラル＆ビタミンの計算 ***
procedure Drying_Calc_CAPAD_A( stD : reDRY ) ;
var
    dMCa  :  Double ;    // カルシウム維持要求量
    dMP   :  Double ;    // リン維持要求量
    dMVA  :  Double ;    // ビタミンA維持要求量
    dMVD  :  Double ;    // ビタミンD維持要求量
    dHCa  :  Double ;    // カルシウム妊娠末期要求量
    dHP   :  Double ;    // リン妊娠末期要求量
    dHVA  :  Double ;    // ビタミンA妊娠末期要求量
    dHVD  :  Double ;    // ビタミンD妊娠末期要求量

begin
  //維持＆増体要求量
    if ( stD.cKind = 'ﾎﾙｽﾀｲﾝ(初妊)' )  then    //ホルスタイン種（初妊）
      begin
        dMCa := 25.4 + 0.00092 * stD.dWt1 + 3.61 * 0.35 ;  // 維持カルシウム,日増体重(0.35kg), JFS2006 p129, 8.2.3-7
        dMP  := 13.5 + 0.00207 * stD.dWt1 + 8.29 * 0.35 ;  // 維持リン,日増体重(0.35kg), JFS2006 p129, 8.3.3-8
        dMVA := 0.078 * stD.dWt1 ;                         // 維持ビタミンＡ, JFS2006 p129, 8.2.3-9
        dMVD := 0.006 * stD.dWt1 ;                         // 維持ビタミンＤ, JFS2006 p129, 8.2.3-10
      end
    else               //ホルスタイン種（初妊）以外のとき
      begin
        dMCa := 0.0154 * stD.dWt1 / 0.38 ;                 // 維持カルシウム, JFS2006 p130, 8.3.3-1
        dMP  := 0.0143 * stD.dWt1 / 0.5  ;                 // 維持リン,       JFS2006 p130, 8.3.3-2
        dMVA := 0.0424 * stD.dWt1 ;                        // 維持ビタミンＡ, JFS2006 p130, 8.3.3-3
        dMVD := 0.006  * stD.dWt1 ;                        // 維持ビタミンＤ, JFS2006 p130, 8.3.3-4
      end;

  //妊娠末期に維持に加える量
    dHCa := 0.0078 * 1.23 * stD.dWt1 / 0.38 ;  // JFS2006 p132, 8.4.3-1
    dHP  := 0.0047 * 1.23 * stD.dWt1 / 0.5  ;  // JFS2006 p132, 8.4.3-2
    dHVA := 0.0336 * stD.dWt1 ;                // JFS2006 p132, 8.4.3-3
    dHVD := 0.004  * stD.dWt1 ;                // JFS2006 p132, 8.4.3-4

    if( stD.cPre = '分娩前9週から4週まで' )  then        //分娩前９週から４週まで
      begin
        dHCa := dHCa * 0.90 ;    // JFS2006 p132, 右段
        dHP  := dHP  * 0.90 ;
      end
    else if( stD.cPre = '分娩前3週から分娩まで' ) then   //分娩前３週から分娩まで
      begin
        dHCa := dHCa * 1.20 ;    // JFS2006 p132, 右段
        dHP  := dHP  * 1.20 ;
      end
    else if( stD.cPre = 'それ以外' )  then               //それ以外
      begin
        dHCa := dHCa * 0 ;
        dHP  := dHP  * 0 ;
        dHVA := dHVA * 0 ;
        dHVD := dHVD * 0 ;
      end;

  //胎子の品種に関係する計算
    if( stD.cKind = '黒毛和種単子' )  then
      begin
        if(( stD.cPre = '分娩前9週から4週まで' ) or ( stD.cPre = '分娩前3週から分娩まで' )) then
          begin
            dHCa := dHCa * 0.70 ;        // JFS2006 p132, 右段
            dHP  := dHP  * 0.70 ;
          end
      end
    else if( stD.cKind = '黒毛和種双子' )  then
      begin
        if(( stD.cPre = '分娩前9週から4週まで' ) or ( stD.cPre = '分娩前3週から分娩まで' )) then
          begin
            dHCa := dHCa * 1.10 ;        // JFS2006 p132, 右段
            dHP  := dHP  * 1.10 ;
          end
      end
    else if( stD.cKind = '交雑種' )  then
      begin
        if(( stD.cPre = '分娩前9週から4週まで' ) or ( stD.cPre = '分娩前3週から分娩まで' )) then
          begin
            dHCa := dHCa * 0.85 ;        // JFS2006 p132, 右段
            dHP  := dHP  * 0.85 ;
          end
      end;

  //グローバル変数に代入
    gdCa := dMCa + dHCa ;        // カルシウム
    gdP  := dMP  + dHP  ;        // リン
    gdVA := dMVA + dHVA ;        // ビタミンA
    gdVD := dMVD + dHVD ;        // ビタミンD
end;

//*** 育成牛の養分要求量計算 ***
function  GrowingCow_A( stG : reGROW ) : reNUTRIENT ;
var
    stRet : reNUTRIENT ;
begin
    //計算結果出力変数の初期化
  	gdFCM  := 0;            // 4%脂肪補正乳量
  	gdDMI  := 0;		// 乾物摂取量DMI（ｋｇ）
  	gdTDN  := 0;  	        // 可消化養分総量
  	gdDE   := 0;   	        // 可消化エネルギー
  	gdME   := 0;   	        // 代謝エネルギーの総要求量(Mcal)
  	gdMJ   := 0;   	        // 代謝エネルギーの総要求量(MJ)
  	gdCPR  := 0;		// 粗蛋白質要求量
  	gdDCPR := 0; 	        // 可消化蛋白質要求量
  	gdMCP  := 0;		// 維持に要する粗蛋白質要求量
  	gdMDCP := 0;		// 維持に要する粗蛋白質要求量
  	gdPCP  := 0;		// 妊娠に要する粗蛋白質要求量
  	gdPDCP := 0;		// 妊娠に要する粗蛋白質要求量
  	gdCa   := 0;		// カルシウム
  	gdP    := 0;		// リン
  	gdVA   := 0;		// ビタミンA
  	gdVD   := 0;		// ビタミンD

    //養分要求量の計算処理
        Growing_Calc_ME_A( stG );     //エネルギーの計算
  	Growing_Calc_DMI_A( stG );    //乾物摂取量の計算
  	Growing_Calc_DCP_A( stG );    //蛋白質の計算
  	Growing_Calc_CAPAD_A( stG );  //ミネラル＆ビタミンの計算

    //養分要求量計算結果の代入
  	stRet.dFCM	:=	gdFCM;	    // 4%脂肪補正乳量
  	stRet.dDMI	:=	gdDMI;      // 乾物摂取量DMI（ｋｇ
  	stRet.dCP	:=	gdCPR;      // 粗蛋白質要求量
  	stRet.dDCP	:=	gdDCPR;     // 可消化蛋白質要求量
  	stRet.dTDN	:=	gdTDN;      // 可消化養分総量
  	stRet.dDE	:=	gdDE;       // 可消化エネルギー
  	stRet.dMEC	:=	gdME;       // 代謝エネルギーの総要求量(Mcal)
  	stRet.dMEJ	:=	gdMJ;       // 代謝エネルギーの総要求量(MJ)
  	stRet.dCa	:=	gdCa;       // カルシウム
  	stRet.dP	:=	gdP;        // リン
  	stRet.dVA	:=	gdVA;       // ビタミンA
  	stRet.dVD	:=	gdVD;       // ビタミンD

  	GrowingCow_A := stRet ;
end;

//*** 育成牛の乾物摂取量の計算 ***
procedure Growing_Calc_DMI_A( stG : reGROW ) ;
var
    dWrk  :  Double ;
begin
    if ( stG.dWt1 <= 45 )  then
        gdDMI := 0.54
        //------------------------------
        // エネルギー要求量からDMIを算出(JFS2006 p7)
        // 全乳4.5Kg、ME含量：2.7216 Mcal(5.04Mcal/kg DM)、DM量：0.54kg
        // カーフスタータ,乾物含量:88%、ME含量:3.15 Mcal/kg DM
        //------------------------------
        // エネルギー要求量からDMIを算出
        // 全乳4.5Kgではエネルギー要求量を満たせない場合、
        // 不足分を全乳の追加で充足するとしてDMIを計算する。
        // 全乳の乾物含量：12%
        // 全乳4.5KgのME含量：2.7216 Mcal(5.04Mcal/kg DM)
    else if ( ( stG.dWt1 > 45 ) and ( stG.dWt1 <= 66 ) )  then
      begin
        // エネルギー要求量からDMIを算出(JFS2006 p7)
        // 全乳4.5Kg、ME含量：2.7216 Mcal(5.04Mcal/kg DM)、DM量：0.54kg
        // カーフスタータ,乾物含量:88%、ME含量:3.15 Mcal/kg DM
        dWrk  := gdME - 2.7216 ;
        gdDMI := dWrk / 3.15 + 0.54 ;
      end
    else if ( ( stG.dWt1 > 66 ) and ( stG.dWt1 < 75 ) )  then
      begin
        // 育成牛の離乳 ( 0：離乳前、1：離乳後 、体重が66kg＜X＜75kg )
        if ( stG.cWean = '離乳前' )  then
          begin
            dWrk  := gdME - 2.7216 ;
            gdDMI := dWrk / 3.15 + 0.54 ;
          end
        else if ( stG.cWean = '離乳後' )  then
            gdDMI := 0.49137 + 0.01768 * stG.dWt1 + 0.91754 * stG.dInWt1   // JFS2006 p7, 1.1-4
      end
    else
        gdDMI := 0.49137 + 0.01768 * stG.dWt1 + 0.91754 * stG.dInWt1 ;     // JFS2006 p7, 1.1-4
end;

//*** 育成牛のエネルギーの計算 ***
procedure Growing_Calc_ME_A( stG : reGROW ) ;
var
    dMME  :  Double ;    //維持ME(MCal)
    dMMJ  :  Double ;    //維持ME(MJ)
    dMDE  :  Double ;    //維持DE(MCal)
    dMTDN :  Double ;    //維持TDN(kg)

    dGME  :  Double ;    //増体ME(MCal)
    dGMJ  :  Double ;    //増体ME(MJ)
    dGDE  :  Double ;    //増体DE(MCal)
    dGTDN :  Double ;    //増体TDN(kg)
begin
    if ( stG.dWt1 <= 66 )  then   // 体重 66kg 以下
      begin
        //dMME := 0.1183 * Power( stG.dWt1, 0.75 );              // JFS2006 p10、左段、L8～L9 (2006年版)
        dMME  := 0.1152 * Power( stG.dWt1, 0.75 );               // JFS1999 p10  (1999年版)
        dGME  := 0.1205 * stG.dInWt1 * Power( stG.dWt1, 0.75 );  // JFS2006 p10、左段、L10～L12

        dMDE  := dMME / 0.82  ;       // JFS2006 p130、8.3.2-2
        dMTDN := dMDE / 4.41  ;       // JFS2006 p130、8.3.2-3
        dMMJ  := dMME * 4.184 ;       // JFS2006 p129、L36

        dGDE  := dGME / 0.82  ;       // JFS2006 p130、8.3.2-2
        dGTDN := dGDE / 4.41  ;       // JFS2006 p130、8.3.2-3
        dGMJ  := dGME * 4.184 ;       // JFS2006 p129、L36

        // 季節の考慮
        if ( stG.cKset = '考慮しない' ) then        //考慮しない
          begin
                gdME  := dMME  + dGME  ;
                gdMJ  := dMMJ  + dGMJ  ;
                gdDE  := dMDE  + dGDE  ;
                gdTDN := dMTDN + dGTDN ;
          end
        else if ( stG.cKset = '夏期' )  then        //夏期
          begin
                gdME  := (dMME  + dGME)  * 0.9 ;    // JFS2006 p129、左段、L8
                gdMJ  := (dMMJ  + dGMJ)  * 0.9 ;
                gdDE  := (dMDE  + dGDE)  * 0.9 ;
                gdTDN := (dMTDN + dGTDN) * 0.9 ;
          end
        else if ( stG.cKset = '冬期' )  then        //冬期
          begin
                gdME  := (dMME  + dGME)  * 1.1 ;    // JFS2006 p129、左段、L8
                gdMJ  := (dMMJ  + dGMJ)  * 1.1 ;
                gdDE  := (dMDE  + dGDE)  * 1.1 ;
                gdTDN := (dMTDN + dGTDN) * 1.1 ;
          end
        else if ( stG.cKset = '冬期(厳寒地)' )  then  //冬期（厳寒地）
          begin
                gdME  := (dMME  + dGME)  * 1.1 ;      // JFS2006 p129、左段、L8
                gdMJ  := (dMMJ  + dGMJ)  * 1.1 ;
                gdDE  := (dMDE  + dGDE)  * 1.1 ;
                gdTDN := (dMTDN + dGTDN) * 1.1 ;
          end
      end
    else if ( stG.dWt1 <= 120 )  then   // 体重 120kg 以下
      begin
        //dMME := 0.1183 * Power( stG.dWt1, 0.75 );               // JFS2006 p10、左段、L16～L17
        //dGME := 0.1293 * stG.dInWt1 * Power( stG.dWt1, 0.75 );  // JFS2006 p10、左段、L18～L20
        dMME := 0.99966 * (0.1183 * Power(stG.dWt1, 0.75));       // JFS2006 p10、左段、L16～L17
        dGME := 0.99966 * (0.1293 * stG.dInWt1 * Power(stG.dWt1, 0.75));  // JFS2006 p10、左段、L18～L20

        dMDE  := dMME / 0.82  ;       // JFS2006 p130、8.3.2-2
        dMTDN := dMDE / 4.41  ;       // JFS2006 p130、8.3.2-3
        dMMJ  := dMME * 4.184 ;       // JFS2006 p129、L36

        dGDE  := dGME / 0.82  ;       // JFS2006 p130、8.3.2-2
        dGTDN := dGDE / 4.41  ;       // JFS2006 p130、8.3.2-3
        dGMJ  := dGME * 4.184 ;       // JFS2006 p129、L36

        // 季節の考慮
        if ( stG.cKset = '考慮しない' ) then          //考慮しない
          begin
                gdME  := dMME  + dGME  ;
                gdMJ  := dMMJ  + dGMJ  ;
                gdDE  := dMDE  + dGDE  ;
                gdTDN := dMTDN + dGTDN ;
          end
        else if ( stG.cKset = '夏期' )  then          //夏期
          begin
                gdME  := (dMME  + dGME)  * 0.9 ;      // JFS2006 p129、左段、L8
                gdMJ  := (dMMJ  + dGMJ)  * 0.9 ;
                gdDE  := (dMDE  + dGDE)  * 0.9 ;
                gdTDN := (dMTDN + dGTDN) * 0.9 ;
          end
        else if ( stG.cKset = '冬期' )  then          //冬期
          begin
                gdME  := (dMME  + dGME)  * 1.1 ;      // JFS2006 p129、左段、L8
                gdMJ  := (dMMJ  + dGMJ)  * 1.1 ;
                gdDE  := (dMDE  + dGDE)  * 1.1 ;
                gdTDN := (dMTDN + dGTDN) * 1.1 ;
          end
        else if ( stG.cKset = '冬期(厳寒地)' )  then  //冬期（厳寒地）
          begin
                gdME  := (dMME  + dGME)  * 1.1 ;      // JFS2006 p129、左段、L8
                gdMJ  := (dMMJ  + dGMJ)  * 1.1 ;
                gdDE  := (dMDE  + dGDE)  * 1.1 ;
                gdTDN := (dMTDN + dGTDN) * 1.1 ;
          end
      end
    else if ( stG.dWt1 > 120 )  then   // 体重 120kg より以上
      begin
        dMME  := 0.1163 * Power(stG.dWt1, 0.75);               // JFS2006 p10、左段、L24～L25
        dGME  := 0.1355 * stG.dInWt1 * Power(stG.dWt1, 0.75);  // JFS2006 p10、左段、L26～L28

        dMDE  := dMME / 0.82  ;       // JFS2006 p130、8.3.2-2
        dMTDN := dMDE / 4.41  ;       // JFS2006 p130、8.3.2-3
        dMMJ  := dMME * 4.184 ;       // JFS2006 p129、L36

        dGDE  := dGME / 0.82  ;       // JFS2006 p130、8.3.2-2
        dGTDN := dGDE / 4.41  ;       // JFS2006 p130、8.3.2-3
        dGMJ  := dGME * 4.184 ;       // JFS2006 p129、L36

        // 季節の考慮
        if ( stG.cKset = '考慮しない' ) then          //考慮しない
          begin
                gdME  := dMME  + dGME  ;
                gdMJ  := dMMJ  + dGMJ  ;
                gdDE  := dMDE  + dGDE  ;
                gdTDN := dMTDN + dGTDN ;
          end
        else if ( stG.cKset = '夏期' )  then          //夏期
          begin
                gdME  := dMME  + dGME  ;
                gdMJ  := dMMJ  + dGMJ  ;
                gdDE  := dMDE  + dGDE  ;
                gdTDN := dMTDN + dGTDN ;
          end
        else if ( stG.cKset = '冬期' )  then          //冬期
          begin
                gdME  := dMME  * 1.07 + dGME  ;       // JFS2006 p129、左段、L10
                gdMJ  := dMMJ  * 1.07 + dGMJ  ;
                gdDE  := dMDE  * 1.07 + dGDE  ;
                gdTDN := dMTDN * 1.07 + dGTDN ;
          end
        else if ( stG.cKset = '冬期(厳寒地)' )  then  //冬期（厳寒地）
          begin
                gdME  := dMME  * 1.15 + dGME  ;       // JFS2006 p129、左段、L11
                gdMJ  := dMMJ  * 1.15 + dGMJ  ;
                gdDE  := dMDE  * 1.15 + dGDE  ;
                gdTDN := dMTDN * 1.15 + dGTDN ;
          end
      end;
end;

//*** 育成牛の蛋白質の計算 ***
procedure Growing_Calc_DCP_A( stG : reGROW ) ;
var
    dUN  :  Double ;
    dSP  :  Double ;
    dRPC :  Double ;
    dRP  :  Double ;
    dNP  :  Double ;
    dEP, dRD, dTmpCP : Double ;
    dFN  :  Double ;
begin
    if ( stG.dWt1 > 66 )  then
        dFN  := 30   * gdDMI / 6.25                 // JFS2006 p128、8.2.1-2
    else
        dFN  := 12.5 * gdDMI / 6.25 ;               // JFS2006 p128、8.2.1-3

    dUN  := 2.75 * (Power( stG.dWt1, 0.5 ) / 6.25);     // 8.2.1-4
    dSP  := 0.2 * Power( stG.dWt1, 0.6 );               // 8.2.1-5
    dRPC := 23.5505 * (1 / Power(stG.dWt1, 0.0645));    // 8.2.1-6
    dRP  := 10 * stG.dInWt1 * dRPC ;                    // 8.2.1-7
    dNP  := dFN * 6.25 + dUN * 6.25 + dSP + dRP;        // 8.2.1-8

    if ( stG.dWt1 < 67 )  then                  //体重 67kg未満
      begin
        dRD := 0.94 ;                           // JFS2006 p128 表8.2.1
        dEP := 0.75 ;
      end
    else if ((stG.dWt1 >= 67) and (stG.dWt1 < 120))  then   //体重 67kg以上 120kg未満
      begin
        dRD := 0.90 ;                           // JFS2006 p128 表8.2.1
        //dEP := 0.3  ;                         // JFS2006 p128 表8.2.1
        dEP := 0.63 ;                           // JFS2006 p128 表8.2.1 (変換効率0.3を0.63に修正)
      end
    else if ( stG.dWt1 >= 120 )  then           //体重 120kg以上
      begin
        dRD := 0.85 ;                           // JFS2006 p128 表8.2.1
        dEP := 0.51 ;
      end;

    gdCPR  := dNP / dEP ;            // 粗蛋白質要求量　JFS2006 p128、8.2.1-1
    if ( stG.cLVCP = 'ON' )  then    // CPの最低値をDMの12%に設定する JFS2006 p48、右段下
      begin
        dTmpCP := gdDMI * 0.12 * 1000 ;
        if ( gdCPR < dTmpCP ) then
            gdCPR := dTmpCP          // 粗蛋白質要求量 (DMの12%)
      end;

    gdDCPR  := gdCPR * dRD - dFN * 6.25;     // 可消化蛋白質要求量  8.2.1-9
end;

//*** 育成牛のミネラル＆ビタミンの計算 ***
procedure Growing_Calc_CAPAD_A( stG : reGROW ) ;
begin
    if ( stG.dWt1 <= 90 )  then                              // 体重が 90kg以下
      begin
        //gdCa := 0.0213 * stG.dWt1 + 20.9 * stG.dInWt1 ;              // JFS2006 p129, 8.2.3-1
        gdCa := 1.00001 * (0.0213 * stG.dWt1) + 20.9 * stG.dInWt1 ;    // カルシウム　JFS2006 p129, 8.2.3-1
        gdP  := 0.0156 * stG.dWt1 + 10.7 * stG.dInWt1 ;                // リン　　　　JFS2006 p129, 8.2.3-2
      end
    else if (( stG.dWt1 > 90 ) and ( stG.dWt1 <= 250 ))  then    // 体重が 90kgより重く、かつ 250kg以下
      begin
        //gdCa := 8.00  + 0.0367 * stG.dWt1 + 8.48 * stG.dInWt1 ;            // JFS2006 p129, 8.2.3-3
        gdCa := 8.00  + 1.00001 * (0.0367 * stG.dWt1) + 8.48 * stG.dInWt1 ;  // カルシウム　JFS2006 p129, 8.2.3-3
        gdP  := 0.884 + 0.0500 * stG.dWt1 + 4.86 * stG.dInWt1 ;              // リン　　　　JFS2006 p129, 8.2.3-4
      end
    else if (( stG.dWt1 > 250 ) and ( stG.dWt1 <= 400 ))  then   // 体重が 250kgより重く、かつ 400kg以下
      begin
        //gdCa := 13.4 + 0.0184 * stG.dWt1 + 7.17 * stG.dInWt1 ;             // JFS2006 p129, 8.2.3-5
        //gdP  :=  7.2 + 0.0215 * stG.dWt1 + 6.02 * stG.dInWt1 ;             // JFS2006 p129, 8.2.3-6
        gdCa := 13.4 + 1.00001 * (0.0184 * stG.dWt1) + 7.17 * stG.dInWt1 ;   // カルシウム　JFS2006 p129, 8.2.3-5
        gdP  :=  7.2 + 1.00001 * (0.0215 * stG.dWt1) + 6.02 * stG.dInWt1 ;   // リン　　　　JFS2006 p129, 8.2.3-6
      end
    else  // それ以外（体重 400kgより重い）
      begin
        //gdCa := 25.4 + 0.00092 * stG.dWt1 + 3.61 * stG.dInWt1 ;            // JFS2006 p129, 8.2.3-7
        //gdP  := 13.5 + 0.00207 * stG.dWt1 + 8.29 * stG.dInWt1 ;            // JFS2006 p129, 8.2.3-8
        gdCa := 25.4 + 1.00001 * (0.00092 * stG.dWt1) + 3.61 * stG.dInWt1 ;  // カルシウム　JFS2006 p129, 8.2.3-7
        gdP  := 13.5 + 1.00001 * (0.00207 * stG.dWt1) + 8.29 * stG.dInWt1 ;  // リン　　　　JFS2006 p129, 8.2.3-8
      end;

    //gdVA := 0.0078 * stG.dWt1 ;    // ビタミンA  JFS2006 p129, 8.2.3-9
    gdVA := 0.078 * stG.dWt1 ;    // ビタミンA  JFS2006 p129, 8.2.3-9
    gdVD := 0.006 * stG.dWt1 ;    // ビタミンD  JFS2006 p129, 8.2.3-10
end;


//*** １日の増体重計算 ***
function StandardDailyWeightGain(var dWt : Double) : Double;
{    体重kg   １日増体重kg
 ----------------------------
    40～42  	  0.0
      43  	  0.1
      44   	  0.2
      45  	  0.3
    46～47  	  0.4
    48～50  	  0.5
    51～54  	  0.6
    55～59  	  0.7
    60～68  	  0.8
    69～72  	  0.9
    73～129  	  1.0
   130～213  	  0.9
   214～281  	  0.8
   282～344  	  0.7
   345～403  	  0.6
   404～461  	  0.5
   462～517       0.4
   518～572       0.3
   573～626       0.2
   627～680       0.1
   681～707       0.0          }

begin
  if( dWt >=  40 ) then StandardDailyWeightGain := 0.0 ;
  if( dWt >=  43 ) then StandardDailyWeightGain := 0.1 ;
  if( dWt >=  44 ) then StandardDailyWeightGain := 0.2 ;
  if( dWt >=  45 ) then StandardDailyWeightGain := 0.3 ;
  if( dWt >=  46 ) then StandardDailyWeightGain := 0.4 ;
  if( dWt >=  48 ) then StandardDailyWeightGain := 0.5 ;
  if( dWt >=  51 ) then StandardDailyWeightGain := 0.6 ;
  if( dWt >=  55 ) then StandardDailyWeightGain := 0.7 ;
  if( dWt >=  60 ) then StandardDailyWeightGain := 0.8 ;
  if( dWt >=  69 ) then StandardDailyWeightGain := 0.9 ;
  if( dWt >=  73 ) then StandardDailyWeightGain := 1.0 ;
  if( dWt >= 130 ) then StandardDailyWeightGain := 0.9 ;
  if( dWt >= 214 ) then StandardDailyWeightGain := 0.8 ;
  if( dWt >= 282 ) then StandardDailyWeightGain := 0.7 ;
  if( dWt >= 345 ) then StandardDailyWeightGain := 0.6 ;
  if( dWt >= 404 ) then StandardDailyWeightGain := 0.5 ;
  if( dWt >= 462 ) then StandardDailyWeightGain := 0.4 ;
  if( dWt >= 518 ) then StandardDailyWeightGain := 0.3 ;
  if( dWt >= 573 ) then StandardDailyWeightGain := 0.2 ;
  if( dWt >= 627 ) then StandardDailyWeightGain := 0.1 ;
  if( dWt >= 681 ) then StandardDailyWeightGain := 0.0 ;

end;

end.

