/*
 * Copyright 2007-2008 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.chronos.core.impl;

import org.seasar.chronos.core.Scheduler;
import org.seasar.chronos.core.TaskScheduleEntry;
import org.seasar.chronos.core.annotation.task.Task;
import org.seasar.chronos.core.autodetector.TaskClassAutoDetector;
import org.seasar.chronos.core.schedule.TaskScheduleEntryManager;
import org.seasar.chronos.core.task.TaskExecutorService;
import org.seasar.chronos.core.task.TaskValidator;
import org.seasar.framework.container.ComponentDef;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.util.SmartDeployUtil;
import org.seasar.framework.container.util.Traversal;
import org.seasar.framework.log.Logger;
import org.seasar.framework.util.ClassTraversal;
import org.seasar.framework.util.ClassUtil;
import org.seasar.framework.util.tiger.ReflectionUtil;

/**
 * 抽象スケジューラクラスです。
 * 
 * @author j5ik2o
 *
 */
public abstract class AbstractScheduler implements Scheduler {

    protected static final Logger log = Logger
            .getLogger(AbstractScheduler.class);

    protected S2Container s2container;

    protected TaskClassAutoDetector taskClassAutoDetector;

    private TaskValidator taskValidator;

    public AbstractScheduler() {
        super();
    }

    /**
     * タスク名からComponentDefを返します．
     * 
     * @param taskName
     *            タスク名
     * @return ComponentDef
     */
    protected ComponentDef findTaskComponentDefByTaskName(final String taskName) {
        final TaskScheduleEntryManager tcsm = TaskScheduleEntryManager
                .getInstance();
        final Object componentDef = tcsm
                .forEach(new TaskScheduleEntryManager.TaskScheduleEntryHanlder() {
                    public Object processTaskScheduleEntry(
                            final TaskScheduleEntry taskScheduleEntry) {
                        final TaskExecutorService tes = taskScheduleEntry
                                .getTaskExecutorService();
                        final String _taskName = tes.getTaskPropertyReader()
                                .getTaskName(null);
                        if (taskName.equals(_taskName)) {
                            return taskScheduleEntry.getComponentDef();
                        }
                        return null;
                    }
                });
        if (componentDef != null) {
            return (ComponentDef) componentDef;
        }
        return null;
    }

    /**
     * S2コンテナ(非SMART Deploy)上のコンポーネントを検索し，スケジューラに登録します．
     * 
     * @param s2Container
     *            S2コンテナ
     */
    protected boolean registerChildTaskComponent(final S2Container s2Container) {
        return this.registerChildTaskComponent(s2Container, null);
    }

    protected boolean registerChildTaskComponentByTarget(
            final S2Container s2Container,
            final Class<?> targetTaskComponentClass) {
        return this.registerChildTaskComponent(s2Container,
                targetTaskComponentClass);
    }

    private boolean registerChildTaskComponent(final S2Container s2Container,
            final Class<?> targetTaskComponentClass) {
        final Object result = Traversal.forEachComponent(s2Container,
                new Traversal.ComponentDefHandler() {
                    public Object processComponent(
                            final ComponentDef componentDef) {
                        final Class<?> clazz = componentDef.getComponentClass();
                        if (clazz != null) {
                            if (taskValidator.isValid(clazz)) {
                                if (targetTaskComponentClass == null) {
                                    AbstractScheduler.this
                                            .scheduleTask(componentDef);
                                } else if (targetTaskComponentClass
                                        .equals(componentDef
                                                .getComponentClass())) {
                                    return AbstractScheduler.this.scheduleTask(
                                            componentDef, true);
                                }
                            }
                        }
                        return null;
                    }

                });
        return result != null;
    }

    protected abstract void registerTaskFromS2Container();

    /**
     * SMART Deploy上のコンポーネントを検索し，スケジューラに登録します．
     * 
     * @param s2Container
     *            S2コンテナ
     */
    protected boolean registerTaskFromS2ContainerOnSmartDeploy(
            final S2Container s2Container) {
        return this.registerTaskFromS2ContainerOnSmartDeploy(s2Container, null);
    }

    protected boolean registerTaskFromS2ContainerOnSmartDeployByTarget(
            final S2Container s2Container,
            final Class<?> targetTaskComponentClass) {
        return this.registerTaskFromS2ContainerOnSmartDeploy(s2Container,
                targetTaskComponentClass);
    }

    protected boolean registerJavascriptTaskComponent() {
        // ReflectionUtil.forNameNoException(className);
        return false;
    }

    private boolean detectResult = false;

    private boolean registerTaskFromS2ContainerOnSmartDeploy(
            final S2Container s2Container,
            final Class<?> targetTaskComponentClass) {
        detectResult = false;
        if (SmartDeployUtil.isSmartdeployMode(s2Container)) {
            taskClassAutoDetector.detect(new ClassTraversal.ClassHandler() {
                public void processClass(final String packageName,
                        final String shortClassName) {
                    final String name = ClassUtil.concatName(packageName,
                            shortClassName);
                    final Class<?> clazz = ReflectionUtil
                            .forNameNoException(name);
                    if (targetTaskComponentClass == null) {
                        if (null != AbstractScheduler.this.scheduleTask(
                                s2Container, clazz)) {
                            detectResult = true;
                        }
                    } else if (targetTaskComponentClass.equals(clazz)) {
                        if (null != AbstractScheduler.this.scheduleTask(
                                s2Container, clazz, true)) {
                            detectResult = true;
                        }
                    }
                }
            });
        }
        return detectResult;
    }

    protected TaskScheduleEntry scheduleTask(final ComponentDef componentDef) {
        return this.scheduleTask(componentDef, false);
    }

    protected TaskScheduleEntry scheduleTask(
            final ComponentDef taskComponentDef, final boolean force) {
        final Class<?> clazz = taskComponentDef.getComponentClass();
        final Task task = clazz.getAnnotation(Task.class);
        if (!task.autoSchedule() && !force) {
            return null;
        }
        final TaskScheduleEntry taskScheduleEntry = (TaskScheduleEntry) s2container
                .getComponent(TaskScheduleEntry.class);
        taskScheduleEntry.setComponentDef(taskComponentDef);

        final TaskExecutorService tes = (TaskExecutorService) s2container
                .getComponent(TaskExecutorService.class);
        taskScheduleEntry.setTaskExecutorService(tes);

        tes.setComponentDef(taskComponentDef);
        tes.setGetterSignal(this);
        tes.setScheduler(this);
        // taskScheduleEntry.setComponentDef(taskComponentDef);
        return taskScheduleEntry;
    }

    protected TaskScheduleEntry scheduleTask(final S2Container s2Container,
            final Class<?> taskClass, final boolean force) {
        final ComponentDef componentDef = s2Container
                .getComponentDef(taskClass);
        return this.scheduleTask(componentDef, force);
    }

    protected TaskScheduleEntry scheduleTask(final S2Container s2Container,
            final Class<?> taskClass) {
        final ComponentDef componentDef = s2Container
                .getComponentDef(taskClass);
        return this.scheduleTask(componentDef);
    }

    public void setS2Container(final S2Container s2container) {
        this.s2container = s2container;
    }

    public void setTaskClassAutoDetector(
            final TaskClassAutoDetector taskClassAutoDetector) {
        this.taskClassAutoDetector = taskClassAutoDetector;
    }

    public void setTaskValidator(final TaskValidator taskValidator) {
        this.taskValidator = taskValidator;
    }
}