/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.helper;

import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Map;
import java.util.regex.Pattern;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

import jp.sf.fess.Constants;
import jp.sf.fess.FessSystemException;
import jp.sf.fess.util.FessProperties;

import org.apache.commons.lang.StringUtils;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.taglib.S2Functions;
import org.seasar.struts.util.RequestUtil;

public class ViewHelper implements Serializable {

    private static final long serialVersionUID = 1L;

    protected static final String GOOGLE_MOBILE_TRANSCODER_LINK = "http://www.google.co.jp/gwt/n?u=";

    private static final String USER_AGENT_TYPE = "ViewHelper.UserAgent";

    private static final String UA_FIREFOX = "Firefox";

    private static final String UA_OPERA = "Opera";

    private static final String UA_OTHER = "Other";

    @Resource
    protected BrowserTypeHelper browserTypeHelper;

    @Resource
    protected FessProperties crawlerProperties;

    public int mobileDescriptionLength = 50;

    public int pcDescriptionLength = 200;

    public int mobileTitleLength = 50;

    public int pcTitleLength = 50;

    public boolean encodeUrlLink = false;

    public String urlLinkEncoding = Constants.UTF_8;

    public boolean useDigest = true;

    public boolean useHighlightContent = true;

    private String getString(Map<String, Object> doc, String key) {
        Object value = doc.get(key);
        if (value == null) {
            return null;
        }
        return value.toString();
    }

    public String getContentTitle(Map<String, Object> document) {
        int size;
        if (browserTypeHelper.isMobile()) {
            size = mobileTitleLength;
        } else {
            size = pcTitleLength;
        }

        String title;
        if (StringUtil.isNotBlank(getString(document, "title"))) {
            title = getString(document, "title");
        } else {
            title = getString(document, "url");
        }
        return StringUtils.abbreviate(title, size);
    }

    public String getContentDescription(Map<String, Object> document) {
        String[] queries = (String[]) RequestUtil.getRequest().getAttribute(
                Constants.QUERIES);
        int size;
        if (browserTypeHelper.isMobile()) {
            size = mobileDescriptionLength;
        } else {
            size = pcDescriptionLength;
        }

        String digest = getString(document, "digest");
        if (StringUtil.isNotBlank(digest) && useDigest
                && !digest.startsWith(Constants.DIGEST_PREFIX)) {
            return highlight(S2Functions.h(StringUtils.abbreviate(
                    removeEmTag(digest), size)), queries);
        }

        String cache = getString(document, "cache");
        if (cache != null && useHighlightContent) {
            if (queries != null && queries.length > 0) {
                String content = removeEmTag(cache);
                String lContent = content.toLowerCase();
                for (String query : queries) {
                    int pos = lContent.indexOf(query.toLowerCase());
                    if (pos >= 0) {
                        int idx = pos - 10;
                        return highlight(S2Functions.h(StringUtils.abbreviate(
                                content, idx, size)), queries);
                    }
                }
            }
        }

        if (StringUtil.isNotBlank(digest) && useDigest
                && !Constants.DIGEST_PREFIX.equals(digest)) {
            return highlight(S2Functions.h(StringUtils.abbreviate(
                    removeEmTag(digest), size)), queries);
        }

        return "";
    }

    private String removeEmTag(String str) {
        return str.replaceAll("<em>", "").replaceAll("</em>", "");
    }

    protected String highlight(String content, String[] queries) {
        if (StringUtil.isBlank(content) || queries == null
                || queries.length == 0) {
            return content;
        }
        for (String query : queries) {
            content = Pattern.compile(Pattern.quote(query),
                    Pattern.CASE_INSENSITIVE).matcher(content).replaceAll(
                    "<em>" + query + "</em>");
        }
        return content;
    }

    public String getUrlLink(Map<String, Object> document) {
        // file protocol
        String url = getString(document, "url");
        if (url != null && url.startsWith("file:")) {
            HttpServletRequest request = RequestUtil.getRequest();
            String uaType = (String) request.getAttribute(USER_AGENT_TYPE);
            if (uaType == null) {
                String userAgent = request.getHeader("user-agent");
                if (userAgent != null) {
                    if (userAgent.indexOf("Firefox") >= 0) {
                        uaType = UA_FIREFOX;
                    } else if (userAgent.indexOf("Opera") >= 0) {
                        uaType = UA_OPERA;
                    }
                }
                if (uaType == null) {
                    uaType = UA_OTHER;
                }
                request.setAttribute(USER_AGENT_TYPE, uaType);
            }

            if (UA_FIREFOX.equals(uaType)) {
                url = url.replaceFirst("file:////", "file://///");
            } else if (UA_OPERA.equals(uaType)) {
                url = url.replaceFirst("file:////", "file://");
            }

            //            if (url.length() > 7 && url.charAt(5) == '/'
            //                    && url.charAt(6) != '/' && url.charAt(7) != ':') {
            //                // file:/ -> file:///
            //                url = url.replaceFirst("file:/", "file:///");
            //            }

            if (encodeUrlLink) {
                return url;
            } else {
                try {
                    return URLDecoder.decode(url, urlLinkEncoding);
                } catch (UnsupportedEncodingException e) {
                    throw new FessSystemException("Unsupported encoding: "
                            + urlLinkEncoding);
                }
            }
        }

        if (browserTypeHelper.isMobile()) {
            String mobileTrasncoder = crawlerProperties.getProperty(
                    Constants.MOBILE_TRANSCODER_PROPERTY,
                    Constants.EMPTY_STRING);
            if (Constants.GOOGLE_MOBILE_TRANSCODER.equals(mobileTrasncoder)) {
                return getGoogleMobileTranscoderLink(url);
            }
        }
        return url;
    }

    protected String getGoogleMobileTranscoderLink(String url) {
        StringBuilder buf = new StringBuilder(255);
        buf.append(GOOGLE_MOBILE_TRANSCODER_LINK);
        try {
            buf.append(URLEncoder.encode(url, Constants.UTF_8));
        } catch (UnsupportedEncodingException e) {
            return url;
        }
        return buf.toString();
    }
}
