/**
 *
 */
package fs.game.reversi.gl;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.util.HashMap;

import javax.microedition.khronos.opengles.GL10;

import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Bitmap.Config;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.opengl.GLUtils;

/**
 * @author noguchi
 *
 */
public class GLUtility {
	private static final BitmapFactory.Options options = new BitmapFactory.Options();

	private static HashMap<Integer, float[]> verticesPool = new HashMap<Integer, float[]>();
	private static HashMap<Integer, float[]> colorsPool = new HashMap<Integer, float[]>();
	private static HashMap<Integer, float[]> coordsPool = new HashMap<Integer, float[]>();

	private static HashMap<Integer, FloatBuffer> polygonVerticesPool = new HashMap<Integer, FloatBuffer>();
	private static HashMap<Integer, FloatBuffer> polygonColorsPool = new HashMap<Integer, FloatBuffer>();
	private static HashMap<Integer, FloatBuffer> polygonCoordsPool = new HashMap<Integer, FloatBuffer>();

	static {
		// リソースの自動リサイズをしない
		options.inScaled = false;

		// 32Bit画像として読み込む
		options.inPreferredConfig = Config.ARGB_8888;
	}

	public static final FloatBuffer makeFloatBuffer(float[] arr) {
		ByteBuffer bb = ByteBuffer.allocateDirect(arr.length * 4);
		bb.order(ByteOrder.nativeOrder());
		FloatBuffer fb = bb.asFloatBuffer();
		fb.put(arr);
		fb.position(0);

		return fb;
	}

	/**
	 * 正方形を描画する
	 *
	 * @param gl
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param red
	 * @param green
	 * @param blue
	 * @param alpha
	 */
	public static final void drawSquare(GL10 gl, float x, float y, float width,
			float height, float red, float green, float blue, float alpha) {

		float[] vertices = getVertices(8);
		vertices[0] = -0.5f * width + x;
		vertices[1] = -0.5f * height + y;
		vertices[2] = 0.5f * width + x;
		vertices[3] = -0.5f * height + y;
		vertices[4] = -0.5f * width + x;
		vertices[5] = 0.5f * height + y;
		vertices[6] = 0.5f * width + x;
		vertices[7] = 0.5f * height + y;

		float[] colors = getColors(16);
		colors[0] = red;
		colors[1] = green;
		colors[2] = blue;
		colors[3] = alpha;
		colors[4] = red;
		colors[5] = green;
		colors[6] = blue;
		colors[7] = alpha;
		colors[8] = red;
		colors[9] = green;
		colors[10] = blue;
		colors[11] = alpha;
		colors[12] = red;
		colors[13] = green;
		colors[14] = blue;
		colors[15] = alpha;

		FloatBuffer squareVertices = makeVerticesBuffer(vertices);
		FloatBuffer squareColors = makeColorsBuffer(colors);

		gl.glVertexPointer(2, GL10.GL_FLOAT, 0, squareVertices);
		gl.glEnableClientState(GL10.GL_VERTEX_ARRAY);
		gl.glColorPointer(4, GL10.GL_FLOAT, 0, squareColors);
		gl.glEnableClientState(GL10.GL_COLOR_ARRAY);

		gl.glDrawArrays(GL10.GL_TRIANGLE_STRIP, 0, 4);
	}

	/**
	 * 長方形を描画する
	 *
	 * @param gl
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param red
	 * @param green
	 * @param blue
	 * @param alpha
	 */
	public static final void drawRectangle(GL10 gl, float x, float y,
			float width, float height, float red, float green, float blue,
			float alpha) {

		float[] vertices = getVertices(8);
		vertices[0] = -0.5f * width + x;
		vertices[1] = -0.5f * height + y;
		vertices[2] = 0.5f * width + x;
		vertices[3] = -0.5f * height + y;
		vertices[4] = -0.5f * width + x;
		vertices[5] = 0.5f * height + y;
		vertices[6] = 0.5f * width + x;
		vertices[7] = 0.5f * height + y;

		float[] colors = getColors(16);
		colors[0] = red;
		colors[1] = green;
		colors[2] = blue;
		colors[3] = alpha;
		colors[4] = red;
		colors[5] = green;
		colors[6] = blue;
		colors[7] = alpha;
		colors[8] = red;
		colors[9] = green;
		colors[10] = blue;
		colors[11] = alpha;
		colors[12] = red;
		colors[13] = green;
		colors[14] = blue;
		colors[15] = alpha;

		FloatBuffer polygonVertices = makeVerticesBuffer(vertices);
		FloatBuffer polygonColors = makeColorsBuffer(colors);

		gl.glVertexPointer(2, GL10.GL_FLOAT, 0, polygonVertices);
		gl.glEnableClientState(GL10.GL_VERTEX_ARRAY);
		gl.glColorPointer(4, GL10.GL_FLOAT, 0, polygonColors);
		gl.glEnableClientState(GL10.GL_COLOR_ARRAY);

		gl.glDrawArrays(GL10.GL_TRIANGLE_STRIP, 0, 4);
	}

	/**
	 * ポリゴンで円を作成する
	 *
	 * @param gl
	 * @param x
	 * @param y
	 * @param divides
	 * @param radius
	 * @param red
	 * @param green
	 * @param blue
	 * @param alpha
	 */
	public static final void drawCircle(GL10 gl, float x, float y, int divides,
			float radius, float red, float green, float blue, float alpha) {

		float[] vertices = getVertices(divides * 3 * 2);

		int vertexId = 0; // 頂点配列の要素番号を記憶しておくための変数

		for (int i = 0; i < divides; i++) {
			// 演習場のi番目の頂点の角度(ラジアン)を計算する
			float theta1 = 2.0f / (float) divides * (float) i * (float) Math.PI;

			// 演習場の(i+1)番目の頂点の角度(ラジアン)を計算する
			float theta2 = 2.0f / (float) divides * (float) (i + 1)
					* (float) Math.PI;

			// i番目の三角形の0番目の頂点情報をセットする
			vertices[vertexId++] = x;
			vertices[vertexId++] = y;

			// i番目の三角形の1番目の頂点情報をセットする(演習上のi番目の頂点)
			vertices[vertexId++] = (float) Math.cos((double) theta1) * radius
					+ x; // x座標
			vertices[vertexId++] = (float) Math.sin((double) theta1) * radius
					+ y; // x座標

			// i番目の三角形の2番目の頂点情報をセットする(演習上のi+1番目の頂点)
			vertices[vertexId++] = (float) Math.cos((double) theta2) * radius
					+ x; // x座標
			vertices[vertexId++] = (float) Math.sin((double) theta2) * radius
					+ y; // x座標
		}

		FloatBuffer polygonVertices = makeVerticesBuffer(vertices);

		// ポリゴンの色を指定する
		gl.glColor4f(red, green, blue, alpha);
		gl.glDisableClientState(GL10.GL_COLOR_ARRAY);

		gl.glVertexPointer(2, GL10.GL_FLOAT, 0, polygonVertices);
		gl.glEnableClientState(GL10.GL_VERTEX_ARRAY);

		gl.glDrawArrays(GL10.GL_TRIANGLES, 0, divides * 3);
	}

	/**
	 * テクスチャーを描画する
	 *
	 * @param gl
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param texture
	 * @param red
	 * @param green
	 * @param blue
	 * @param alpha
	 */
	public static final void drawTexture(GL10 gl, float x, float y,
			float width, float height, int texture, float red, float green,
			float blue, float alpha) {

		float[] vertices = getVertices(8);
		vertices[0] = -0.5f * width + x;
		vertices[1] = -0.5f * height + y;
		vertices[2] = 0.5f * width + x;
		vertices[3] = -0.5f * height + y;
		vertices[4] = -0.5f * width + x;
		vertices[5] = 0.5f * height + y;
		vertices[6] = 0.5f * width + x;
		vertices[7] = 0.5f * height + y;

		float[] colors = getColors(16);
		colors[0] = red;
		colors[1] = green;
		colors[2] = blue;
		colors[3] = alpha;
		colors[4] = red;
		colors[5] = green;
		colors[6] = blue;
		colors[7] = alpha;
		colors[8] = red;
		colors[9] = green;
		colors[10] = blue;
		colors[11] = alpha;
		colors[12] = red;
		colors[13] = green;
		colors[14] = blue;
		colors[15] = alpha;

		float[] coords = getCoords(8);
		coords[0] = 0.0f;
		coords[1] = 1.0f;
		coords[2] = 1.0f;
		coords[3] = 1.0f;
		coords[4] = 0.0f;
		coords[5] = 0.0f;
		coords[6] = 1.0f;
		coords[7] = 0.0f;

		FloatBuffer polygonVertices = makeVerticesBuffer(vertices);
		FloatBuffer polygonColors = makeColorsBuffer(colors);
		FloatBuffer texCoords = makeCoordsBuffer(coords);

		gl.glEnable(GL10.GL_TEXTURE_2D);

		// テクスチャオブジェクトの指定
		gl.glBindTexture(GL10.GL_TEXTURE_2D, texture);
		gl.glVertexPointer(2, GL10.GL_FLOAT, 0, polygonVertices);
		gl.glEnableClientState(GL10.GL_VERTEX_ARRAY);
		gl.glColorPointer(4, GL10.GL_FLOAT, 0, polygonColors);
		gl.glEnableClientState(GL10.GL_COLOR_ARRAY);
		gl.glTexCoordPointer(2, GL10.GL_FLOAT, 0, texCoords);
		gl.glEnableClientState(GL10.GL_TEXTURE_COORD_ARRAY);

		gl.glDrawArrays(GL10.GL_TRIANGLE_STRIP, 0, 4);

		gl.glDisableClientState(GL10.GL_TEXTURE_COORD_ARRAY);
		gl.glDisable(GL10.GL_TEXTURE_2D);
	}

	/**
	 * テクスチャの一部分を使ってテクスチャを描画する
	 *
	 * @param gl
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param texture
	 * @param u
	 * @param v
	 * @param tex_w
	 * @param tex_h
	 * @param red
	 * @param green
	 * @param blue
	 * @param alpha
	 */
	public static final void drawTexture(GL10 gl, float x, float y,
			float width, float height, int texture, float u, float v,
			float tex_w, float tex_h, float red, float green, float blue,
			float alpha) {

		float[] vertices = getVertices(8);
		vertices[0] = -0.5f * width + x;
		vertices[1] = -0.5f * height + y;
		vertices[2] = 0.5f * width + x;
		vertices[3] = -0.5f * height + y;
		vertices[4] = -0.5f * width + x;
		vertices[5] = 0.5f * height + y;
		vertices[6] = 0.5f * width + x;
		vertices[7] = 0.5f * height + y;

		float[] colors = getColors(16);
		colors[0] = red;
		colors[1] = green;
		colors[2] = blue;
		colors[3] = alpha;
		colors[4] = red;
		colors[5] = green;
		colors[6] = blue;
		colors[7] = alpha;
		colors[8] = red;
		colors[9] = green;
		colors[10] = blue;
		colors[11] = alpha;
		colors[12] = red;
		colors[13] = green;
		colors[14] = blue;
		colors[15] = alpha;

		float[] coords = getCoords(8);
		coords[0] = u;
		coords[1] = v + tex_h;
		coords[2] = u + tex_w;
		coords[3] = v + tex_h;
		coords[4] = u;
		coords[5] = v;
		coords[6] = u + tex_w;
		coords[7] = v;

		FloatBuffer polygonVertices = makeVerticesBuffer(vertices);
		FloatBuffer polygonColors = makeColorsBuffer(colors);
		FloatBuffer texCoords = makeCoordsBuffer(coords);

		gl.glEnable(GL10.GL_TEXTURE_2D);
		gl.glBindTexture(GL10.GL_TEXTURE_2D, texture);
		gl.glVertexPointer(2, GL10.GL_FLOAT, 0, polygonVertices);
		gl.glEnableClientState(GL10.GL_VERTEX_ARRAY);
		gl.glColorPointer(4, GL10.GL_FLOAT, 0, polygonColors);
		gl.glEnableClientState(GL10.GL_COLOR_ARRAY);
		gl.glTexCoordPointer(2, GL10.GL_FLOAT, 0, texCoords);
		gl.glEnableClientState(GL10.GL_TEXTURE_COORD_ARRAY);

		gl.glDrawArrays(GL10.GL_TRIANGLE_STRIP, 0, 4);

		gl.glDisableClientState(GL10.GL_TEXTURE_COORD_ARRAY);
		gl.glDisable(GL10.GL_TEXTURE_2D);
	}

	/**
	 * テクスチャーを読み込む
	 *
	 * @param gl
	 * @param resources
	 * @param resId
	 * @return
	 */
	public static final int loadTexture(GL10 gl, Resources resources, int resId) {
		int[] textures = new int[1];

		// Bitmapの生成
		Bitmap bmp = BitmapFactory.decodeResource(resources, resId, options);
		if (bmp == null) {
			return 0;
		}

		// OpenGL用のテクスチャを生成する
		gl.glGenTextures(1, textures, 0);
		gl.glBindTexture(GL10.GL_TEXTURE_2D, textures[0]);
		GLUtils.texImage2D(GL10.GL_TEXTURE_2D, 0, bmp, 0);
		gl.glTexParameterf(GL10.GL_TEXTURE_2D, GL10.GL_TEXTURE_MIN_FILTER,
				GL10.GL_LINEAR);
		gl.glTexParameterf(GL10.GL_TEXTURE_2D, GL10.GL_TEXTURE_MAG_FILTER,
				GL10.GL_LINEAR);
		gl.glBindTexture(GL10.GL_TEXTURE_2D, 0);

		// OpenGLへの転送が完了したので、VMメモリ上に作成したBitmapを開放する
		bmp.recycle();

		// TextureManagerに登録する
		TextureManager.addTexture(resId, textures[0]);

		return textures[0];
	}

	public static final int loadTexture(GL10 gl, Bitmap bitmap, int id ){
		int[] textures = new int[1];

		// OpenGL用のテクスチャを生成する
		gl.glGenTextures(1, textures, 0);
		gl.glBindTexture(GL10.GL_TEXTURE_2D, textures[0]);
		GLUtils.texImage2D(GL10.GL_TEXTURE_2D, 0, bitmap, 0);
		gl.glTexParameterf(GL10.GL_TEXTURE_2D, GL10.GL_TEXTURE_MIN_FILTER,
				GL10.GL_LINEAR);
		gl.glTexParameterf(GL10.GL_TEXTURE_2D, GL10.GL_TEXTURE_MAG_FILTER,
				GL10.GL_LINEAR);
		gl.glBindTexture(GL10.GL_TEXTURE_2D, 0);

		// TextureManagerに登録する
		TextureManager.addTexture(id, textures[0]);

		return textures[0];
	}

	public static final Bitmap createStringBitmap( String text, int width, int height, float textSize, float red, float green, float blue, float alpha ){
		float[] widths = new float[text.length()];

		Paint paint = new Paint();
		paint.setTextSize(textSize);
		paint.getTextWidths(text, widths);

		float sw = 0.0f;
		for (int i = 0; i < widths.length ; i++){
				sw += widths[i];
		}

		float centerX = ((float)width - sw) / 2.0f;
		float centerY = ((float)height + textSize) / 2.0f;

	    Bitmap bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
	    Canvas canvas = new Canvas(bitmap);

	    paint.setColor(parseColor(red, green, blue, alpha));
	    canvas.drawText(text, centerX, centerY, paint);

		return bitmap;
	}

	/**
	 * 色の変換(float⇒int)
	 *
	 * @param red
	 * @param green
	 * @param blue
	 * @param alpha
	 * @return
	 */
	public static final int parseColor( float red, float green, float blue, float alpha ){
		return Color.argb((int)alpha * 255, (int)red * 255, (int)green * 255, (int)blue * 255);
	}


	/**
	 * 指定した数のVertices配列領域を返します
	 *
	 * @param n
	 * @return
	 */
	public static float[] getVertices(int n) {
		if (verticesPool.containsKey(n)) {
			return verticesPool.get(n);
		} else {
			float[] wkVertices = new float[n];
			verticesPool.put(n, wkVertices);
			return wkVertices;
		}
	}

	/**
	 * 指定した数のColors配列領域を返します
	 *
	 * @param n
	 * @return
	 */
	public static float[] getColors(int n) {
		if (colorsPool.containsKey(n)) {
			return colorsPool.get(n);
		} else {
			float[] wkColors = new float[n];
			colorsPool.put(n, wkColors);
			return wkColors;
		}
	}

	/**
	 * 指定した数のCoords配列領域を返します
	 *
	 * @param n
	 * @return
	 */
	public static float[] getCoords(int n) {
		if (coordsPool.containsKey(n)) {
			return coordsPool.get(n);
		} else {
			float[] wkCoords = new float[n];
			coordsPool.put(n, wkCoords);
			return wkCoords;
		}
	}

	public static FloatBuffer makeVerticesBuffer(float[] arr) {
		FloatBuffer fb = null;
		if (polygonVerticesPool.containsKey(arr.length)) {
			fb = polygonVerticesPool.get(arr.length);
			fb.clear();
			fb.put(arr);
			fb.position(0);
			return fb;
		}
		fb = makeFloatBuffer(arr);
		polygonVerticesPool.put(arr.length, fb);
		return fb;
	}

	public static FloatBuffer makeColorsBuffer(float[] arr) {
		FloatBuffer fb = null;
		if (polygonColorsPool.containsKey(arr.length)) {
			fb = polygonColorsPool.get(arr.length);
			fb.clear();
			fb.put(arr);
			fb.position(0);
			return fb;
		}
		fb = makeFloatBuffer(arr);
		polygonColorsPool.put(arr.length, fb);
		return fb;
	}

	public static FloatBuffer makeCoordsBuffer(float[] arr) {
		FloatBuffer fb = null;
		if (polygonCoordsPool.containsKey(arr.length)) {
			fb = polygonCoordsPool.get(arr.length);
			fb.clear();
			fb.put(arr);
			fb.position(0);
			return fb;
		}
		fb = makeFloatBuffer(arr);
		polygonCoordsPool.put(arr.length, fb);
		return fb;
	}
}
