#include <CoreFoundation/CoreFoundation.h>
#include <CoreServices/CoreServices.h>
#include <QuickLook/QuickLook.h>

#import <Cocoa/Cocoa.h>
#import <WebKit/WebKit.h>

/* -----------------------------------------------------------------------------
   Generate a preview for file

   This function's job is to create preview for designated file
   ----------------------------------------------------------------------------- */

static BOOL getDataAndAttachmentsWithWebArchive(NSData **data, NSDictionary **attachments, WebArchive *archive);
static NSData *dataReplacedURLsWithCID(NSData *srcData, NSURL *baseURL, NSArray *URLs);
static NSData *dataReplacedWithDictionaries(NSData *srcData, NSArray *rangesAndSubstitutes);
static NSData *CSSDataByAdaptingCIDStyleURL(NSData *srcData, NSURL *baseURL, NSArray *URLs);
static NSString *cidWithURL(NSURL *url, BOOL withSchema);

//#define DEBUG


#pragma mark -
#pragma mark Public Functions

OSStatus GeneratePreviewForURL(void *thisInterface, QLPreviewRequestRef preview, CFURLRef url, CFStringRef contentTypeUTI, CFDictionaryRef options)
{
	NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
	
	if (CFEqual(contentTypeUTI, kUTTypeWebArchive))
	{
		NSData *contents = [NSData dataWithContentsOfURL:(NSURL *)url];
		WebArchive *archive = [[WebArchive alloc] initWithData:contents];
		WebResource *mainResource = [archive mainResource];
		
		NSData *mainData;
		NSDictionary *attachments;
		if (getDataAndAttachmentsWithWebArchive(&mainData, &attachments, archive) == NO)
		{
			mainData = [mainResource data];
			attachments = nil;
		}
		
		NSDictionary *properties;
		properties = [NSDictionary dictionaryWithObjectsAndKeys:
					  [mainResource textEncodingName], kQLPreviewPropertyTextEncodingNameKey,
					  [mainResource MIMEType], kQLPreviewPropertyMIMETypeKey,
					  attachments, kQLPreviewPropertyAttachmentsKey,
					  nil];
		// pass the data to the client.
		QLPreviewRequestSetDataRepresentation(preview, (CFDataRef)mainData, kUTTypeHTML, (CFDictionaryRef)properties);
		[archive release];
	}
	[pool drain];
	return noErr;
}

void CancelPreviewGeneration(void* thisInterface, QLPreviewRequestRef preview)
{
	// implement only if supported
}

#pragma mark Private Functions

static BOOL getDataAndAttachmentsWithWebArchive(NSData **data, NSDictionary **attachments, WebArchive *archive)
{
	WebResource *mainResource = [archive mainResource];
	NSArray *subresources = [archive subresources];
	NSArray *subFrameArchives = [archive subframeArchives];
	
	NSArray *subresourceURLs = [subresources valueForKey:@"URL"];
	NSArray *subFrameArchivesURLs = [subFrameArchives valueForKeyPath:@"mainResource.URL"];
	if (subFrameArchivesURLs)
	{
		subresourceURLs = [subresourceURLs arrayByAddingObjectsFromArray:subFrameArchivesURLs];
	}
	
	NSMutableDictionary *mutableAttachments = [NSMutableDictionary dictionaryWithCapacity:[subresources count]];
	for (WebResource *subresource in subresources)
	{
		NSString *MIMEType = [subresource MIMEType];
		NSString *textEncodingName = [subresource textEncodingName];
		NSData *data = [subresource data];
		NSURL *url = [subresource URL];
		if ([MIMEType isEqualToString:@"text/css"])
		{
#ifdef DEBUG
			NSLog(@"%@", [subresource URL]);
			NSLog(@"%@", [subresource textEncodingName]);
#endif
			NSData *newData = CSSDataByAdaptingCIDStyleURL(data, url, subresourceURLs);
			if (newData) data = newData;
		}
		
		NSDictionary *attachment = [NSDictionary dictionaryWithObjectsAndKeys:
									MIMEType,
									(NSString *)kQLPreviewPropertyMIMETypeKey,
									data,
									(NSString *)kQLPreviewPropertyAttachmentDataKey,
									textEncodingName,
									(NSString *)kQLPreviewPropertyTextEncodingNameKey,
									nil];
		// [subresource textEncodingName] can return nil (e.g., the resource represents image.), 
		// thus, [subresource textEncodingName] must be the last items in initializing a dictionary.
		[mutableAttachments setObject:attachment forKey:cidWithURL([subresource URL], NO)];
	}
	
	for (WebArchive *subFrameArchive in subFrameArchives)
	{
		WebResource *subFrameMainResouce = [subFrameArchive mainResource];
		
		NSData *subFrameData;
		NSDictionary *subFrameAttachments;
		if (getDataAndAttachmentsWithWebArchive(&subFrameData, &subFrameAttachments, subFrameArchive) == NO)
		{
			subFrameData = [subFrameMainResouce data];
			subFrameAttachments = nil;
		}
		
		NSDictionary *subFrameAsAttachment = [NSDictionary dictionaryWithObjectsAndKeys:
											  [subFrameMainResouce MIMEType],
											  (NSString *)kQLPreviewPropertyMIMETypeKey,
											  subFrameData,
											  (NSString *)kQLPreviewPropertyAttachmentDataKey,
											  [subFrameMainResouce textEncodingName],
											  (NSString *)kQLPreviewPropertyTextEncodingNameKey,
											  nil];
		[mutableAttachments setObject:subFrameAsAttachment forKey:cidWithURL([[subFrameArchive mainResource] URL], NO)];
		if (subFrameAttachments)
		{
			[mutableAttachments addEntriesFromDictionary:subFrameAttachments];
		}
	}
	
	// create main content.
	// (replace links to subresources from main resource using cid:identifier rule.)
	*data = dataReplacedURLsWithCID([mainResource data], [mainResource URL], subresourceURLs);
	*attachments = mutableAttachments;
	
	return YES;
}

static NSData *dataReplacedURLsWithCID(NSData *srcData, NSURL *baseURL, NSArray *URLs)
{
	const char *pStart = [srcData bytes];
	const char *pEnd = pStart + [srcData length];
	const char *pCur = pStart, *pTagStart, *pTagEnd;
	
	NSMutableArray *rangesAndSubstitutes = [NSMutableArray array];
	
	while (pCur < pEnd)
	{
		// now pointer is outside the tag.
		// seek '<' (tag opening).
		pTagStart = memchr(pCur, '<', pEnd - pCur);
		if (pTagStart == NULL) break;
		pTagStart += 1;
		
		// if '<' is found, seek '>' (tag closing).
		pTagEnd = memchr(pTagStart, '>', pEnd - pTagStart);
		if (pTagEnd == NULL) break;
		
		pCur = pTagEnd + 1;
		
		// if both '<' and '>' are found, seek "href=..." and "src=..." inside tag.
		char buffer[pTagEnd - pTagStart + 1];
		memcpy(buffer, pTagStart, pTagEnd - pTagStart);
		buffer[pTagEnd - pTagStart] = '\0';
		
		const char *pLinkStart, *pLinkEnd;
		
		if ((pLinkStart = strcasestr(buffer, "href=\"")) != NULL)
		{
			pLinkStart += 6;
		}
		else if ((pLinkStart = strcasestr(buffer, "src=\"")) != NULL)
		{
			pLinkStart += 5;
		}
		else
		{
			continue;
		}
		
		pLinkEnd = strcasestr(pLinkStart, "\"");
		if (pLinkEnd == NULL) continue;
		
		NSString *relativeString = [[[NSString alloc] initWithBytes:pLinkStart length:(pLinkEnd - pLinkStart) encoding:NSASCIIStringEncoding] autorelease];
		
		NSURL *url = [[NSURL URLWithString:relativeString relativeToURL:baseURL] absoluteURL];
		NSUInteger index;
		index = [URLs indexOfObject:url];
		if (index != NSNotFound)
		{
			NSData *substitute = [cidWithURL(url, YES) dataUsingEncoding:NSASCIIStringEncoding];
			NSRange range = NSMakeRange((pTagStart - pStart) + (pLinkStart - buffer), pLinkEnd - pLinkStart);
			NSDictionary *dict = [NSDictionary dictionaryWithObjectsAndKeys:
								  [NSValue valueWithRange:range], @"range",
								  substitute, @"substitute",
								  nil];
			[rangesAndSubstitutes addObject:dict];
		}
	}
	
	NSData *dataUsedCIDForLink = dataReplacedWithDictionaries(srcData, rangesAndSubstitutes);
	
#ifdef DEBUG
	[srcData writeToFile:[NSHomeDirectory() stringByAppendingPathComponent:@"src.html"] atomically:YES];
	NSLog(@"%@", rangesAndSubstitutes);
	[dataUsedCIDForLink writeToFile:[NSHomeDirectory() stringByAppendingPathComponent:@"dest.html"] atomically:YES];
#endif
	
	return dataUsedCIDForLink;
}

static NSData *dataReplacedWithDictionaries(NSData *srcData, NSArray *rangesAndSubstitutes)
{
	const char *bytes = [srcData bytes];
	NSMutableData *destData = [NSMutableData dataWithCapacity:[srcData length]];
	NSUInteger location = 0;
	
	for (NSDictionary *dict in rangesAndSubstitutes)
	{
		NSRange range = [[dict objectForKey:@"range"] rangeValue];
		[destData appendBytes:(bytes + location) length:(range.location - location)];
		[destData appendData:[dict objectForKey:@"substitute"]];
		
		location = range.location + range.length;
	}
	[destData appendBytes:(bytes + location) length:([srcData length] - location)];
	
	return destData;
}

NSData *CSSDataByAdaptingCIDStyleURL(NSData *srcData, NSURL *baseURL, NSArray *URLs)
{
	NSUInteger length = [srcData length];
	char buffer[length + 1];
	[srcData getBytes:buffer];
	buffer[length] = '\0';
	NSMutableArray *rangesAndSubstitutes = [NSMutableArray array];
	
	char *pCur = buffer;
	char *pCur2;
	
	while ((pCur = strcasestr(pCur, "url(")) != NULL)
	{
		pCur += 4;
		if ((pCur2 = strstr(pCur, ")")) == NULL)
		{
			return nil;
		}
		NSString *urlString = [[[NSString alloc] initWithBytes:pCur length:(pCur2 - pCur) encoding:NSASCIIStringEncoding] autorelease];
		if (urlString == nil)
		{
			return nil;
		}
		urlString = [urlString stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]];
		if ([urlString length] > 1 && 
			([urlString hasPrefix:@"\""] && [urlString hasSuffix:@"\""] || 
			 [urlString hasPrefix:@"'"] && [urlString hasSuffix:@"'"]))
		{
			urlString = [urlString substringWithRange:NSMakeRange(1, [urlString length] - 2)];
		}
		
		NSURL *url = [[NSURL URLWithString:urlString relativeToURL:baseURL] absoluteURL];
//		NSUInteger index;
//		index = [URLs indexOfObject:url];
//		if (index != NSNotFound)
		{
			NSData *substitute = [cidWithURL(url, YES) dataUsingEncoding:NSASCIIStringEncoding];
			NSRange range = NSMakeRange(pCur - buffer, pCur2 - pCur);
			NSDictionary *dict = [NSDictionary dictionaryWithObjectsAndKeys:
								  [NSValue valueWithRange:range], @"range",
								  substitute, @"substitute",
								  nil];
			[rangesAndSubstitutes addObject:dict];
		}
	}
	NSData *replacedData = dataReplacedWithDictionaries(srcData, rangesAndSubstitutes);
	
#ifdef DEBUG
	NSLog(@"%@", rangesAndSubstitutes);
	[srcData writeToFile:@"a.css" atomically:YES];
	[replacedData writeToFile:@"b.css" atomically:YES];
	return replacedData;
#endif
}

NSString *cidWithURL(NSURL *url, BOOL withSchema)
{
	NSString *pathWithSlashReplaced = [[url path] stringByReplacingOccurrencesOfString:@"/" withString:@"%2F"];
	if (withSchema)
	{
		return [NSString stringWithFormat:@"cid:%@@%@", pathWithSlashReplaced, [url host]];
	}
	else
	{
		return [NSString stringWithFormat:@"%@@%@", pathWithSlashReplaced, [url host]];
	}
}
