﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using DWriteWarpper;

namespace FooEditEngine.WPF
{
    class MarkerFactory : IDisposable
    {
        DFactory factory;
        ResourceManager<HilightType, IMarkerEffecter> resources = new ResourceManager<HilightType, IMarkerEffecter>();
        ResourceManager<HilightType, DColorBrush> brushes = new ResourceManager<HilightType, DColorBrush>();
        ResourceManager<HilightType, DStrokeStyle> strokes = new ResourceManager<HilightType, DStrokeStyle>();
        public const float thickness = 1.0f;
        public const float boldThickness = 2.0f;

        public MarkerFactory(DFactory factory)
        {
            this.strokes[HilightType.Sold] = factory.CrateStrokeStyke(DLineStyle.Solid);
        }

        public IDictionary<HilightType, DColorBrush> Brushes
        {
            get { return this.brushes; }
        }

        public IDictionary<HilightType, DStrokeStyle> Strokes
        {
            get { return this.strokes; }
        }

        public IMarkerEffecter Create(HilightType type)
        {
            IMarkerEffecter effecter;
            if(this.resources.TryGetValue(type,out effecter))
                return effecter;

            switch(type)
            {
                case HilightType.Url:
                    effecter = new LineMarker(this.brushes[type],this.strokes[HilightType.Sold], thickness);
                    break;
                case HilightType.Sold:
                    effecter = new LineMarker(this.brushes[type], this.strokes[HilightType.Sold], thickness);
                    break;
                case HilightType.BoldSold:
                    effecter = new LineMarker(this.brushes[type], this.strokes[HilightType.Sold], boldThickness);
                    break;
                case HilightType.Select:
                    effecter = new HilightMarker(this.brushes[type]);
                    break;
                default:
                    throw new ArgumentOutOfRangeException();
            }
            this.resources.Add(type,effecter);
            return effecter;
        }

        public void Dispose()
        {
            this.brushes.Clear();
            this.strokes.Clear();
        }
    }

    interface IMarkerEffecter
    {
        void Apply(DRenderBase render, DTextLayout layout, int index, int length, double x, double y);
    }

    class LineMarker : IMarkerEffecter
    {
        DColorBrush brush;
        DStrokeStyle stroke;
        float thickness;
        public LineMarker(DColorBrush brush, DStrokeStyle stroke,float thickness)
        {
            this.brush = brush;
            this.stroke = stroke;
            this.thickness = thickness;
        }

        public void Apply(DRenderBase render,DTextLayout layout, int start, int length,double x, double y)
        {
            render.AntialiasMode = DAntialias.Alias;

            DHitTestMetrics[] metrics = layout.HitTestTextRange(start, length, (float)x, (float)y);
            foreach (DHitTestMetrics metric in metrics)
            {
                render.DrawLine(new Point2F(metric.left, metric.top + metric.height),
                    new Point2F(metric.left + metric.width, metric.top + metric.height),
                    this.thickness,
                    this.brush,
                    this.stroke);
            }

            render.AntialiasMode = DAntialias.Antialias;
        }
    }

    class HilightMarker : IMarkerEffecter
    {
        DColorBrush brush;
        public HilightMarker(DColorBrush brush)
        {
            this.brush = brush;
        }

        public void Apply(DRenderBase render,DTextLayout layout, int start, int length, double x, double y)
        {
            render.AntialiasMode = DAntialias.Alias;

            DHitTestMetrics[] metrics = layout.HitTestTextRange(start, length, (float)x, (float)y);
            foreach (DHitTestMetrics metric in metrics)
            {
                render.FillRectangle(new RectF(metric.left, metric.top, metric.left + metric.width, metric.top + metric.height),brush);
            }

            render.AntialiasMode = DAntialias.Antialias;
        }
    }
}
