﻿using System;
using SharpDX;
using D2D = SharpDX.Direct2D1;
using DW = SharpDX.DirectWrite;

namespace FooEditEngine
{
    class CustomTextRenderer : CallbackBase, DW.TextRenderer
    {
        D2D.RenderTarget render;
        ColorBrushCollection brushes;

        public CustomTextRenderer(D2D.RenderTarget render, ColorBrushCollection brushes,Color4 defalut)
        {
            this.render = render;
            this.DefaultFore = defalut;
            this.brushes = brushes;
        }

        public Color4 DefaultFore
        {
            get;
            set;
        }

        #region TextRenderer Members

        public Result DrawGlyphRun(object clientDrawingContext, float baselineOriginX, float baselineOriginY, D2D.MeasuringMode measuringMode, DW.GlyphRun glyphRun, DW.GlyphRunDescription glyphRunDescription, ComObject clientDrawingEffect)
        {
            D2D.SolidColorBrush foreBrush = this.brushes.Get(this.DefaultFore);
            if (clientDrawingEffect != null)
            {
                if (clientDrawingEffect is DrawingEffect)
                {
                    DrawingEffect effect;
                    effect = clientDrawingEffect as DrawingEffect;
                    foreBrush = this.brushes.Get(effect.Fore);
                }
                else if (clientDrawingEffect is D2D.SolidColorBrush)
                {
                    D2D.SolidColorBrush effect;
                    effect = clientDrawingEffect as D2D.SolidColorBrush;
                    foreBrush = effect;
                }
            }

            render.DrawGlyphRun(new DrawingPointF(baselineOriginX, baselineOriginY),
                glyphRun,
                foreBrush,
                measuringMode);

            return SharpDX.Result.Ok;
        }

        public Result DrawInlineObject(object clientDrawingContext, float originX, float originY, DW.InlineObject inlineObject, bool isSideways, bool isRightToLeft, ComObject clientDrawingEffect)
        {
            inlineObject.Draw(this.render, this, originX, originY, isSideways, isRightToLeft, clientDrawingEffect);
            return Result.Ok;
        }

        public Result DrawStrikethrough(object clientDrawingContext, float baselineOriginX, float baselineOriginY, ref DW.Strikethrough strikethrough, ComObject clientDrawingEffect)
        {
            D2D.SolidColorBrush foreBrush = this.brushes.Get(this.DefaultFore);
            DrawingEffect effect = null;
            if (clientDrawingEffect != null && clientDrawingEffect is DrawingEffect)
            {
                effect = clientDrawingEffect as DrawingEffect;
                foreBrush = this.brushes.Get(effect.Fore);
            }
            if (effect == null)
            {
                render.DrawLine(new DrawingPointF(baselineOriginX, baselineOriginY + strikethrough.Offset),
                    new DrawingPointF(baselineOriginX + strikethrough.Width, baselineOriginY + strikethrough.Offset),
                    foreBrush,
                    GetThickness(strikethrough.Thickness));
            }
            else
            {
                LineMarker marker = new LineMarker(render,this.brushes.Get(effect.Fore),effect.Stroke, GetThickness(strikethrough.Thickness));
                marker.Draw(baselineOriginX, baselineOriginY + strikethrough.Offset, strikethrough.Width, 0);
            }
            return Result.Ok;
        }

        public Result DrawUnderline(object clientDrawingContext, float baselineOriginX, float baselineOriginY, ref DW.Underline underline, ComObject clientDrawingEffect)
        {
            D2D.SolidColorBrush foreBrush = this.brushes.Get(this.DefaultFore);
            DrawingEffect effect = null;
            if (clientDrawingEffect != null && clientDrawingEffect is DrawingEffect)
            {
                effect = clientDrawingEffect as DrawingEffect;
                foreBrush = this.brushes.Get(effect.Fore);
            }
            if (effect == null)
            {
                render.DrawLine(new DrawingPointF(baselineOriginX, baselineOriginY + underline.Offset),
                    new DrawingPointF(baselineOriginX + underline.Width, baselineOriginY + underline.Offset),
                    foreBrush,
                    GetThickness(underline.Thickness));
            }
            else if (effect.StrokeType == LineType.Squiggle)
            {
                SquilleLineMarker marker = new D2DSquilleLineMarker(this.render, this.brushes.Get(effect.Fore), effect.Stroke, GetThickness(underline.Thickness));
                marker.Draw(baselineOriginX, baselineOriginY + underline.Offset, underline.Width, underline.RunHeight);
            }
            else
            {
                LineMarker marker = new LineMarker(this.render, this.brushes.Get(effect.Fore), effect.Stroke, GetThickness(underline.Thickness));
                marker.Draw(baselineOriginX, baselineOriginY + underline.Offset, underline.Width, underline.Offset);
            }

            return SharpDX.Result.Ok;
        }

        #endregion

        #region PixelSnapping Members

        public DW.Matrix GetCurrentTransform(object clientDrawingContext)
        {
            Matrix3x2 d2Dmatrix = render.Transform;
            var dwMatrix = new SharpDX.DirectWrite.Matrix()
            {
                M11 = d2Dmatrix.M11,
                M12 = d2Dmatrix.M12,
                M21 = d2Dmatrix.M22,
                M22 = d2Dmatrix.M22,
                Dx = d2Dmatrix.M31,
                Dy = d2Dmatrix.M32
            };
            return dwMatrix;
        }

        public float GetPixelsPerDip(object clientDrawingContext)
        {
            return render.PixelSize.Width / 96f;
        }

        public bool IsPixelSnappingDisabled(object clientDrawingContext)
        {
            return false;
        }

        #endregion

        float GetThickness(float thickness)
        {
            if (this.render.AntialiasMode == D2D.AntialiasMode.Aliased)
                return (int)(thickness + 0.5);
            else
                return thickness;
        }
    }
}
