﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using FooEditEngine;
using SharpDX;
using D2D = SharpDX.Direct2D1;
using DW = SharpDX.DirectWrite;
using DXGI = SharpDX.DXGI;
using System.Runtime.InteropServices;

namespace FooEditEngine
{
    delegate void PreDrawOneLineHandler(MyTextLayout layout);

    /// <summary>
    /// 文字列のアンチエイリアシングモードを指定する
    /// </summary>
    public enum TextAntialiasMode
    {
        /// <summary>
        /// 最適なものが選択されます
        /// </summary>
        Default = D2D.TextAntialiasMode.Default,
        /// <summary>
        /// ClearTypeでアンチエイリアシングを行います
        /// </summary>
        ClearType = D2D.TextAntialiasMode.Cleartype,
        /// <summary>
        /// グレイスケールモードでアンチエイリアシングを行います
        /// </summary>
        GrayScale = D2D.TextAntialiasMode.Grayscale,
        /// <summary>
        /// アンチエイリアシングを行いません
        /// </summary>
        Aliased = D2D.TextAntialiasMode.Aliased,
    }

    class ColorBrushCollection
    {
        ResourceManager<Color4, D2D.SolidColorBrush> cache = new ResourceManager<Color4, D2D.SolidColorBrush>();

        public D2D.RenderTarget Render;

        public D2D.SolidColorBrush Get(Color4 key)
        {
            if (this.Render == null || this.Render.IsDisposed)
                throw new InvalidOperationException();
            D2D.SolidColorBrush brush;
            bool result = cache.TryGetValue(key, out brush);
            if (!result)
            {
                brush = new D2D.SolidColorBrush(this.Render, key);
                cache.Add(key, brush);
            }
            
            return brush;
        }

        public void Clear()
        {
            cache.Clear();
        }
    }

    class D2DRenderCommon : IDisposable
    {
        ResourceManager<HilightType, DrawingEffect> MarkerEffects = new ResourceManager<HilightType, DrawingEffect>();
        ColorBrushCollection Brushes = new ColorBrushCollection();
        InlineManager HiddenChars;
        TextAntialiasMode _TextAntialiasMode;
        Color4 _ControlChar;
        DW.Factory DWFactory;
        D2D.Factory D2DFactory;
        DW.TextFormat format;
        D2D.Bitmap bitmap;
        D2D.RenderTarget render;
        CustomTextRenderer textRender;
        float tabWidth = 64.0f;
        int tabLength = 8;
        bool hasCache;
        Size renderSize = new Size();
        Color4 _Comment, _Literal, _Keyword1, _Keyword2;

#if WINFORM
        public D2DRenderCommon(FooEditEngine.Windows.FooTextBox textbox, double width, double height)
#endif
#if WPF
        public D2DRenderCommon(FooEditEngine.WPF.FooTextBox textbox, double width, double height)
#endif
        {
            this.DWFactory = new DW.Factory(DW.FactoryType.Shared);
            this.D2DFactory = new D2D.Factory(D2D.FactoryType.MultiThreaded);
            this.ChangedRenderResource += (s, e) => { };
            this.ChangedRightToLeft += (s, e) => { };
        }

        public event ChangedRenderResourceEventHandler ChangedRenderResource;

        public event EventHandler ChangedRightToLeft;

        public void InitTextFormat(string fontName, float fontSize,bool rtl)
        {
            if(this.format != null)
                this.format.Dispose();
            
            this.format = new DW.TextFormat(this.DWFactory, fontName, fontSize / 72.0f * 96.0f);
            this.format.WordWrapping = DW.WordWrapping.NoWrap;
            this.format.ReadingDirection = rtl ? DW.ReadingDirection.RightToLeft : DW.ReadingDirection.LeftToRight;

            if (this.HiddenChars == null)
            {
                this.HiddenChars = new InlineManager(this.DWFactory, this.format, this.ControlChar, this.Brushes);
                this.HiddenChars.TabLength = this.tabLength;
            }
            else
                this.HiddenChars.Format = this.format;
            this.hasCache = false;
            this.ChangedRightToLeft(this, null);
            this.ChangedRenderResource(this,new ChangedRenderRsourceEventArgs(ResourceType.Font));
        }

        public TextAntialiasMode TextAntialiasMode
        {
            get
            {
                return this._TextAntialiasMode;
            }
            set
            {
                if (this.render == null)
                    throw new InvalidOperationException();
                this._TextAntialiasMode = value;
                this.render.TextAntialiasMode = (D2D.TextAntialiasMode)value;
                this.ChangedRenderResource(this, new ChangedRenderRsourceEventArgs(ResourceType.Antialias));
            }
        }

        public bool ShowFullSpace
        {
            get
            {
                if (this.HiddenChars == null)
                    return false;
                else
                    return this.HiddenChars.ContainsSymbol('　');
            }
            set
            {
                if (this.HiddenChars == null)
                    throw new InvalidOperationException();
                if (value)
                    this.HiddenChars.AddSymbol('　', '□');
                else
                    this.HiddenChars.RemoveSymbol('　');
                this.ChangedRenderResource(this, new ChangedRenderRsourceEventArgs(ResourceType.InlineChar));
            }
        }

        public bool ShowHalfSpace
        {
            get
            {
                if (this.HiddenChars == null)
                    return false;
                else
                    return this.HiddenChars.ContainsSymbol(' ');
            }
            set
            {
                if (this.HiddenChars == null)
                    throw new InvalidOperationException();
                if (value)
                    this.HiddenChars.AddSymbol(' ', 'ﾛ');
                else
                    this.HiddenChars.RemoveSymbol(' ');
                this.ChangedRenderResource(this, new ChangedRenderRsourceEventArgs(ResourceType.InlineChar));
            }
        }

        public bool ShowTab
        {
            get
            {
                if (this.HiddenChars == null)
                    return false;
                else
                    return this.HiddenChars.ContainsSymbol('\t');
            }
            set
            {
                if (this.HiddenChars == null)
                    throw new InvalidOperationException();
                if (value)
                    this.HiddenChars.AddSymbol('\t', '>');
                else
                    this.HiddenChars.RemoveSymbol('\t');
                this.ChangedRenderResource(this, new ChangedRenderRsourceEventArgs(ResourceType.InlineChar));
            }
        }

        public Color4 Foreground
        {
            get
            {
                return this.MarkerEffects[HilightType.Sold].Fore;
            }
            set
            {
                if (this.render == null)
                    return;
                D2D.StrokeStyleProperties prop = new D2D.StrokeStyleProperties();
                prop.DashCap = D2D.CapStyle.Flat;
                prop.DashOffset = 0;
                prop.DashStyle = D2D.DashStyle.Solid;
                prop.EndCap = D2D.CapStyle.Flat;
                prop.LineJoin = D2D.LineJoin.Miter;
                prop.MiterLimit = 0;
                prop.StartCap = D2D.CapStyle.Flat;
                this.MarkerEffects[HilightType.Sold] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    value);

                prop.DashStyle = D2D.DashStyle.Dash;
                this.MarkerEffects[HilightType.Dash] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    value);

                prop.DashStyle = D2D.DashStyle.DashDot;
                this.MarkerEffects[HilightType.DashDot] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    value);

                prop.DashStyle = D2D.DashStyle.DashDotDot;
                this.MarkerEffects[HilightType.DashDotDot] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    value);

                prop.DashStyle = D2D.DashStyle.Dot;
                this.MarkerEffects[HilightType.Dot] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    value);

                prop.DashStyle = D2D.DashStyle.Solid;
                this.MarkerEffects[HilightType.Squiggle] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    value,
                    LineType.Squiggle);

                if (this.textRender != null)
                    this.textRender.DefaultFore = this.MarkerEffects[HilightType.Sold].Fore;
            }
        }

        public Color4 Background
        {
            get;
            set;
        }

        public Color4 InsertCaret
        {
            get;
            set;
        }

        public Color4 OverwriteCaret
        {
            get;
            set;
        }

        public Color4 LineMarker
        {
            get;
            set;
        }

        public Color4 ControlChar
        {
            get
            {
                return this._ControlChar;
            }
            set
            {
                if (this.render == null)
                    return;
                this._ControlChar = value;
                if (this.HiddenChars != null)
                    this.HiddenChars.Fore = value;
            }
        }

        public Color4 Url
        {
            get
            {
                return this.MarkerEffects[HilightType.Url].Fore;
            }
            set
            {
                if (this.render == null)
                    return;

                D2D.StrokeStyleProperties prop = new D2D.StrokeStyleProperties();
                prop.DashCap = D2D.CapStyle.Flat;
                prop.DashOffset = 0;
                prop.DashStyle = D2D.DashStyle.Solid;
                prop.EndCap = D2D.CapStyle.Flat;
                prop.LineJoin = D2D.LineJoin.Miter;
                prop.MiterLimit = 0;
                prop.StartCap = D2D.CapStyle.Flat;
                this.MarkerEffects[HilightType.Url] = new DrawingEffect(
                    new D2D.StrokeStyle(this.D2DFactory, prop),
                    value);
                this.ChangedRenderResource(this, new ChangedRenderRsourceEventArgs(ResourceType.Brush));
            }
        }

        public Color4 Hilight
        {
            get;
            set;
        }

        public Color4 Comment
        {
            get
            {
                return this._Comment;
            }
            set
            {
                this._Comment = value;
                this.ChangedRenderResource(this, new ChangedRenderRsourceEventArgs(ResourceType.Brush));
            }
        }

        public Color4 Literal
        {
            get
            {
                return this._Literal;
            }
            set
            {
                this._Literal = value;
                this.ChangedRenderResource(this, new ChangedRenderRsourceEventArgs(ResourceType.Brush));
            }
        }

        public Color4 Keyword1
        {
            get
            {
                return this._Keyword1;
            }
            set
            {
                this._Keyword1 = value;
                this.ChangedRenderResource(this, new ChangedRenderRsourceEventArgs(ResourceType.Brush));
            }
        }

        public Color4 Keyword2
        {
            get
            {
                return this._Keyword2;
            }
            set
            {
                this._Keyword2 = value;
                this.ChangedRenderResource(this, new ChangedRenderRsourceEventArgs(ResourceType.Brush));
            }
        }

        public Rectangle ClipRect
        {
            get;
            set;
        }

        const int LineNumberLength = 6;

        public double LineNemberWidth
        {
            get
            {
                DW.TextLayout layout = new DW.TextLayout(this.DWFactory, "0", this.format, float.MaxValue, float.MaxValue);
                layout.ReadingDirection = DW.ReadingDirection.LeftToRight;
                double width = layout.Metrics.Width;
                return width * (LineNumberLength + 1);
            }
        }

        public int TabWidthChar
        {
            get { return this.tabLength; }
            set
            {
                this.tabLength = value;
                DW.TextLayout layout = new DW.TextLayout(this.DWFactory, "0", this.format, float.MaxValue, float.MaxValue);
                this.tabWidth = (float)layout.Metrics.Width * value;
                this.format.IncrementalTabStop = this.tabWidth;
                this.HiddenChars.TabLength = value;
            }
        }

        public void ReConstructDeviceResource()
        {
            this.ReConstructDeviceResource(this.renderSize.Width, this.renderSize.Height);
        }

        public void ReConstructDeviceResource(double width, double height)
        {
            this.DestructDeviceResource();
            this.ConstructDeviceResource(width, height);
        }

        public void DrawCachedBitmap(Rectangle rect)
        {
            if (this.render == null || this.render.IsDisposed)
                return;
            render.DrawBitmap(this.bitmap, rect, 1.0f, D2D.BitmapInterpolationMode.Linear, rect);
        }

        public void CacheContent()
        {
            if (this.render == null || this.bitmap == null || this.bitmap.IsDisposed || this.render.IsDisposed)
                return;
            render.Flush();
            this.bitmap.CopyFromRenderTarget(this.render, new DrawingPoint(), new SharpDX.Rectangle(0, 0, (int)this.renderSize.Width, (int)this.renderSize.Height));
            this.hasCache = true;
        }

        public bool IsVaildCache()
        {
            return this.hasCache;
        }

        public void BegineDraw()
        {
            if (this.render == null || this.render.IsDisposed)
                return;
            this.render.BeginDraw();
            this.render.AntialiasMode = D2D.AntialiasMode.Aliased;
        }

        public System.Action ReCreateTarget;

        public void EndDraw()
        {
            if (this.render == null || this.render.IsDisposed)
                return;
            this.render.AntialiasMode = D2D.AntialiasMode.PerPrimitive;
            this.render.EndDraw();
        }

        public void DrawLineNumber(int lineNumber, double x, double y)
        {
            if (this.render == null || this.render.IsDisposed)
                return;
            string lineNumberFormat = "{0," + LineNumberLength + "}";
            DW.TextLayout layout = new DW.TextLayout(this.DWFactory, string.Format(lineNumberFormat, lineNumber), this.format, float.MaxValue, float.MaxValue);
            layout.ReadingDirection = DW.ReadingDirection.LeftToRight;
            D2D.SolidColorBrush brush = this.Brushes.Get(this.MarkerEffects[HilightType.Sold].Fore);
            this.render.DrawTextLayout(new DrawingPointF((float)x, (float)y), layout, brush);
            layout.Dispose();
        }

        public void DrawLineMarker(Rectangle rect)
        {
            if (this.render == null || this.render.IsDisposed)
                return;
            D2D.SolidColorBrush brush = this.Brushes.Get(this.LineMarker);
            this.render.FillRectangle(rect, brush);
        }

        public void DrawCaret(Rectangle rect, bool transparent)
        {
            if (this.render == null || this.render.IsDisposed)
                return;
            D2D.SolidColorBrush brush;
            if (transparent == false)
                brush = this.Brushes.Get(this.InsertCaret);
            else
                brush = this.Brushes.Get(this.OverwriteCaret);
            this.render.FillRectangle(rect, brush);
        }

        public void FillBackground(Rectangle rect)
        {
            if (this.render == null || this.render.IsDisposed)
                return;
            this.render.Clear(this.Background);
        }

        public void DrawOneLine(LineToIndexTable lti, int row, double x, double y, IEnumerable<Selection> SelectRanges,PreDrawOneLineHandler PreDrawOneLine)
        {
            LineToIndexTableData lineData = lti.GetData(row);

            if (lineData.Length == 0 || this.render == null || this.render.IsDisposed)
                return;

            MyTextLayout layout = (MyTextLayout)lti.GetData(row).Layout;

            this.render.PushAxisAlignedClip(this.ClipRect, D2D.AntialiasMode.Aliased);

            foreach (Selection sel in SelectRanges)
            {
                if (sel.length == 0 || sel.start == -1)
                    continue;

                this.DrawMarkerEffect(layout, sel.hilight, sel.start, sel.length, x, y, false);
            }

            if(PreDrawOneLine != null)
                PreDrawOneLine(layout);

            layout.Draw(textRender, (float)x, (float)y);

            this.render.PopAxisAlignedClip();
        }

        public void DrawMarkerEffect(MyTextLayout layout, HilightType type, int start, int length, double x, double y, bool isBold,Color4? effectColor = null)
        {
            if (type == HilightType.None)
                return;

            float thickness = isBold ? 2 : 1;

            Color4 color;
            if (effectColor != null)
                color = (Color4)effectColor;
            else if (type == HilightType.Select)
                color = this.Hilight;
            else
                color = this.MarkerEffects[type].Fore;

            IMarkerEffecter effecter = null;
            D2D.SolidColorBrush brush = this.Brushes.Get(color);

            if (type == HilightType.Squiggle)
                effecter = new D2DSquilleLineMarker(this.render, brush, this.MarkerEffects[type].Stroke, thickness);
            else if (type == HilightType.Select)
                effecter = new HilightMarker(this.render, brush);
            else if (type == HilightType.None)
                effecter = null;
            else
                effecter = new LineMarker(this.render, brush, this.MarkerEffects[type].Stroke, thickness);

            if (effecter != null)
            {
                render.AntialiasMode = D2D.AntialiasMode.Aliased;

                bool isUnderline = type != HilightType.Select;

                DW.HitTestMetrics[] metrics = layout.HitTestTextRange(start, length, (float)x, (float)y);
                foreach (DW.HitTestMetrics metric in metrics)
                {
                    float offset = isUnderline ? metric.Height : 0;
                    effecter.Draw(metric.Left, metric.Top + offset, metric.Width, metric.Height);
                }

                render.AntialiasMode = D2D.AntialiasMode.PerPrimitive;
            }
        }

        public void SetTextColor(MyTextLayout layout,int start, int length, Color4? color)
        {
            if (color == null)
                return;
            layout.SetDrawingEffect(this.Brushes.Get((Color4)color), new DW.TextRange(start, length));
        }

        public ITextLayout CreateLaytout(string str, SyntaxInfo[] syntaxCollection, IEnumerable<Marker> MarkerRanges)
       {
            MyTextLayout newLayout = new MyTextLayout(new DW.TextLayout(this.DWFactory, str, this.format, (float)this.ClipRect.Width, (float)this.ClipRect.Height));
            ParseLayout(newLayout, str);
            if (syntaxCollection != null)
            {
                foreach (SyntaxInfo s in syntaxCollection)
                {
                    D2D.SolidColorBrush brush = this.Brushes.Get(this.Foreground);
                    switch (s.type)
                    {
                        case TokenType.Comment:
                            brush = this.Brushes.Get(this.Comment);
                            break;
                        case TokenType.Keyword1:
                            brush = this.Brushes.Get(this.Keyword1);
                            break;
                        case TokenType.Keyword2:
                            brush = this.Brushes.Get(this.Keyword2);
                            break;
                        case TokenType.Literal:
                            brush = this.Brushes.Get(this.Literal);
                            break;
                    }
                    newLayout.SetDrawingEffect(brush, new DW.TextRange(s.index, s.length));
                }
            }

            if (MarkerRanges != null)
            {
                foreach (Marker m in MarkerRanges)
                {
                    if (m.start == -1 || m.length == 0)
                        continue;
                    if (m.hilight != HilightType.None)
                        newLayout.SetDrawingEffect(this.MarkerEffects[m.hilight], new DW.TextRange(m.start, m.length));
                    if (m.hilight != HilightType.Select && m.hilight != HilightType.None)
                        newLayout.SetUnderline(true, new DW.TextRange(m.start, m.length));
                }
            }

            return newLayout;
       }

       public List<LineToIndexTableData> BreakLine(Document doc, int startIndex, int endIndex, double wrapwidth)
       {
            List<LineToIndexTableData> output = new List<LineToIndexTableData>();

            this.format.WordWrapping = DW.WordWrapping.Wrap;

            foreach (string str in Util.GetLines(doc, startIndex, endIndex))
            {
                DW.TextLayout layout = new DW.TextLayout(this.DWFactory, str, this.format, (float)wrapwidth, float.MaxValue);

                int i = startIndex;
                foreach (DW.LineMetrics metrics in layout.GetLineMetrics())
                {
                    if (metrics.Length == 0 && metrics.NewlineLength == 0)
                        continue;

                    bool lineend = false;
                    if (metrics.NewlineLength > 0)
                        lineend = true;

                    output.Add(new LineToIndexTableData(i, (int)metrics.Length, lineend, null));
                    i += Util.RoundUp(metrics.Length);
                }

                layout.Dispose();

                startIndex += str.Length;
            }

            this.format.WordWrapping = DW.WordWrapping.NoWrap;

            if (output.Count > 0)
                output.Last().LineEnd = true;

            return output;
        }

        public void Dispose()
        {
            this.DestructDeviceResource();
            this.HiddenChars.Clear();
            if (this.format != null)
                this.format.Dispose();
            if(this.DWFactory != null)
                this.DWFactory.Dispose();
            if(this.D2DFactory != null)
                this.D2DFactory.Dispose();
        }

        public void ConstructDeviceResource(double width, double height)
        {
            int dpiX, dpiY;
            this.GetDpi(out dpiX, out dpiY);
            D2D.RenderTargetProperties prop = new D2D.RenderTargetProperties(
                D2D.RenderTargetType.Default,
                new D2D.PixelFormat(DXGI.Format.B8G8R8A8_UNorm, D2D.AlphaMode.Ignore),
                dpiX,
                dpiY,
                D2D.RenderTargetUsage.None,
                D2D.FeatureLevel.Level_DEFAULT);

            this.render = this.ConstructRender(this.D2DFactory,prop,width,height);

            this.Brushes.Render = this.render;

            D2D.BitmapProperties bmpProp = new D2D.BitmapProperties();
            bmpProp.DpiX = dpiX;
            bmpProp.DpiY = dpiY;
            bmpProp.PixelFormat = new D2D.PixelFormat(DXGI.Format.B8G8R8A8_UNorm, D2D.AlphaMode.Ignore);
            this.bitmap = new D2D.Bitmap(this.render, new DrawingSize((int)width, (int)height), bmpProp);
            this.hasCache = false;

            this.ConstrctedResource();

            this.textRender = new CustomTextRenderer(this.render, this.Brushes, this.MarkerEffects[HilightType.Sold].Fore);

            this.renderSize.Width = width;
            this.renderSize.Height = height;

            this.TextAntialiasMode = this._TextAntialiasMode;
        }


        public System.Func<D2D.Factory,D2D.RenderTargetProperties,double,double,D2D.RenderTarget> ConstructRender;
        public System.Action ConstrctedResource;

        public const int LOGPIXELSX = 88;
        public const int LOGPIXELSY = 90;

        void GetDpi(out int dpiX, out int dpiY)
        {
            IntPtr hDc = NativeMethods.GetDC(IntPtr.Zero);
            dpiX = NativeMethods.GetDeviceCaps(hDc, LOGPIXELSX);
            dpiY = NativeMethods.GetDeviceCaps(hDc, LOGPIXELSY);

            NativeMethods.ReleaseDC(IntPtr.Zero, hDc);
        }

        public System.Action DestructRender;

        public void DestructDeviceResource()
        {
            this.hasCache = false;
            if (this.bitmap != null)
                this.bitmap.Dispose();
            this.MarkerEffects.Clear();
            this.Brushes.Clear();
            if (this.textRender != null)
                this.textRender.Dispose();
            this.DestructRender();
        }

        void ParseLayout(MyTextLayout layout, string str)
        {
            for (int i = 0; i < str.Length; i++)
            {
                DW.InlineObject inlineObject = this.HiddenChars.Get(i, str);
                if (inlineObject != null)
                {
                    layout.SetInlineObject(inlineObject, new DW.TextRange(i, 1));
                    layout.SetDrawingEffect(this.Brushes.Get(this.ControlChar), new DW.TextRange(i, 1));
                }
            }
            return;
        }
    }

    internal static class NativeMethods
    {
        [DllImport("gdi32.dll")]
        public static extern int GetDeviceCaps(IntPtr hDc, int nIndex);

        [DllImport("user32.dll")]
        public static extern IntPtr GetDC(IntPtr hWnd);

        [DllImport("user32.dll")]
        public static extern int ReleaseDC(IntPtr hWnd, IntPtr hDc);
    }
}
