﻿using System;
using System.Linq;
using System.Collections.Generic;

namespace FooEditEngine
{
    /// <summary>
    /// マーカーのタイプを表す列挙体
    /// </summary>
    [Flags]
    public enum HilightType
    {
        /// <summary>
        /// マーカーとして表示しないことを表す
        /// </summary>
        None,
        /// <summary>
        /// 選択状態を表す（内部で使用しているので指定しないでください）
        /// </summary>
        Select,
        /// <summary>
        /// URLを表す
        /// </summary>
        Url,
        /// <summary>
        /// 実線を表す
        /// </summary>
        Sold,
        /// <summary>
        /// 破線を表す
        /// </summary>
        Dash,
        /// <summary>
        /// 一点鎖線を表す
        /// </summary>
        DashDot,
        /// <summary>
        /// 二点鎖線を表す
        /// </summary>
        DashDotDot,
        /// <summary>
        /// 点線を表す
        /// </summary>
        Dot,
        /// <summary>
        /// 波線を表す
        /// </summary>
        Squiggle,
    }

    /// <summary>
    /// マーカー自身を表します
    /// </summary>
    public struct Marker : IRange, IEqualityComparer<Marker>
    {
        #region IRange メンバー

        /// <summary>
        /// 開始位置
        /// </summary>
        public int start
        {
            get;
            set;
        }

        /// <summary>
        /// 長さ
        /// </summary>
        public int length
        {
            get;
            set;
        }

        #endregion

        /// <summary>
        /// マーカーのタイプ
        /// </summary>
        public HilightType hilight { get; set; }

        /// <summary>
        /// マーカーを作成します
        /// </summary>
        /// <param name="start">開始インデックス</param>
        /// <param name="length">長さ</param>
        /// <param name="hilight">タイプ</param>
        /// <returns>マーカー</returns>
        public static Marker Create(int start, int length, HilightType hilight)
        {
            return new Marker { start = start, length = length, hilight = hilight };
        }

        /// <summary>
        /// 等しいかどうかを調べます
        /// </summary>
        /// <param name="x">比較されるマーカー</param>
        /// <param name="y">比較するマーカー</param>
        /// <returns>等しいなら真。そうでなければ偽</returns>
        public bool Equals(Marker x, Marker y)
        {
            return x.hilight == y.hilight && x.length == y.length && x.start == y.start;
        }

        /// <summary>
        /// ハッシュを得ます
        /// </summary>
        /// <param name="obj">マーカー</param>
        /// <returns>ハッシュ</returns>
        public int GetHashCode(Marker obj)
        {
            return this.start ^ this.length ^ (int)this.hilight;
        }
    }

    /// <summary>
    /// マーカークラスのコレクションを表します
    /// </summary>
    public class MarkerCollection
    {
        RangeCollection<Marker> collection = new RangeCollection<Marker>();

        internal MarkerCollection(Document doc)
        {
            doc.Update += new DocumentUpdateEventHandler(doc_Update);
            this.Updated +=new EventHandler((s,e)=>{});
        }

        /// <summary>
        /// 更新されたことを通知します
        /// </summary>
        public event EventHandler Updated;

        internal void Add(Marker m)
        {
            this.collection.Remove(m.start, m.length);
            this.collection.Add(m);
            this.Updated(this, null);
        }

        internal void RemoveAll(int start, int length)
        {
            this.collection.Remove(start, length);
            this.Updated(this, null);
        }

        internal void RemoveAll(HilightType type)
        {
            for (int i = 0; i < this.collection.Count; i++)
            {
                if (this.collection[i].hilight == type)
                    this.collection.RemoveAt(i);
            }
            this.Updated(this, null);
        }

        internal IEnumerable<Marker> Get(int index)
        {
            return this.collection.Get(index);
        }

        internal IEnumerable<Marker> Get(int index, int length)
        {
            return this.collection.Get(index,length);
        }

        /// <summary>
        /// マーカーをすべて削除します
        /// </summary>
        public void Clear()
        {
            this.collection.Clear();
            this.Updated(this, null);
        }

        void doc_Update(object sender, DocumentUpdateEventArgs e)
        {
            int deltaLength = 0;
            switch (e.type)
            {
                case UpdateType.Replace:
                    deltaLength = e.insertLength - e.removeLength;
                    for (int i = 0; i < this.collection.Count; i++)
                    {
                        Marker m = this.collection[i];
                        if (m.start + m.length - 1 <= e.startIndex)
                        {
                            continue;
                        }
                        else
                        {
                            m.start += deltaLength;
                        }
                        this.collection[i] = m;
                    }
                    break;
                default:
                    return;
            }
        }

    }
}
