﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Media.TextFormatting;

namespace FooEditEngine.WPF
{
    class PrintableTextRender : IPrintableTextRender
    {
        FontFamily Font;
        double FontSize;
        DrawingContext Context;
        int _TabWidthChar;
        double TabWidth, _LineNumberWidth,_LineHeight;
        ResourceManager<TokenType, Brush> SysntaxResources = new ResourceManager<TokenType, Brush>();
        ResourceManager<HilightType, Brush> MarkerResources = new ResourceManager<HilightType, Brush>();

        const int LineNumberLength = 6;

        public PrintableTextRender(FontFamily font, double fontSize)
        {
            this.Font = font;
            this.FontSize = fontSize;

            TextLayout layout = new TextLayout("0", this.Font, this.FontSize, Brushes.Black, 0);
            this._LineNumberWidth = layout.Lines[0].Width * LineNumberLength;
            this._LineHeight = layout.Lines[0].Height;
            layout.Dispose();
        }

        public event EventHandler ChangedRightToLeft;

        public bool RightToLeft
        {
            get;
            set;
        }

        public bool InsertMode
        {
            get;
            set;
        }

        public Rectangle ClipRect
        {
            get;
            set;
        }

        public double LineNemberWidth
        {
            get { return this._LineNumberWidth; }
        }

        public int TabWidthChar
        {
            get
            {
                return this._TabWidthChar;
            }
            set
            {
                TextLayout layout = new TextLayout("a", this.Font, this.FontSize, Brushes.Black, 0);
                double width = layout.Lines[0].WidthIncludingTrailingWhitespace;
                layout.Dispose();
                this.TabWidth = width * value;
                this._TabWidthChar = value;
            }
        }

        public Brush Foreground
        {
            get
            {
                return this.SysntaxResources[TokenType.None];
            }
            set
            {
                this.SysntaxResources[TokenType.None] = value;
                this.MarkerResources[HilightType.Sold] = value;
                this.MarkerResources[HilightType.Dash] = value;
                this.MarkerResources[HilightType.DashDot] = value;
                this.MarkerResources[HilightType.DashDotDot] = value;
                this.MarkerResources[HilightType.Dot] = value;
                this.MarkerResources[HilightType.Squiggle] = value;
            }
        }

        public Brush URL
        {
            get
            {
                return this.MarkerResources[HilightType.Url];
            }
            set
            {
                this.MarkerResources[HilightType.Url] = value;
            }
        }

        public Brush Keyword1
        {
            get
            {
                return this.SysntaxResources[TokenType.Keyword1];
            }
            set
            {
                this.SysntaxResources[TokenType.Keyword1] = value;
            }
        }

        public Brush Keyword2
        {
            get
            {
                return this.SysntaxResources[TokenType.Keyword2];
            }
            set
            {
                this.SysntaxResources[TokenType.Keyword2] = value;
            }
        }

        public Brush Comment
        {
            get
            {
                return this.SysntaxResources[TokenType.Comment];
            }
            set
            {
                this.SysntaxResources[TokenType.Comment] = value;
            }
        }

        public Brush Litral
        {
            get
            {
                return this.SysntaxResources[TokenType.Literal];
            }
            set
            {
                this.SysntaxResources[TokenType.Literal] = value;
            }
        }

        public event ChangedRenderResourceEventHandler ChangedRenderResource;

        public void SetDrawingContext(DrawingContext dc)
        {
            this.Context = dc;
        }

        public void DrawCachedBitmap(Rectangle rect)
        {
            return;
        }

        public void CacheContent()
        {
            return;
        }

        public bool IsVaildCache()
        {
            return false;
        }

        public void DrawLineNumber(int lineNumber, double x, double y)
        {
            string lineNumberFormat = "{0," + LineNumberLength + "}";
            TextLayout layout = new TextLayout(string.Format(lineNumberFormat, lineNumber), this.Font, this.FontSize, Brushes.Black, 0);
            layout.Draw(this.Context, x, y);
            layout.Dispose();
        }

        public void DrawLineMarker(Rectangle rect)
        {
            return;
        }

        public void DrawCaret(Rectangle rect,bool transparent)
        {
            return;
        }

        public void FillBackground(Rectangle rect)
        {
            return;
        }

        public void DrawOneLine(LineToIndexTable lti, int row, double x, double y, IEnumerable<Selection> SelectRanges)
        {
            TextLayout layout = (TextLayout)lti.GetData(row).Layout;
            LineToIndexTableData lineData = lti.GetData(row);

            if (lineData.Length == 0)
                return;

            foreach (Marker m in layout.Markers)
            {
                if (m.start == -1 || m.length == 0)
                    continue;
                
                Pen pen = new Pen(this.MarkerResources[m.hilight], 1.0);
                switch (m.hilight)
                {
                    case HilightType.Sold:
                    case HilightType.Url:
                    case HilightType.Squiggle:
                        pen.DashStyle = DashStyles.Solid;
                        break;
                    case HilightType.Dash:
                        pen.DashStyle = DashStyles.Dash;
                        break;
                    case HilightType.DashDot:
                        pen.DashStyle = DashStyles.DashDot;
                        break;
                    case HilightType.DashDotDot:
                        pen.DashStyle = DashStyles.DashDotDot;
                        break;
                    case HilightType.Dot:
                        pen.DashStyle = DashStyles.Dot;
                        break;
                }
                pen.Freeze();

                if (m.hilight == HilightType.Url)
                {
                    TextEffect effect = new TextEffect(null, this.MarkerResources[m.hilight], null, m.start, m.length);
                    effect.Freeze();
                    layout.SetTextEffect(effect);
                }

                if (m.hilight == HilightType.Squiggle)
                {
                    SquilleLineMarker marker = new WPFSquilleLineMarker(this.Context, pen);
                    foreach(TextBounds bound in layout.GetTextBounds(m.start,m.length))
                    {
                        Rect rect = bound.Rectangle;
                        marker.Draw(x + rect.Left,y + rect.Top + rect.Height,rect.Width,rect.Height);
                    }
                }
                else
                {
                    TextDecorationCollection collection = new TextDecorationCollection();
                    TextDecoration decoration = new TextDecoration();
                    decoration.Pen = pen;
                    decoration.Location = TextDecorationLocation.Underline;
                    decoration.Freeze();
                    collection.Add(decoration);

                    layout.SetTextDecoration(m.start, m.length, collection);
                }
            }

            layout.Draw(this.Context, x, y);
            layout.Dispose();
        }

        public List<LineToIndexTableData> BreakLine(Document doc, int startIndex, int endIndex, double wrapwidth)
        {
            List<LineToIndexTableData> output = new List<LineToIndexTableData>();

            foreach (string str in Util.GetLines(doc, startIndex, endIndex))
            {
                TextLayout layout = new TextLayout(str, this.Font, this.FontSize,Brushes.Black, wrapwidth);

                int i = startIndex;
                foreach (TextLine line in layout.Lines)
                {
                    if (line.Length == 0 && line.NewlineLength == 0)
                        continue;

                    int length = line.Length;

                    IList<TextSpan<TextRun>> runs = line.GetTextRunSpans();
                    if (runs.Last().Value is TextEndOfParagraph)
                        length--;

                    bool lineend = false;
                    if (line.NewlineLength > 0)
                        lineend = true;

                    output.Add(new LineToIndexTableData(i, length, lineend, null));
                    i += line.Length;
                }

                layout.Dispose();

                startIndex += str.Length;
            }

            if (output.Count > 0)
                output.Last().LineEnd = true;

            return output;
        }

        public void DrawString(string str, double x, double y, StringAlignment align)
        {
            TextAlignment TextAlign = TextAlignment.Left;
            switch (align)
            {
                case StringAlignment.Left:
                    TextAlign = TextAlignment.Left;
                    break;
                case StringAlignment.Center:
                    TextAlign = TextAlignment.Center;
                    break;
                case StringAlignment.Right:
                    TextAlign = TextAlignment.Right;
                    break;
            }
            TextLayout layout = new TextLayout(str, this.Font, this.FontSize, Brushes.Black, this.ClipRect.Width, TextAlign);
            layout.Draw(this.Context, x, y);
            layout.Dispose();
        }

        public ITextLayout CreateLaytout(string str, SyntaxInfo[] syntaxCollection, IEnumerable<Marker> MarkerRanges)
        {
            TextLayout layout = new TextLayout(str,this.Font,this.FontSize,Brushes.Black,this.ClipRect.Width);
            layout.FlowDirection = this.RightToLeft ? FlowDirection.RightToLeft : FlowDirection.LeftToRight;
            layout.Markers = MarkerRanges;

            if (syntaxCollection != null)
            {
                foreach (SyntaxInfo s in syntaxCollection)
                {
                    TextEffect effect = new TextEffect(null, this.SysntaxResources[s.type], null, s.index, s.length);
                    effect.Freeze();
                    layout.SetTextEffect(effect);
                }
            }
            
            return layout;
        }

        public float HeaderHeight
        {
            get { return (float)this._LineHeight; }
        }

        public float FooterHeight
        {
            get { return (float)this._LineHeight; }
        }
    }

    class WPFSquilleLineMarker : SquilleLineMarker
    {
        DrawingContext Context;
        Pen Pen;
        public WPFSquilleLineMarker(DrawingContext context,Pen pen)
        {
            this.Context = context;
            this.Pen = pen;
        }

        public override void DrawLine(double x, double y, double tox, double toy)
        {
            this.Context.DrawLine(this.Pen, new System.Windows.Point(x, y), new System.Windows.Point(tox, toy));
        }
    }
}
