﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Media;
using FooEditEngine;
using FooEditEngine.WPF.Properties;
using DotNetTextStore;
using DotNetTextStore.UnmanagedAPI.WinDef;
using DWriteWarpper;

namespace FooEditEngine.WPF
{
    class WPFRender : ITextRender,IDisposable
    {
        CacheManager<string, DTextLayout> Cache = new CacheManager<string, DTextLayout>();
        ResourceManager<TokenType, DColorBrush> SyntaxResources = new ResourceManager<TokenType, DColorBrush>();
        Size textureSize;
        Color ForegroundColor, BackgroundColor, HilightColor, Keyword1Color, Keyword2Color, LiteralColor, UrlColor, ControlCharColor,CommentColor;
        DColorBrush background;
        MarkerFactory markerFactory;
        TextStore store;
        D2DTexture texture;
        DRenderBase render;
        DXGISurface surface;
        DFactory factory;
        D3D10Device device;
        D3D9Device device9;
        D3D9Surface surface9;
        DTextFormat format;
        DBitmap bitmap;
        double fontSize;
        FontFamily fontFamily;
        int tabLength;
        bool hasCache;

        public WPFRender(FooTextBox textbox,double width,double height)
        {
            this.fontFamily = textbox.FontFamily;
            this.fontSize = textbox.FontSize;
            this.ForegroundColor = textbox.Foreground;
            this.BackgroundColor = textbox.Background;
            this.ControlCharColor = textbox.ControlChar;
            this.HilightColor = textbox.Hilight;
            this.CommentColor = textbox.Comment;
            this.UrlColor = textbox.URL;
            this.Keyword1Color = textbox.Keyword1;
            this.Keyword2Color = textbox.Keyword2;
            this.LiteralColor = textbox.Literal;
            this.store = textbox.TextStore;

            this.ConstructResource(width, height);
        }

        public D3D9Surface Surface
        {
            get { return this.surface9; }
        }

        public FontFamily FontFamily
        {
            get { return this.fontFamily; }
            set {
                this.fontFamily = value;
                this.DestructResource();
                this.ConstructResource(this.textureSize.Width,this.textureSize.Height);
                this.ChangedRenderResource(this, null);
            }
        }

        public double FontSize
        {
            get { return this.fontSize; }
            set {
                this.fontSize = value;
                this.DestructResource();
                this.ConstructResource(this.textureSize.Width, this.textureSize.Height);
                this.ChangedRenderResource(this, null);
            }
        }

        public Color Foreground
        {
            get
            {
                return this.ForegroundColor;
            }
            set
            {
                this.ForegroundColor = value;
                Color2F color = new Color2F(value.R,value.G,value.B,value.A);
                this.markerFactory.Brushes[HilightType.Sold] = this.render.CreateBrush(color);
                this.markerFactory.Brushes[HilightType.BoldSold] = this.render.CreateBrush(color);
                if(this.render.HasTextRender)
                    this.render.SetDefalutBrush(this.markerFactory.Brushes[HilightType.Sold]);
                this.Cache.Clear();
            }
        }

        public Color Background
        {
            get
            {
                return this.BackgroundColor;
            }
            set
            {
                this.BackgroundColor = value;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.background = this.render.CreateBrush(color);
            }
        }

        public Color ControlChar
        {
            get
            {
                return this.ControlCharColor;
            }
            set
            {
                this.ControlCharColor = value;
                Color2F color = new Color2F(value.R,value.G,value.B,value.A);
                this.SyntaxResources[TokenType.Control] = this.render.CreateBrush(color);
                if (this.render.HasTextRender)
                    this.render.SetDefalutBrush(this.SyntaxResources[TokenType.Control]);
                this.Cache.Clear();
            }
        }
        
        public Color Url
        {
            get
            {
                return this.UrlColor;
            }
            set
            {
                this.UrlColor = value;
                Color2F color = new Color2F(value.R,value.G,value.B,value.A);
                this.markerFactory.Brushes[HilightType.Url] = this.render.CreateBrush(color);
                this.Cache.Clear();
            }
        }

        public Color Hilight
        {
            get
            {
                return this.HilightColor;
            }
            set
            {
                this.HilightColor = value;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.markerFactory.Brushes[HilightType.Select] = this.render.CreateBrush(color);
                this.Cache.Clear();
            }
        }

        public Color Comment
        {
            get
            {
                return this.CommentColor;
            }
            set
            {
                this.CommentColor = value;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.SyntaxResources[TokenType.Comment] = this.render.CreateBrush(color);
                this.Cache.Clear();
            }
        }

        public Color Literal
        {
            get
            {
                return this.LiteralColor;
            }
            set
            {
                this.LiteralColor = value;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.SyntaxResources[TokenType.Literal] = this.render.CreateBrush(color);
                this.Cache.Clear();
            }
        }

        public Color Keyword1
        {
            get
            {
                return this.Keyword1Color;
            }
            set
            {
                this.Keyword1Color = value;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.SyntaxResources[TokenType.Keyword1] = this.render.CreateBrush(color);
                this.Cache.Clear();
            }
        }

        public Color Keyword2
        {
            get
            {
                return this.Keyword2Color;
            }
            set
            {
                this.Keyword2Color = value;
                Color2F color = new Color2F(value.R, value.G, value.B, value.A);
                this.SyntaxResources[TokenType.Keyword2] = this.render.CreateBrush(color);
                this.Cache.Clear();
            }
        }

        public Rectangle ClipRect
        {
            get;
            set;
        }

        public double LineNemberWidth
        {
            get
            {
                double width = this.GetWidth("0");
                return width * Int32.Parse(Resources.LineNumberLength);
            }
        }

        public int TabWidthChar
        {
            get { return this.tabLength; }
            set
            {
                this.tabLength = value;
                this.format.TabWidth = (float)this.GetWidth("a") * value;
                this.Cache.Clear();
            }
        }

        public event ChangedRenderResourceEventHandler ChangedRenderResource;

        public void Resize(double width, double height)
        {
            this.textureSize = new Size(width, height);
            this.DestructResource();
            this.ConstructResource(width, height);
        }

        public void DrawCachedBitmap(Rectangle rect)
        {
            render.DrawBitmap(this.bitmap, rect, 1.0f, rect);
        }

        public void CacheContent()
        {
            render.Flush();
            this.bitmap.CopyFromRenderTarget(new Point2U(), this.render, new RectU(0, 0, (uint)this.textureSize.Width, (uint)this.textureSize.Height));
            this.hasCache = true;
        }

        public bool IsVaildCache()
        {
            return this.hasCache;
        }

        public void BegineDraw()
        {
            this.render.BeginDraw();
        }

        public void EndDraw()
        {
            if (this.render.EndDraw())
            {
                this.DestructResource();
                this.ConstructResource(this.textureSize.Width, this.textureSize.Height);
            }
            else
            {
                this.device.Flush();
            }
        }

        public void DrawLineNumber(int lineNumber, double x, double y)
        {
            string lineNumberFormat = "{0," + Resources.LineNumberLength + "}";
            DTextLayout layout = this.factory.CreateTextLayout(this.format, string.Format(lineNumberFormat, lineNumber), Int32.MaxValue, Int32.MaxValue);
            this.render.DrawTextLayout(layout, (float)x, (float)y);
        }

        public void DrawLine(Point from, Point to)
        {
            this.render.AntialiasMode = DAntialias.Alias;
            this.render.DrawLine(from,
                to,
                1.0f,
                this.markerFactory.Brushes[HilightType.Sold],
                this.markerFactory.Strokes[HilightType.Sold]);
            this.render.AntialiasMode = DAntialias.Antialias;
        }

        public void DrawCaret(Rectangle rect)
        {
            this.render.AntialiasMode = DAntialias.Alias;
            this.render.FillRectangle(rect, this.markerFactory.Brushes[HilightType.Sold]);
            this.render.AntialiasMode = DAntialias.Antialias;
        }

        public void FillBackground(Rectangle rect)
        {
            this.render.FillRectangle(rect, this.background);
        }

        public void DrawOneLine(LineToIndexTable lti, int row, double x, double y, IEnumerable<Selection> SelectRanges, IEnumerable<Marker> MarkerRanges)
        {
            string str = lti[row];

            if (str == string.Empty || str == null)
                return;

            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            if ((bool)layout.Tag == false)
            {
                LineToIndexTableData lineData = lti.GetData(row);
                if (lineData.syntax != null)
                {
                    foreach (SyntaxInfo s in lineData.syntax)
                    {
                        layout.SetTextEffect(s.index, s.length, this.SyntaxResources[s.type]);
                    }
                }

                foreach (Marker m in MarkerRanges)
                {
                    if (m.start == -1 || m.length == 0)
                        continue;
                    if(m.hilight != HilightType.None || m.hilight != HilightType.Select)
                        layout.SetTextEffect(m.start, m.length, this.markerFactory.Brushes[m.hilight]);
                }
                layout.Tag = true;
            }

            this.render.PushAxisAlignedClip(this.ClipRect, DAntialias.Alias);

            IMarkerEffecter effecter;
            foreach (Marker m in MarkerRanges)
            {
                if (m.start == -1 || m.length == 0)
                    continue;
                effecter = this.markerFactory.Create(m.hilight);
                effecter.Apply(this.render, layout, m.start, m.length, x, y);
            }

            effecter = this.markerFactory.Create(HilightType.Select);
            foreach (Selection sel in SelectRanges)
            {
                if (sel.length == 0 || sel.start == -1)
                    continue;

                effecter.Apply(this.render, layout, sel.start, sel.length, x, y);
            }

            using (Unlocker locker = this.store.LockDocument(false))
            {
                foreach (TextDisplayAttribute attr in this.store.EnumAttributes())
                {
                    if (attr.startIndex == attr.endIndex)
                        continue;
                    int length = attr.endIndex - attr.startIndex;
                    int start = attr.startIndex - lti.GetIndexFromLineNumber(row);
                    if (start < 0 || start > lti.GetLengthFromLineNumber(row))
                        continue;
                    if (attr.attribute.fBoldLine)
                        effecter = this.markerFactory.Create(HilightType.BoldSold);
                    else
                        effecter = this.markerFactory.Create(HilightType.Sold);
                    effecter.Apply(this.render, layout, start, length, x, y);
                }
            }

            render.DrawTextLayout(layout, (float)x, (float)y);

            this.render.PopAxisAlignedClip();
        }

        public int GetIndexFromX(string str, double x)
        {
            if (str == string.Empty || str == null || str[0] == Document.EndOfFile)
                return 0;

            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            bool isTrailing, isInsed;
            DHitTestMetrics metrics;
            metrics = layout.HitTextPoint((float)x, 0, out isTrailing, out isInsed);
            if (isTrailing)
                return Util.RoundUp(metrics.textPosition + metrics.length);
            else
                return Util.RoundUp(metrics.textPosition);
        }

        public double GetWidthFromIndex(string str, int index)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);
            float x2;
            layout.HitTestTextPostion(index, true, out x2, out y);

            return x2 - x;
        }

        public double GetWidth(string str)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            return Util.RoundUp(layout.metrics.widthIncludingTrailingWhitespace);
        }

        public double GetXFromIndex(string str, int index)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);
            return x;
        }

        public double GetHeight(string str)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            double height = 0;
            DLineMetrics[] metrics = layout.LineMetrics;
            if (metrics != null && metrics.Length > 0)
                height = metrics[0].height;
            return height;
        }

        public int AlignIndexToNearestCluster(string str, int index, AlignDirection flow)
        {
            DTextLayout layout;
            this.CreateTextLayout(this.format, str, this.ClipRect.Size, out layout);
            float x, y;
            DHitTestMetrics metrics;
            metrics = layout.HitTestTextPostion(index, false, out x, out y);

            if (flow == AlignDirection.Forward)
                return Util.RoundUp(metrics.textPosition + metrics.length);
            else if (flow == AlignDirection.Back)
                return Util.RoundUp(metrics.textPosition);
            throw new ArgumentOutOfRangeException();
        }

        public List<LineToIndexTableData> BreakLine(Document doc, int startIndex, int endIndex, double wrapwidth)
        {
            List<LineToIndexTableData> output = new List<LineToIndexTableData>();

            this.format.WordWrapping = DWordWrapping.Wrapping;

            foreach (string str in Util.GetLines(doc, startIndex, endIndex))
            {
                DTextLayout layout = this.factory.CreateTextLayout(this.format, str, (float)wrapwidth, Int32.MaxValue);

                int i = startIndex;
                foreach (DLineMetrics metrics in layout.LineMetrics)
                {
                    if (metrics.length == 0 && metrics.newlineLength == 0)
                        continue;

                    bool lineend = false;
                    if (metrics.newlineLength > 0 || doc[i + (int)metrics.length - 1] == Document.EndOfFile)
                        lineend = true;

                    output.Add(new LineToIndexTableData(i, (int)metrics.length, lineend, null));
                    i += Util.RoundUp(metrics.length);
                }

                layout.Dispose();

                startIndex += str.Length;
            }

            this.format.WordWrapping = DWordWrapping.NoWrapping;

            return output;
        }

        public void Dispose()
        {
            this.DestructResource();
        }

        void ConstructResource(double width, double height)
        {
            this.factory = new DFactory();
            this.device = new D3D10Device();
            this.texture = this.device.CreateTexture2D((int)width, (int)height);
            this.surface = new DXGISurface(this.texture);
            this.render = this.factory.CreateDxgiSurfaceRenderTarget(this.surface);
            this.device9 = new D3D9Device();
            this.surface9 = this.device9.CreateTexture(this.texture);
            this.bitmap = render.CreateBitmap(new SizeU((uint)width, (uint)height));
            this.hasCache = false;

            this.format = this.factory.CreateTextFormat(this.FontFamily.Source, (float)this.FontSize);
            this.format.WordWrapping = DWordWrapping.NoWrapping;
            this.markerFactory = new MarkerFactory(this.factory);
            this.Foreground = this.ForegroundColor;
            this.Background = this.BackgroundColor;
            this.ControlChar = this.ControlCharColor;
            this.Url = this.UrlColor;
            this.Keyword1 = this.Keyword1Color;
            this.Keyword2 = this.Keyword2Color;
            this.Literal = this.LiteralColor;
            this.Comment = this.CommentColor;
            this.Hilight = this.HilightColor;
            this.render.InitTextRender(this.markerFactory.Brushes[HilightType.Sold], this.SyntaxResources[TokenType.Control]);
        }

        void DestructResource()
        {
            this.hasCache = false;
            this.bitmap.Dispose();
            this.Cache.Clear();
            this.markerFactory.Dispose();
            this.SyntaxResources.Clear();
            this.format.Dispose();
            this.render.Dispose();
            this.surface9.Dispose();
            this.device9.Dispose();
            this.surface.Dispose();
            this.texture.Dispose();
            this.device.Dispose();
            this.factory.Dispose();
        }

        bool CreateTextLayout(DTextFormat format, string str, Size size, out DTextLayout layout)
        {
            bool has = this.Cache.TryGetValue(str, out layout);
            if (has == false)
            {
                layout = this.factory.CreateTextLayout(format, str, (float)size.Width, (float)size.Height);
                layout.Tag = false;
                this.Cache.Add(str, layout);
            }
            return has;
        }

        Color ToColor(Color2F color)
        {
            Color retval = new Color();
            retval.A = (byte)(color.A * 255.0f);
            retval.B = (byte)(color.B * 255.0f);
            retval.G = (byte)(color.G * 255.0f);
            retval.R = (byte)(color.R * 255.0f);
            return retval;
        }
    }
}
