﻿using System;
using System.Collections.Generic;
using SharpDX;
using DW = SharpDX.DirectWrite;

namespace FooEditEngine
{
    class InlineManager
    {
        DW.Factory Factory;
        DW.TextFormat _Format;
        MultiSet<char, InlineChar> InlineChars = new MultiSet<char, InlineChar>();
        MultiSet<int, InlineTab> InlineTabs = new MultiSet<int, InlineTab>();
        int _TabLength;
        const int DuplicateCount = 2;   //1だとDirectWriteが一つにまとめてしまう

        public InlineManager(DW.Factory factory, DW.TextFormat format)
        {
            this.Factory = factory;
            this._Format = format;
        }

        public bool ContainsSymbol(char c)
        {
            if (c == '\t' && this.InlineTabs.Count > 0)
                return true;
            return this.InlineChars.ContainsKey(c);
        }

        public void AddSymbol(char c, char alt)
        {
            if (c == '\t')
            {
                this.InlineTabs.Clear();
                for (int i = 0; i < DuplicateCount; i++)
                {
                    for (int j = 1; j <= this.TabLength; j++)
                        this.InlineTabs.Add(j, new InlineTab(this.Factory, this.Format, alt, j));
                }
            }
            else
            {
                for (int i = 0; i < DuplicateCount; i++)
                {
                    this.InlineChars.Add(c, new InlineChar(this.Factory, this.Format, alt));
                }
            }
        }

        public void RemoveSymbol(char c)
        {
            if (c == '\t')
                this.InlineTabs.Remove(c);
            else
                this.InlineChars.Remove(c);
        }

        public DW.TextFormat Format
        {
            get
            {
                return this._Format;
            }
            set
            {
                this._Format = value;

                this.TabLength = this._TabLength;

                List<KeyValuePair<char, char>> list = new List<KeyValuePair<char, char>>(this.InlineChars.Count);
                foreach (KeyValuePair<char, InlineChar> kv in this.InlineChars.EnumrateKeyAndFirstValue())
                    list.Add(new KeyValuePair<char, char>(kv.Key, kv.Value.AlternativeChar));

                this.InlineChars.Clear();

                foreach (KeyValuePair<char, char> kv in list)
                    for(int i = 0; i < DuplicateCount; i++)
                        this.InlineChars.Add(kv.Key, new InlineChar(this.Factory, this.Format, kv.Value));
            }
        }

        public int TabLength
        {
            get
            {
                return this._TabLength;
            }
            set
            {
                this._TabLength = value;
                
                char c;
                if (this.InlineTabs.Count > 0 && this._TabLength > 0)
                    c = this.InlineTabs.Get(1, 0).AlternativeChar;
                else
                    return;

                this.InlineTabs.Clear();
                
                for (int i = 0; i < DuplicateCount; i++)
                {
                    for (int j = 1; j <= this.TabLength; j++)
                        this.InlineTabs.Add(j, new InlineTab(this.Factory, this.Format, c, j));
                }
            }
        }

        public DW.InlineObject Get(int index, string str)
        {
            if (str[index] == '\t')
            {
                int tabcount;
                if (index > 0 && str[index - 1] == '\t')
                    tabcount = this.TabLength;
                else
                    tabcount = this.TabLength - index % this.TabLength;
                if (this.InlineTabs.ContainsKey(tabcount))
                    return this.InlineTabs.Get(tabcount, index % DuplicateCount);
                else
                    return null;
            }
            else
            {
                char c = str[index];
                if (this.InlineChars.ContainsKey(c) == false)
                    return null;
                return this.InlineChars.Get(c, index % DuplicateCount);
            }
        }

        public void Clear()
        {
            this.InlineChars.Clear();
            this.InlineTabs.Clear();
        }
    }
    class MultiSet<T, J>
        where J : IDisposable
    {
        Dictionary<T, List<J>> Collection = new Dictionary<T, List<J>>();

        public void Add(T key, J value)
        {
            if (this.Collection.ContainsKey(key) == false)
                this.Collection.Add(key, new List<J>());
            this.Collection[key].Add(value);
        }

        public int Count
        {
            get
            {
                return this.Collection.Count;
            }
        }

        public bool ContainsKey(T key)
        {
            return this.Collection.ContainsKey(key);
        }

        public J Get(T key,int index)
        {
            return this.Collection[key][index];
        }

        public void Remove(T key)
        {
            if (this.Collection.ContainsKey(key) == false)
                return;
            foreach (J value in this.Collection[key])
                value.Dispose();
            this.Collection.Remove(key);
        }

        public void Clear()
        {
            foreach (List<J> list in this.Collection.Values)
                foreach (J value in list)
                    value.Dispose();
            this.Collection.Clear();
        }

        public IEnumerable<KeyValuePair<T, J>> EnumrateKeyAndFirstValue()
        {
            foreach (KeyValuePair<T, List<J>> kv in this.Collection)
                yield return new KeyValuePair<T, J>(kv.Key, kv.Value[0]);
        }
    }
}
