package jp.sourceforge.freegantt.persistence;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import jp.sourceforge.freegantt.data.serializer.FileFormatException;
import jp.sourceforge.freegantt.util.XmlUtil;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

public abstract class Persistence extends ConcurrentHashMap<String, Object> {
	private static final long serialVersionUID = 7514665998880664425L;
	
	public static final String TYPE_ATTR = "type";
	public static final String INT_ATTR = "int";
	public static final String LONG_ATTR = "long";
	public static final String DOUBLE_ATTR = "double";
	
	abstract public String getFilename();
	
	public synchronized static Persistence getInstance() {
		return null;
	}
	
	public void putValue(String key, Object value) {
		put(key, value);
	}
	
	public Object getValue(String key, Object defaultValue) {
		Object value = get(key);
		if (value != null) return value;
		
		System.out.println("Preference not found: " + key);
		put(key, defaultValue);
		return defaultValue;
	}
	
	public Integer getInt(String key, Integer defaultValue) {
		Object value = getValue(key, defaultValue);
		if (value instanceof Integer) return (Integer)value;
		
		System.out.println("Preference class cast failed: " + key + ", " + value);
		return defaultValue;
	}

	public void save() {
		FileOutputStream out = null;
		try {
			out = new FileOutputStream(getFilename());
			save(out);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} finally {
			try {
				if (out != null) out.close();
			} catch (IOException e) {
			}
		}
	}
	
	public void save(OutputStream out) {
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			DOMImplementation dom = builder.getDOMImplementation();
			Document doc = dom.createDocument("", "freeganttProject", null);
			Element root = doc.getDocumentElement();
			
			for (String key: keySet()) {
				Object value = get(key);
				Element ele = XmlUtil.createTextElement(doc, key, String.valueOf(value));
				if (value instanceof Integer) ele.setAttribute(TYPE_ATTR, INT_ATTR);
				if (value instanceof Long) ele.setAttribute(TYPE_ATTR, LONG_ATTR);
				if (value instanceof Double) ele.setAttribute(TYPE_ATTR, DOUBLE_ATTR);
				root.appendChild(ele);
			}
			
			DOMSource source = new DOMSource(doc);
			StreamResult result = new StreamResult(out);
			
			TransformerFactory transFactory = TransformerFactory.newInstance();
			Transformer transformer = transFactory.newTransformer();
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty(OutputKeys.METHOD, "xml");
			transformer.transform(source, result);	
			
		} catch (ParserConfigurationException e) {
		} catch (TransformerConfigurationException e) {
		} catch (TransformerException e) {
		}
	}
	
	public void load() {
		FileInputStream in = null;
		try {
			in = new FileInputStream(getFilename());
			load(in);
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (FileFormatException e) {
			e.printStackTrace();
		} finally {
			try {
				if (in != null) in.close();
			} catch (IOException e) {
			}
		}
	}
	
	public void load(InputStream in) throws SAXException, IOException, FileFormatException {
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = factory.newDocumentBuilder();
			Document doc = builder.parse(in);
			
			Element root = doc.getDocumentElement();
			List<Element> children = XmlUtil.getChildElements(root);
			for (Element child: children) {
				String key = child.getTagName();
				String value = child.getTextContent();
				if (INT_ATTR.equals(child.getAttribute(TYPE_ATTR))) {
					put(key, Integer.parseInt(value));
				} else if (LONG_ATTR.equals(child.getAttribute(TYPE_ATTR))) {
					put(key, Long.parseLong(value));
				} else if (DOUBLE_ATTR.equals(child.getAttribute(TYPE_ATTR))) {
					put(key, Double.parseDouble(value));
				} else {
					put(key, value);
				}
			}
			
		} catch (ParserConfigurationException e) {
		} catch (NumberFormatException e) {
			throw new FileFormatException(e);
		}
	}
	
}

