package jp.sourceforge.freegantt.data.model;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;
import javax.swing.undo.CompoundEdit;

import jp.sourceforge.freegantt.data.Project;
import jp.sourceforge.freegantt.locale.Resource;
import jp.sourceforge.freegantt.util.CalendarUtil;

public class HolidayTableModel implements TableModel {

	Project project;
	String[] columnNames;
	List<TableModelListener> tableModelListeners = new ArrayList<TableModelListener>();

	public HolidayTableModel(Project project) {
		this.project = project;
		columnNames = new String[]{"", Resource.get("holidayDate")};
	}
	
	@Override
	public int getRowCount() {
		return project.getAdditionalHolidays().size() + 1;
	}

	@Override
	public int getColumnCount() {
		return columnNames.length;
	}

	@Override
	public String getColumnName(int columnIndex) {
		if (columnIndex >= columnNames.length) return "";
		return columnNames[columnIndex];
	}

	@Override
	public Class<?> getColumnClass(int columnIndex) {
		return String.class;
	}

	@Override
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		return columnIndex > 0;
	}

	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		if (columnIndex == 0) return rowIndex + 1;
		if (rowIndex >= project.getAdditionalHolidays().size()) return null;
		Calendar calendar = project.getAdditionalHolidays().get(rowIndex);
		SimpleDateFormat format = new SimpleDateFormat(Resource.get("dateFormat"), Resource.getLocale());
		return format.format(calendar.getTime());
	}

	@Override
	public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
		if (rowIndex > project.getAdditionalHolidays().size()) return;
		if (columnIndex != 1) return;
		if (aValue == null || ((String)aValue).isEmpty()) return;
		
		Calendar newCalendar = CalendarUtil.parseDate((String)aValue, Resource.getLocale());
		if (newCalendar == null) return;
		
		CompoundEdit compound = new CompoundEdit();
		project.getUndoManager().addEdit(compound);
		try {
			if (rowIndex == project.getAdditionalHolidays().size()) {
				project.getController().addAdditionalHoliday(newCalendar, project.getAdditionalHolidays().size());
			} else {
				project.getController().setAdditionalHoliday(newCalendar, rowIndex);
			}
			project.update();
		} finally {
			compound.end();
		}
	}

	@Override
	public void addTableModelListener(TableModelListener l) {
		tableModelListeners.add(l);
	}

	@Override
	public void removeTableModelListener(TableModelListener l) {
		tableModelListeners.remove(l);
	}
	
	public void fireTableChanged() {
		for (TableModelListener listener: tableModelListeners) {
			TableModelEvent e = new TableModelEvent(this, 0, 0, 0, TableModelEvent.UPDATE);
			listener.tableChanged(e);
		}
	}

}
