/*
 * FreeModbus Libary: ATMega168 Port
 * Copyright (C) 2006 Christian Walter <wolti@sil.at>
 *   - Initial version and ATmega168 support
 * Modfications Copyright (C) 2006 Tran Minh Hoang:
 *   - ATmega8, ATmega16, ATmega32 support
 *   - RS485 support for DS75176
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * File: $Id: portserial.c,v 1.6 2006/09/17 16:45:53 wolti Exp $
 */

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/signal.h>

#include <port.h>
#include "contiki-conf.h"
#include <contiki.h>
#include "mbtask.h"

#include <usart-defs.h>

/* ----------------------- Modbus includes ----------------------------------*/
#include <mb.h>
#include <mbport.h>
#include <mbstruct.h>

#define UART_BAUD_CALC(UART_BAUD_RATE,F_OSC) \
    ( ( F_OSC ) / ( ( UART_BAUD_RATE ) * 16UL ) - 1 )

//#define UART_UCSRB  UCSR0B

void
vMBPortSerialEnable( BOOL xRxEnable, BOOL xTxEnable )
{
#ifdef RTS_ENABLE
    MBUCSRB |= _BV( MBTXEN ) | _BV(MBTXCIE);
#else
    MBUCSRB |= _BV( MBTXEN );
#endif

    if( xRxEnable )
    {
        MBUCSRB |= _BV( MBRXEN ) | _BV( MBRXCIE );
    }
    else
    {
        MBUCSRB &= ~( _BV( MBRXEN ) | _BV( MBRXCIE ) );
    }

    if( xTxEnable )
    {
        MBUCSRB |= _BV( MBTXEN ) | _BV( MBUDRE );
#ifdef RTS_ENABLE
        RTS_HIGH;
#endif
    }
    else
    {
        MBUCSRB &= ~( _BV( MBUDRE ) );
    }
}

BOOL
xMBPortSerialInit( UCHAR ucPORT, ULONG ulBaudRate, UCHAR ucDataBits, eMBParity eParity )
{
    UCHAR ucUCSRC = 0;

    /* prevent compiler warning. */
    (void)ucPORT;

    switch ( eParity )
    {
        case MB_PAR_EVEN:
            ucUCSRC |= USART_PAR_EVEN;
            break;
        case MB_PAR_ODD:
            ucUCSRC |= USART_PAR_ODD;
            break;
        case MB_PAR_NONE:
            break;
    }

    switch ( ucDataBits )
    {
        case 8:
            ucUCSRC |= _BV( MBUCSZ0 ) | _BV( MBUCSZ1 );
            break;
        case 7:
            ucUCSRC |= _BV( MBUCSZ1 );
            break;
    }

#if defined (__AVR_ATmega168__)
    MBUCSRC = ucUCSRC;
#elif defined (__AVR_ATmega169__)
    MBUCSRC = ucUCSRC;
#elif defined (__AVR_ATmega162__)
    MBUCSRC = _BV( MBURSEL ) | ucUCSRC;
#elif defined (__AVR_ATmega8__)
    MBUCSRC = _BV( MBURSEL ) | ucUCSRC;
#elif defined (__AVR_ATmega16__)
    MBUCSRC = _BV( MBURSEL ) | ucUCSRC;
#elif defined (__AVR_ATmega32__)
    MBUCSRC = _BV( MBURSEL ) | ucUCSRC;
#elif defined (__AVR_ATmega128__)
    MBUCSRC = ucUCSRC;
#endif

    uint16_t tmp = UART_BAUD_CALC( ulBaudRate, F_CPU );
    MBUBRRH = (tmp >> 8);
    MBUBRRL = (uint8_t)tmp;


    vMBPortSerialEnable( FALSE, FALSE );

#ifdef RTS_ENABLE
    RTS_INIT;
#endif
    return TRUE;
}

BOOL
xMBPortSerialPutByte( CHAR ucByte )
{
    MBUDR = ucByte;
    DEBUG_MBtx(ucByte);
    return TRUE;
}

BOOL
xMBPortSerialGetByte( CHAR * pucByte )
{
    *pucByte = MBUDR;
    DEBUG_MBrx(*pucByte);
    return TRUE;
}

ISR( MBSIG_USART_DATA )
{
    if (pxMBFrameCBTransmitterEmpty()) {
        process_poll(&modbus_poller);
    }
}

ISR( MBSIG_USART_RECV )
{
    if (pxMBFrameCBByteReceived()) { 
        process_poll(&modbus_poller);
    }
}

#ifdef RTS_ENABLE
ISR( MBSIG_UART_TRANS )
{
    RTS_LOW;
}
#endif

