using System;
using System.IO;
using System.Drawing;
using nft.framework.drawing;
using System.Diagnostics;

namespace nft.drawing.ddraw7
{
	/// <summary>
	/// SimpleTexture ̊Tv̐łB
	/// </summary>
	public class DD7StaticTexture : IStaticTexture
	{
		protected DD7Surface surface;
        protected bool surfaceOwner;
        protected IPixelFilter filter;
		protected Rectangle srcRegion;
		protected Point drawOffset;
        protected ImageRef imgRef;

        public DD7StaticTexture(DD7Surface source, Rectangle region, Point offset, bool surfaceOwner) {
            this.surface = source as DD7Surface;
            this.srcRegion = new Rectangle(0, 0, source.Size.Width, source.Size.Height);
            this.srcRegion.Intersect(region);
            this.surfaceOwner = surfaceOwner;
            this.drawOffset = offset;
		}

        public DD7StaticTexture(ImageRef image, Rectangle region, Point offset) {
            this.surface = null; // create on demand
            this.imgRef = image;
            this.srcRegion = region;
            this.surfaceOwner = true;            
            this.drawOffset = offset;
        }

		#region ITexture o

		public Rectangle Boundary { get { return new Rectangle(drawOffset,srcRegion.Size); } }

		public bool HitTest(Point pos) {
			if(Boundary.Contains(pos)) {
				int x = pos.X-drawOffset.X;
				int y = pos.Y-drawOffset.Y;
				return Source.HitTest(x, y);
			}
			else
				return false;
		}

		#endregion
        
        protected DrawParams Adjust(DrawParams orig){
            DrawParams dp = orig.Clone();
            dp.Location.X += drawOffset.X;
            dp.Location.Y += drawOffset.Y;
            if (dp.SoruceClip.Equals(Rectangle.Empty)) {
                dp.SoruceClip = srcRegion;
            } else {
                dp.SoruceClip.Intersect(srcRegion); 
            }
            return dp;
        }

		#region IDrawable o
        public void Draw(DrawParams drawparam) {
            DrawParams dp = Adjust(drawparam);
            if (filter != null)
                dp.Destination.BitBlt(Source, dp, filter);
            else
                dp.Destination.BitBlt(Source, dp);
        }

        public void DrawEx(DrawParams drawparam, IPixelFilter f) {
            DrawParams dp = Adjust(drawparam);
            if (filter != null) {
                CompositFilter cf = new CompositFilter(filter, f);
                dp.Destination.BitBlt(Source, dp, cf);
            } else
                dp.Destination.BitBlt(Source, dp, f);            
        }

        public void DrawEx(DrawParams drawparam, ISurfaceDrawer drawer) {
            DrawParams dp = Adjust(drawparam);
            dp.Destination.BitBlt(Source, dp, drawer);
        }
        /*
        [Obsolete]
        public void Draw(ISurface dest, System.Drawing.Point location, Scaler scale, int frame) {
			Point pt = new Point(location.X+drawOffset.X,location.Y+drawOffset.Y);
			if(filter!=null)
                dest.BitBlt(pt, Source, srcRegion, scale, filter);
			else
                dest.BitBlt(pt, Source, srcRegion, scale);
		}
        [Obsolete]
        public void DrawEx(ISurface dest, System.Drawing.Point location, Scaler scale, IPixelFilter f, int frame) {
			Point pt = new Point(location.X+drawOffset.X,location.Y+drawOffset.Y);
			if(filter!=null) {
				CompositFilter cf = new CompositFilter(filter,f);
                dest.BitBlt(pt, Source, srcRegion, scale, cf);
			}
			else
                dest.BitBlt(pt, Source, srcRegion, scale, f);
		}
        [Obsolete]
        public void DrawEx(ISurface dest, System.Drawing.Point location, Scaler scale, ISurfaceDrawer drawer, int frame) {
			Point pt = new Point(location.X+drawOffset.X,location.Y+drawOffset.Y);
            dest.BitBlt(pt, Source, srcRegion, scale, drawer);
		}
        */
		public IPixelFilter Filter {
			get { return filter; }
			set { filter = value; }
		}
		#endregion

        #region IDisposable o

		public void Dispose() {
            Dispose(true);
            GC.SuppressFinalize(this);
		}

        protected void Dispose(bool disposing) {
            if (surface != null) {
                if (disposing) {
                    if(surfaceOwner)
                        surface.Dispose();
                    surface = null;
                }
            }
        }

        ~DD7StaticTexture() {
            Dispose(false);
        }
		#endregion

		#region ISimpleTexture o
		// Becare!! If the source surface is refered from other objects,
		// ColorKey setting will affects all those objects.
		public Color ColorKey { 
			get { return Source.ColorKey; } 
			set { Source.ColorKey = value; }
		}

		public void PickColorKeyFromSource( int x, int y ){
			x += srcRegion.X;
			y += srcRegion.Y;
			if(srcRegion.Contains(x,y)){
                PixelColorMode mode = Source.PixelColorMode;
				ColorKey = PixelFormatUtil.ToColor(surface.GetPixel(x,y), mode);
			}
			else
				ColorKey = Color.Empty;
		}

		public Point DrawOffset { get { return drawOffset; } }

        public Bitmap CreateBitamap() {
            throw new NotImplementedException();
        }
        #endregion

        //public Rectangle SourceRegion { get { return srcRegion; } }

        internal ISurface Source {
            get {
                if (surface == null || surface.IsSurfaceLost) {
                    if (imgRef != null) {
                        IGraphicManager gm = DD7GraphicManager.TheInstance;
                        surface = (DD7Surface)gm.CreateSurfaceFromBitmap(imgRef.AddRef(),srcRegion);
                        imgRef.ReleaseRef();
                    } else {
                        throw new IOException("DirectDraw surface is lost.");
                    }
                }
                return surface;
            }
        }
    }
}
