/*
 * 
 * XNua 0.5.0
 * CodeGen.cs - Code generator for the interface layer
 * Copyright 2003-2005 Fabio Mascarenhas
 *                2007 Dean Calver
 * 
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

namespace XNua
{
    public class LuaStack
    {
        public static readonly LuaValue[] NilStack = new LuaValue[1] { new LuaValue(NilClass.Instance) };

        public int Base;
        public int Top;
        public LuaValue[] Values;

        public LuaStack(int beginStack)
        {
            Values = new LuaValue[beginStack];
            this.Base = 0;
            this.Top = 0;
        }

        public void Check(int grow)
        {
            if (Values.Length < (Top + grow))
            {
                int newLength = Values.Length * 2;
                while (newLength < (Top + grow))
                    newLength *= 2;
                LuaValue[] newStack = new LuaValue[newLength];
                Values.CopyTo(newStack, 0);
                Values = newStack;
            }
        }
        public LuaValue BaseValue()
        {
            return Values[Base];
        }
        /// <summary>
        /// access the value in the stack
        /// </summary>
        /// <param name="index">address to access in the stack</param>
        /// <returns></returns>
        public LuaValue this[int index]
        {
            get
            {
                return Values[index];
            }
            set
            {
                Values[index] = value;
            }
        }

        // this provide an API similar to the C api, good for porting stuff over
        // as such it also follows the naming convention rather than the C# one

        #region Lua C API A like
        internal LuaValue? negindex(int idx)
        {
            if (idx > REGISTRYINDEX)
            {
                return Values[ Top + idx];
            }
            else
            {
                switch (idx)
                {
                    case REGISTRYINDEX:
                        throw new Exception();
                    case GLOBALSINDEX:
                        throw new Exception();
                    default:
                        {
                            Debug.Assert(Values[Base - 1].IsReference && Values[Base - 1].O.Tag == LuaTypes.LUA_TFUNCTION);
                            if (Values[Base - 1].O is LuaClosure)
                            {
                                LuaClosure clos = (LuaClosure)Values[Base - 1].O;
                                idx = GLOBALSINDEX - idx;
                                if (idx >= clos.UpVals.Length)
                                    return null;
                                return clos.UpVals[idx].Value;
                            }
                            else
                            {
                                return null;
                            }
                        }
                }
            }
        }
        internal int A_index(int idx)
        {
            if (idx > 0)
            {
                return Base + idx - 1;
            }
            else
            {
                return Top + idx;
            }
        }
        internal LuaValue? A_indexAcceptable(int idx)
        {
            if (idx > 0)
            {
                int index = Base + (idx - 1);
                if (index >= Top)
                    return null;
                else
                    return Values[index];
            }
            else
            {
                return negindex(idx);
            }
        }
        public void pushobject( LuaValue o )
        {
            Values[Top++] = 0;
        }

        public int gettop()
        {
            return Top - Base;
        }
        public void settop(int idx)
        {
            if (idx >= 0)
            {
                while (Top < Base + idx)
                {
                    Values[Top++] = NilClass.Instance;
                }
                Top = Base + idx;
            }
            else
            {
                Top += idx + 1; // 'subtract' index
            }
        }

        public void remove(int idx)
        {
            int p = A_index(idx);
            while (++p < Top)
            {
                Values[p - 1] = Values[p];
            }
            Top--;
        }

        public void insert(int idx)
        {
            int p = A_index(idx);
            for (int q = Top; q>p; q--)
            {
                Values[q] = Values[q-1];
            }
            Values[p] = Values[Top];
        }
        public void replace(int idx)
        {
            int p = A_index(idx);
            Values[p] = Values[Top - 1];
            Top--;
        }
        public void pushvalue(int idx)
        {
            Check(1);
            Values[Top] = Values[A_index(idx)];
            Top++;
        }
        public LuaTypes type(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            if (o.HasValue == false)
            {
                return LuaTypes.LUA_TNONE;
            }
            else
            {
                return o.Value.type;
            }
        }
        public string typename(int idx)
        {
            LuaTypes t = type(idx);
            switch (t)
            {
                case LuaTypes.LUA_TNONE:
                    return "no value";
                case LuaTypes.LUA_TNIL:
                    return "nil";
                case LuaTypes.LUA_TBOOLEAN:
                    return "boolean";
                case LuaTypes.LUA_TLIGHTUSERDATA:
                    return "userdata";
                case LuaTypes.LUA_TNUMBER:
                    return "number";
                case LuaTypes.LUA_TSTRING:
                    return "string";
                case LuaTypes.LUA_TTABLE:
                    return "table";
                case LuaTypes.LUA_TFUNCTION:
                    return "function";
                case LuaTypes.LUA_TUSERDATA:
                    return "userdata";
                default:
                    return "no value";
            }
        }
        public bool isnumber(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            if (o.HasValue == false)
                return false;
            else if (!o.Value.IsReference)
                return true;
            else return (o.Value.O.Tag == LuaTypes.LUA_TNUMBER);
        }
        public bool isstring(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            return (!o.HasValue) ? false : (o.Value.O.Tag == LuaTypes.LUA_TSTRING);
        }
        public bool isuserdata(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            return (!o.HasValue) ? false : ((o.Value.O.Tag == LuaTypes.LUA_TUSERDATA) | (o.Value.O.Tag == LuaTypes.LUA_TLIGHTUSERDATA));
        }
        public bool rawequal(int i0, int i1)
        {
            LuaValue? o0 = A_indexAcceptable(i0);
            LuaValue? o1 = A_indexAcceptable(i1);
            return( (!o0.HasValue || !o1.HasValue) ? false : o0.Value.Equals(o1.Value));
        }
        public bool equal(int i0, int i1)
        {
            // TODO Meta table stuff...
            LuaValue? o0 = A_indexAcceptable(i0);
            LuaValue? o1 = A_indexAcceptable(i1);
            return ((!o0.HasValue || !o1.HasValue) ? false : o0.Value.Equals(o1.Value));
        }
        public bool lessthan(int i0, int i1)
        {
            LuaValue? o0 = A_indexAcceptable(i0);
            LuaValue? o1 = A_indexAcceptable(i1);
            return ((!o0.HasValue || !o1.HasValue) ? false : o0.Value.LessThan(o1.Value));
        }
        public double tonumber(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            return (!o.HasValue)? 0.0f : o.Value.ToNumber();
        }
        public bool toboolean(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            return (!o.HasValue) ? false : o.Value.ToBoolean();
        }
        public string tostring(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            return (!o.HasValue) ? String.Empty : o.Value.ToStr();
        }

        public int strlen(int idx)
        {
            return tostring(idx).Length;
        }

        public void pushnil()
        {
            Values[Top++] = NilClass.Instance;
        }
        public void pushnumber(double n)
        {
            Values[Top++] = n;
        }
        public void pushstring(string str)
        {
            if (str == null)
            {
                pushnil();
            }
            else
            {
                Values[Top++] = str;
            }
        }
        public void pushboolean(bool b)
        {
            Values[Top++] = b ? (LuaReference)TrueClass.Instance : (LuaReference)FalseClass.Instance;
        }
        public void gettable(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            if (!o.HasValue|| !o.Value.IsReference || (o.Value.O.Tag != LuaTypes.LUA_TTABLE))
                Values[Top - 1] = NilClass.Instance;
            else
                Values[Top - 1] = o.Value.O;
        }
        public void rawget(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            if (!o.HasValue || !o.Value.IsReference || (o.Value.O.Tag != LuaTypes.LUA_TTABLE))
                Values[Top - 1] = NilClass.Instance;
            else
            {
                LuaTable table = (LuaTable)o.Value.O;
                Values[Top - 1] = table[Values[Top - 1]];
            }
        }
        public void rawgeti(int idx, int n)
        {
            LuaValue? o = A_indexAcceptable(idx);
            if (!o.HasValue || !o.Value.IsReference || (o.Value.O.Tag != LuaTypes.LUA_TTABLE))
                Values[Top++] = NilClass.Instance;
            else
            {
                LuaTable table = (LuaTable)o.Value.O;
                Values[Top++] = table[n];
            }
        }
        public void newtable()
        {
            Values[Top++] = new LuaTable();
        }
        public void settable(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            if (!o.HasValue || !o.Value.IsReference || (o.Value.O.Tag != LuaTypes.LUA_TTABLE))
            {
                // do nothing except put the stack back
                Top -= 2;
            }
            else
            {
                LuaTable table = (LuaTable)o.Value.O;
                table[Values[Top - 2]] = Values[Top - 1];
                Top -= 2;
            }
        }
        public void rawset(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            if (!o.HasValue || !o.Value.IsReference || (o.Value.O.Tag != LuaTypes.LUA_TTABLE))
            {
                // do nothing except put the stack back
                Top -= 2;
            }
            else
            {
                LuaTable table = (LuaTable)o.Value.O;
                table[Values[Top - 2]] = Values[Top - 1];
                Top -= 2;
            }
        }
        public void rawseti(int idx, int n)
        {
            LuaValue? o = A_indexAcceptable(idx);
            if (!o.HasValue || !o.Value.IsReference || (o.Value.O.Tag != LuaTypes.LUA_TTABLE))
            {
                // do nothing except put the stack back
                Top -= 1;
            }
            else
            {
                LuaTable table = (LuaTable)o.Value.O;
                table.SetNum(n, Values[Top - 1]);
                Top -= 1;
            }
        }
        public double checknumber(int idx)
        {
            double d = tonumber(idx);
            if (d == 0 && !isnumber(idx))
                throw new Exception();
            return d;
        }
        public int checkint(int idx)
        {
            double d = checknumber(idx);
            return (int)d;
        }
        public double optnumber(int idx, double def)
        {
            double d = tonumber(idx);
            if (d == 0 && !isnumber(idx))
                return def;
            return d;
        }
        public int optint(int idx, int def)
        {
            double d = optnumber(idx, def);
            return (int)d;
        }
        internal int aux_checkint()
        {
            int n = (int)tonumber(-1);
            if( n == 0 && !isnumber(-1))
                n = -1;
            Top--;
            return n;
        }
        public int aux_getn(int idx)
        {
            LuaValue? o = A_indexAcceptable(idx);
            if (!o.HasValue || !o.Value.IsReference || (o.Value.O.Tag != LuaTypes.LUA_TTABLE))
            {
                return -1;
            }
            else
            {
                LuaTable table = (LuaTable)o.Value.O;
                pushstring("n");
                rawget(idx);
                int n = aux_checkint();
                if (n >= 0)
                    return n;
                n = table.SizeN;
                if (n >= 0)
                    return n;

                // count integers til nil
                Node[] currentStructure = null;
                int currentIndex = -1;
                while (table.GetNextInt(ref currentStructure, ref currentIndex))
                {
                    // do nothing
                }
                return currentIndex - 1;
            }
        }
        public void aux_setn(int t, int n)
        {
            LuaValue? o = A_indexAcceptable(t);
            if (!o.HasValue || !o.Value.IsReference || (o.Value.O.Tag != LuaTypes.LUA_TTABLE))
            {
                return;
            }
            else
            {
                LuaTable table = (LuaTable)o.Value.O;
                int ni = (int)table["n"].ToNumber();
                if (ni >= 0)
                {
                    pushstring("n");
                    pushnumber(n);
                    rawset(t);
                }
                else
                {
                    table.SizeN = n;
                }
            }
        }
        const int REGISTRYINDEX = -10000;
        const int GLOBALSINDEX  = -10001;
        public int upvalueindex(int idx)
        {
            return GLOBALSINDEX - idx;
        }
        #endregion

    }
}
