/*
 * 
 * XNua 0.1 Based Massively on Lua2IL 0.5
 * Lua2IL 0.5.0
 * LuaTable.cs - Fast implementation of Lua tables
 * Copyright 2003-2005 Fabio Mascarenhas
 *                2007 Dean Calver
 * 
 */

using System;
using System.Collections;
using System.Collections.Generic;

namespace XNua
{
	class Node 
	{
		public static readonly Node DummyNode=new Node();
		public static readonly Node NilNode=new Node();

		public LuaValue key;
		public LuaValue val;
		public Node next;

		public Node() 
		{
			key.O=NilClass.Instance;
			val.O=NilClass.Instance;
			next=null;
		}

		public void Copy(Node c) 
		{
			key=c.key;
			val=c.val;
			next=c.next;
		}
	}

	public class LuaTable : LuaReference, IEnumerable<LuaValue>
	{
		static readonly int MAXBITS=32;

		static readonly byte[] log_8=new byte[]	{
			0,
			1,1,
			2,2,2,2,
			3,3,3,3,3,3,3,3,
			4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
			5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
			6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
			6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
			7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
			7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
			7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
			7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7
		};

        /// <summary>
        /// getN and setN are wierd this is just to make my life easy
        /// and means i don't need a registry
        /// </summary>
        public int SizeN
        {
            get 
            {
                return sizeN;
            }
            set 
            {
                if (value > this.sizeArray)
                    SetArrayVector(value);
                sizeN = value;
            }
        }
        int sizeN = -1;

		int sizeArray;
		Node[] array;
		int[] nums=new int[LuaTable.MAXBITS+1];

		byte logSizeNode;
		Node[] node;
		int firstFree;

		LuaReference meta;

		int ArrayIndex(LuaValue key) 
		{
			if(key.O==null) 
			{
				int k=(int)key.N;
				if((double)k==key.N && k>=1 && k<Int32.MaxValue)
					return k;
			}
			return -1;
		}

		void ComputeSizes(int[] nums, int ntotal, ref int narray, ref int nhash) 
		{
			int i;
			int a = nums[0];  /* number of elements smaller than 2^i */
			int na = a;  /* number of elements to go to array part */
			int n = (na == 0) ? -1 : 0;  /* (log of) optimal size for array part */
			for (i = 1; a < narray && narray >= (1<<(i-1)); i++) 
			{
				if (nums[i] > 0) 
				{
					a += nums[i];
					if (a >= (1<<(i-1))) 
					{  /* more than half elements in use? */
						n = i;
						na = a;
					}
				}
			}
			nhash = ntotal - na;
			narray = (n == -1) ? 0 : (1<<n);
		}

		void NumUse(ref int narray, ref int nhash) 
		{
			for(int k=0;k<nums.Length;k++) nums[k]=0;
			int i, lg;
			int totaluse = 0;
			/* count elements in array part */
			for (i=0, lg=0; lg<=LuaTable.MAXBITS; lg++) 
			{  /* for each slice [2^(lg-1) to 2^lg) */
				int ttlg = (1<<lg);  /* 2^lg */
				if (ttlg > this.sizeArray) 
				{
					ttlg = this.sizeArray;
					if (i >= ttlg) break;
				}
				for (; i<ttlg; i++) 
				{
					if (this.array[i].val.O!=NilClass.Instance) 
					{
						nums[lg]++;
						totaluse++;
					}
				}
			}
			//for (; lg<=LuaTable.MAXBITS; lg++) nums[lg] = 0;  /* reset other counts */
			narray = totaluse;  /* all previous uses were in array part */
			/* count elements in hash part */
			i = (1<<this.logSizeNode);
			while (i-->0) 
			{
				Node n = this.node[i];
				if (n.val.O!=NilClass.Instance) 
				{
					int k = ArrayIndex(n.key);
					if (k >= 0) 
					{  /* is `key' an appropriate array index? */
						nums[FastLog2((uint)k-1)+1]++;  /* count as such */
						narray++;
					}
					totaluse++;
				}
			}
			ComputeSizes(nums, totaluse, ref narray, ref nhash);
		}

		int FastLog2(uint x) 
		{
			if (x >= 0x00010000) 
			{
				if (x >= 0x01000000) return LuaTable.log_8[((x>>24) & 0xff) - 1]+24;
				else return LuaTable.log_8[((x>>16) & 0xff) - 1]+16;
			}
			else 
			{
				if (x >= 0x00000100) return LuaTable.log_8[((x>>8) & 0xff) - 1]+8;
				else if(x!=0) return LuaTable.log_8[(x & 0xff) - 1];
				return -1;  /* special `log' for 0 */
			}
		}

		void SetArrayVector(int size) 
		{
			Node[] newArr=new Node[size];
			for(int i=0;i<this.sizeArray;i++)
				newArr[i]=this.array[i];
			for(int i=this.sizeArray;i<newArr.Length;i++)
				newArr[i]=new Node();
			this.array = newArr;
			this.sizeArray = size;
		}

		void SetNodeVector(int lsize) 
		{
			int i;
			int size = (1<<lsize);
			if (lsize > LuaTable.MAXBITS)
				throw new RuntimeException("table overflow");
			if (lsize == 0) 
			{  /* no elements to hash part? */
				this.node=new Node[1];
				node[0]=Node.DummyNode;
			}
			else 
			{
				this.node = new Node[size];
				for (i=0; i<size; i++) 
				{
					this.node[i] = new Node();
				}
			}
			this.logSizeNode = (byte)lsize;
			this.firstFree = size-1;  /* first free position to be used */
		}

		void Resize(int nasize, int nhsize) 
		{
			int i;
			int oldasize = this.sizeArray;
			int oldhsize = this.logSizeNode;
			Node[] nold;
			Node[] temp=new Node[] { new Node() };
			if(oldhsize!=0)
				nold = this.node;  /* save old hash ... */
			else 
			{  /* old hash is `dummynode' */
				temp[0].key = this.node[0].key; /* copy it to `temp' */
				temp[0].val = this.node[0].val;
				temp[0].next = this.node[0].next;
				nold = temp;
				Node.DummyNode.key.O = NilClass.Instance;  /* restate invariant */
				Node.DummyNode.val.O = NilClass.Instance;
				Node.DummyNode.next = null;
			}
			if (nasize > oldasize)  /* array part must grow? */
				SetArrayVector(nasize);
			/* create new hash part with appropriate size */
			SetNodeVector(nhsize);  
			/* re-insert elements */
			if (nasize < oldasize) 
			{  /* array part must shrink? */
				this.sizeArray = nasize;
				/* re-insert elements from vanishing slice */
				for (i=nasize; i<oldasize; i++) 
				{
					if (this.array[i].val.O!=NilClass.Instance)
						SetNum(i+1,this.array[i].val);
				}
				/* shrink array */
				Node[] newarr=new Node[nasize];
				for(i=0;i<newarr.Length;i++)
					newarr[i]=this.array[i];
				this.array=newarr;
			}
			/* re-insert elements in hash part */
			for (i = (1<<oldhsize) - 1; i >= 0; i--) 
			{
				Node old = nold[i];
				if (old.val.O!=NilClass.Instance)
					Set(old.key,old.val);
			}
		}

		void Rehash() 
		{
			int nasize = 0;
			int nhsize = 0;
			NumUse(ref nasize, ref nhsize);  /* compute new sizes for array and hash parts */
			Resize(nasize, FastLog2((uint)nhsize)+1);
		}

		public LuaTable(int narray, int lnhash) 
		{
			SetArrayVector(narray);
			SetNodeVector(lnhash);
			meta=NilClass.Instance;
			Tag=LuaTypes.LUA_TTABLE;
			Hash=(uint)GetHashCode();
		}

		public LuaTable() : this(0,0) {}

		~LuaTable() 
		{
			LuaReference gcFunction;
			if(meta!=null && meta!=NilClass.Instance && ((gcFunction=meta[MetaMethods.GC].O)!=null) &&
				(gcFunction!=NilClass.Instance)) 
			{
				LuaState L=new LuaState();
				L.Stack.Values[L.Stack.Top++].O=this;
				gcFunction.Call(L,0,0);
			}
		}

		Node NewKey(LuaValue key) 
		{
			Node mp = MainPosition(key);
			if (mp.val.O!=NilClass.Instance) 
			{  /* main position is not free? */
				Node othern = MainPosition(mp.key);  /* `mp' of colliding node */
				Node n = this.node[this.firstFree];  /* get a free place */
				if (othern != mp) 
				{  /* is colliding node out of its main position? */
					/* yes; move colliding node into free position */
					while (othern.next != mp) othern=othern.next;  /* find previous */
					othern.next = n;  /* redo the chain with `n' in place of `mp' */
					n.Copy(mp); /* copy colliding node into free pos. (mp->next also goes) */
					mp.next = null;  /* now `mp' is free */
					mp.val.O = NilClass.Instance;
				}
				else 
				{  /* colliding node is in its own main position */
					/* new node will go into free position */
					n.next = mp.next;  /* chain new position */
					mp.next = n;
					mp = n;
				}
			}
			mp.key = key;  /* write barrier */
			for (;;) 
			{  /* correct `firstfree' */
				if (this.node[this.firstFree].key.O==NilClass.Instance)
					return mp;  /* OK; table still has a free place */
				else if (this.firstFree == 0) break;  /* cannot decrement from here */
				else this.firstFree=this.firstFree-1;
			}
			/* no more free places; must create one */
			mp.val.O = FalseClass.Instance; /* avoid new key being removed */
			Rehash();  /* grow table */
			Node newn = Get(key);  /* get new position */
			newn.val.O = NilClass.Instance;
			return newn;
		}

		Node GetAny(LuaValue key) 
		{
			if (key.O==NilClass.Instance) return Node.NilNode;
			else 
			{
				Node n = MainPosition(key);
				do 
				{  /* check whether `key' is somewhere in the chain */
					if (key==n.key) return n;  /* that's it */
					else n = n.next;
				} while (n!=null);
				return Node.NilNode;
			}
		}

        internal Node GetNum(int key) 
		{
			if (1 <= key && key <= this.sizeArray) 
			{
				return this.array[key-1];
			}
			else
			{
				//Console.WriteLine("hash access "+key+" "+this);
				double nk = (double)key;
				Node n = HashNum(nk);
				do 
				{  /* check whether `key' is somewhere in the chain */
					if (n.key.O==null && n.key.N == nk)
						return n;  /* that's it */
					else n = n.next;
				} while (n!=null);
				return Node.NilNode;
			}
		}

        internal Node GetStr(LuaString key) 
		{
			Node n = HashRef(key);
			do 
			{  /* check whether `key' is somewhere in the chain */
				if (n.key.O!=null && n.key.O.Tag == LuaTypes.LUA_TSTRING &&
				  ((LuaString)n.key.O).s==key.s)
					return n;  /* that's it */
				else n = n.next;
			} while (n!=null);
			return Node.NilNode;
		}

        internal Node Get(LuaValue key) 
		{
			if(key.O==null) 
			{
				int k=(int)key.N;
				if (((double)k) == key.N)  /* is an integer index? */
					return GetNum(k);  /* use specialized version */
				else
					return GetAny(key);
			} 
			else 
			{
				if(key.O.Tag==LuaTypes.LUA_TSTRING) 
					return GetStr((LuaString)key.O);
				else
					return GetAny(key);
			}
		}

        internal void Set(LuaValue key, LuaValue val) 
		{
			Node p = Get(key);
			if(p != Node.NilNode) 
			{
				p.val = val;
			} 
			else 
			{
				if(key.O==null && Double.IsNaN(key.N))
					throw new RuntimeException("table index is NaN");
				if(key.O!=null && key.O==NilClass.Instance)
					throw new RuntimeException("table index is nil");
				p = NewKey(key);
				p.val = val;
			}
		}

		internal void SetNum(int key, LuaValue val) 
		{
			Node p = GetNum(key);
			if(p != Node.NilNode) 
			{
				p.val = val;
			} 
			else 
			{
				p = NewKey(new LuaValue((double)key,null));
				p.val = val;
			}
		}

		Node MainPosition(LuaValue key) 
		{
			if(key.O==null)
				return HashNum(key.N);
			else 
				return HashRef(key.O);
		}

		Node HashNum(double n) 
		{
			uint hsh=(uint)n.GetHashCode();
			return this.node[hsh%((this.node.Length-1)|1)];
		}
        
		Node HashRef(LuaReference r) 
		{
			if(r.Tag==LuaTypes.LUA_TUSERDATA)
				return this.node[((uint)r.GetHashCode())%((this.node.Length-1)|1)];
			else
				return this.node[r.Hash%((this.node.Length-1)|1)];
		}

		public override LuaReference Metatable 
		{
			get 
			{
				return meta;
			}
			set 
			{
				meta=value;
				/*string mode=(string)meta[MetaMethods.Mode].CLRObject;
				if(mode!=null) 
				{
					bool weakk=(mode.IndexOf('k')!=-1);
					bool weakv=(mode.IndexOf('k')!=-1);
					SetMode(weakk,weakv);
				}*/
			}
		}

		public override object CLRObject 
		{
			get 
			{
				return this;
			}
		}
	
		public override LuaValue this[LuaState L, LuaValue index] 
		{
			get 
			{
				LuaValue v = Get(index).val;
				if(v.O==NilClass.Instance && meta!=NilClass.Instance) 
				{
					LuaReference metaIndex=meta[MetaMethods.Index].O;
					if(metaIndex!=NilClass.Instance && metaIndex!=null) 
					{
						if(metaIndex.Tag==LuaTypes.LUA_TFUNCTION) 
						{
							L.Stack.Values[L.Stack.Top].O=metaIndex;
							L.Stack.Values[L.Stack.Top+1].O=this;
							L.Stack.Values[L.Stack.Top+2]=index;
							L.Stack.Check(3);
							L.Stack.Top+=3;
							metaIndex.Call(L,1,L.Stack.Top-2);
							L.Stack.Top--;
							v=L.Stack.Values[L.Stack.Top];
						} 
						else v=metaIndex[L,index];
					}
					else v.O=NilClass.Instance;
				}
				return v;
			}
			set 
			{
				if(meta==NilClass.Instance) 
				{
					Set(index,value);
				} 
				else 
				{
					LuaValue v=Get(index).val;
					if(v.O!=NilClass.Instance)
						Set(index,value);
					else
					{
						LuaReference metaNewIndex=meta[MetaMethods.NewIndex].O;
						if(metaNewIndex==NilClass.Instance || metaNewIndex==null) 
							Set(index,value);
						else if(metaNewIndex.Tag==LuaTypes.LUA_TFUNCTION)
						{
							L.Stack.Values[L.Stack.Top].O=metaNewIndex;
							L.Stack.Values[L.Stack.Top+1].O=this;
							L.Stack.Values[L.Stack.Top+2]=index;
							L.Stack.Values[L.Stack.Top+3]=value;
							L.Stack.Check(4);
							L.Stack.Top+=4;
							metaNewIndex.Call(L,1,L.Stack.Top-3);
						}
						else metaNewIndex[L,index]=value;
					}
				}
			}
		}

		public override LuaValue this[LuaValue index] 
		{
			get 
			{
				return Get(index).val;
			}
			set 
			{
				Set(index,value);
			}
		}

		public LuaValue this[LuaState L, LuaString index] 
		{
			get 
			{
				LuaValue v=GetStr(index).val;
				if(v.O==NilClass.Instance && meta!=NilClass.Instance) 
				{
					LuaReference metaIndex=meta[MetaMethods.Index].O;
					if(metaIndex!=NilClass.Instance && metaIndex!=null) 
					{
						if(metaIndex.Tag==LuaTypes.LUA_TFUNCTION) 
						{
							L.Stack.Values[L.Stack.Top].O=metaIndex;
							L.Stack.Values[L.Stack.Top+1].O=this;
							L.Stack.Values[L.Stack.Top+2]=new LuaValue(index);
							L.Stack.Check(3);
							L.Stack.Top+=3;
							metaIndex.Call(L,1,L.Stack.Top-2);
							L.Stack.Top--;
							v=L.Stack.Values[L.Stack.Top];
						} 
						else v=metaIndex[L,new LuaValue(index)];
					}
					else v.O=NilClass.Instance;
				}
				return v;
			}
		}

		public LuaValue this[LuaString index] 
		{
			get 
			{
				return GetStr(index).val;
			}
		}

        /// <summary>
        /// helper function when ur in C# to access a table value
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public override LuaValue this[string index] 
        {
            get
            {
                return GetStr(new LuaString(index)).val;
            }
            set
            {
                Set(new LuaString(index), value);
            }
        }

		public override string ToString() 
		{
			LuaReference tostrFunction;
			if(meta!=null && meta!=NilClass.Instance && ((tostrFunction=meta[MetaMethods.Tostring].O)!=null) &&
				(tostrFunction!=NilClass.Instance)) 
			{
				LuaState L=new LuaState();
				L.Stack.Values[L.Stack.Top++].O=this;
				tostrFunction.Call(L,0,0);
				return L.Stack.Values[L.Stack.Top-1].ToStr();
			} else
				return "table: <"+base.ToString()+">";
		}

		internal bool GetNext(ref Node[] currentStructure, ref int currentIndex) 
		{
            bool intGet = GetNextInt(ref currentStructure, ref currentIndex);
            if (intGet == true)
                return true;
			if(currentStructure == array) 
			{
				currentStructure = node;
				currentIndex = -1;
				return GetNext(ref currentStructure, ref currentIndex);
			}
			return false;
		}

        /// <summary>
        /// IPairs basically only goes thro the array not the hash table
        /// </summary>
        /// <param name="currentStructure"></param>
        /// <param name="currentIndex"></param>
        /// <returns></returns>
        internal bool GetNextInt(ref Node[] currentStructure, ref int currentIndex)
        {
            if (currentStructure == null)
                currentStructure = array;
            for (currentIndex++; currentIndex < currentStructure.Length; currentIndex++)
            {
                if (currentStructure[currentIndex].val.O != NilClass.Instance || currentStructure[currentIndex].key.O != NilClass.Instance)
                {
                    if (currentStructure[currentIndex].key.O == NilClass.Instance)
                        currentStructure[currentIndex].key = new LuaValue(currentIndex + 1);
                    return true;
                }
            }
            return false;
        }

        private abstract class TableEnumerator : IEnumerator<LuaValue>
        {
            protected Node[] currentStructure;
            protected int currentIndex;
            LuaTable table;

            public TableEnumerator(LuaTable _table)
            {
                table = _table;
                Reset();
            }

            #region IEnumerator<LuaValue> Members

            public abstract LuaValue Current
            {
                get;
            }

            #endregion

            #region IDisposable Members

            public void Dispose()
            {
                currentStructure = null;
                currentIndex = -1;
                table = null;
            }

            #endregion

            #region IEnumerator Members

            public bool MoveNext()
            {
                return (table.GetNext(ref currentStructure, ref currentIndex));
            }

            public void Reset()
            {
                currentStructure = null;
                currentIndex = -1;
            }
        

            object IEnumerator.Current
            {
                get 
                {
                    return Current;
                }
            }

            #endregion
        }

 
        private class TableValueEnumarator : TableEnumerator
        {
            public TableValueEnumarator(LuaTable _table) :
                base(_table)
            {
            }
            public override LuaValue Current
            {
                get
                {
                    return currentStructure[currentIndex].val;
                }
            }
        }
        private class TableKeyEnumarator : TableEnumerator
        {
            public TableKeyEnumarator(LuaTable _table)
                :
                base(_table)
            {
            }
            public override LuaValue Current
            {
                get
                {
                    return currentStructure[currentIndex].key;
                }
            }
        }

        #region IEnumerable Members
        IEnumerator<LuaValue> IEnumerable<LuaValue>.GetEnumerator()
        {
            return new TableValueEnumarator(this);
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return new TableValueEnumarator(this);
        }

        public IEnumerator<LuaValue> GetKeyEnumerator()
        {
            return new TableKeyEnumarator(this);
        }

        #endregion


    }
}
