/*
 * XNua 0.1 Massively based on Lua2IL 0.5.0
 * Lua2IL 0.5.0
 * Compiler.cs - CIL LuaOpCode emitter for Lua2IL
 * Copyright 2003-2005 Fabio Mascarenhas
 *                2007 Dean Calver
 * 
 */

using System;
using System.Diagnostics;
using System.Collections;
using System.Reflection;
using System.Reflection.Emit;
using System.IO;
using System.Diagnostics.SymbolStore;

namespace XNua
{

	public class TailCall 
	{
		int RA;
		int RB;
		int RC;
        XNuaCompiler compiler;
		Label label;

        public TailCall(XNuaCompiler compiler, ILGenerator il, int RA, int RB, int RC) 
		{
			this.compiler=compiler;
			label=il.DefineLabel();
			il.Emit(OpCodes.Leave,label);
			this.RA=RA;
			this.RB=RB;
			this.RC=RC;
		}

		public void CompileTailCall(ILGenerator il) 
		{
			il.MarkLabel(label);
			LocalBuilder receiver=il.DeclareLocal(typeof(LuaReference));
			compiler.CompileAdjustTop(il,RA,RB);

			LocalBuilder firstIndex=il.DeclareLocal(typeof(int));
			LocalBuilder secondIndex=il.DeclareLocal(typeof(int));
			LocalBuilder top=il.DeclareLocal(typeof(int));

			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4_1);
			il.Emit(OpCodes.Sub);
			il.Emit(OpCodes.Stloc,firstIndex);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Stloc,secondIndex);
			il.Emit(OpCodes.Ldloc_0);
			il.Emit(OpCodes.Ldfld,typeof(LuaStack).GetField("Top"));
			il.Emit(OpCodes.Stloc,top);
			Label test=il.DefineLabel();
			Label loop=il.DefineLabel();
			il.Emit(OpCodes.Br,test);
			
			il.MarkLabel(loop);
			compiler.CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc,firstIndex);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			compiler.CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc,secondIndex);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Ldobj,typeof(LuaValue));
			il.Emit(OpCodes.Stobj,typeof(LuaValue));
			il.Emit(OpCodes.Ldloc,firstIndex);
			il.Emit(OpCodes.Ldc_I4_1);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Stloc,firstIndex);
			il.Emit(OpCodes.Ldloc,secondIndex);
			il.Emit(OpCodes.Ldc_I4_1);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Stloc,secondIndex);

			il.MarkLabel(test);
			il.Emit(OpCodes.Ldloc,secondIndex);
			il.Emit(OpCodes.Ldloc,top);
			il.Emit(OpCodes.Blt,loop);

			il.Emit(OpCodes.Ldloc_0);
			il.Emit(OpCodes.Ldloc,firstIndex);
			il.Emit(OpCodes.Stfld,typeof(LuaStack).GetField("Top"));

			il.Emit(OpCodes.Ldloc_0);
			il.Emit(OpCodes.Ldloc_2);
			il.Emit(OpCodes.Stfld,typeof(LuaStack).GetField("Base"));

			compiler.CompileLoadStack(il);	// L.Stack.Values
			il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
			il.Emit(OpCodes.Ldc_I4_1);
			il.Emit(OpCodes.Sub);	// L.Stack.Base-1
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));	
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));	// L.Stack.Values[L.Stack.Base-1].O - receiver
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RC-1);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldloc_2);
			il.Emit(OpCodes.Sub);
			il.Emit(OpCodes.Tailcall);
			il.Emit(OpCodes.Callvirt,typeof(LuaReference).GetMethod("Call",new Type[] {
				typeof(LuaState),typeof(int),typeof(int)}));

			il.Emit(OpCodes.Ret);
		}
	}

	public class XNuaCompiler 
	{
		public static readonly Guid LuaGUID=new Guid("{40E2B0CD-910C-4a8c-890C-982FD3665D15}");
		public static readonly Guid PucGUID=new Guid("{C48E9A9C-EB12-4257-8A7E-6EF8FE9DB293}");

		LocalBuilder doubleTemp1;
		LocalBuilder doubleTemp2;
		LocalBuilder doubleTemp3;

		int functionNumber=0;

		public Type CompileFunction(Prototype proto,string name,ModuleBuilder module,ISymbolDocumentWriter sourceDoc) 
		{
            Type[] subFunctions=CompileSubFunctions(proto,module,sourceDoc);
			string typeName=FunctionName(proto.Source,name);
			TypeBuilder compiledFunc=module.DefineType(typeName,TypeAttributes.Class | TypeAttributes.Public,typeof(LuaClosure));
			TypeBuilder compiledProto;
			FieldInfo[] constants=new FieldInfo[proto.Constants.Length];
			for(int i=0;i<constants.Length;i++)
				constants[i]=compiledFunc.DefineField("constant"+i,typeof(LuaValue),FieldAttributes.Static);
			FieldInfo compiledProtoInstance;
			CompilePrototype(proto,compiledFunc,module,out compiledProto,out compiledProtoInstance);
			ConstructorInfo cons=CompileConstructor(compiledProtoInstance,compiledFunc);
			ConstructorBuilder ccons=compiledFunc.DefineConstructor(MethodAttributes.Static,CallingConventions.Standard,new Type[0]);
			ILGenerator il=ccons.GetILGenerator();
			CompileConstants(il,proto,constants);
			CompileBody(proto,constants,subFunctions,compiledFunc,sourceDoc);
			Type t=compiledFunc.CreateType();
			compiledProto.CreateType();
			return t;
		}

		public void CompilePrototype(Prototype proto,TypeBuilder compiledFunc,ModuleBuilder module,out TypeBuilder compiledProto,out FieldInfo instance) 
		{
			compiledProto=compiledFunc.DefineNestedType("Prototype",TypeAttributes.NestedPrivate | TypeAttributes.Class,typeof(CompiledPrototype));
			instance=compiledProto.DefineField("Instance",typeof(CompiledPrototype),FieldAttributes.Static | FieldAttributes.Public);
			ConstructorBuilder cons=compiledProto.DefineConstructor(MethodAttributes.Private,CallingConventions.Standard,new Type[0]);
			ILGenerator il=cons.GetILGenerator();
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Call,typeof(CompiledPrototype).GetConstructor(new Type[0]));
			CompileConstants(proto,il);
			CompileLineInfo(proto,il);
			CompileUpvalues(proto,il);
			CompileStoreToStrField(proto.Source,typeof(CompiledPrototype).GetField("Source"),il);
			CompileStoreToIntField(proto.LineDefined,typeof(CompiledPrototype).GetField("LineDefined"),il);
			CompileStoreToIntField(proto.NUpvalues,typeof(CompiledPrototype).GetField("NUpvalues"),il);
			CompileStoreToIntField(proto.NParams,typeof(CompiledPrototype).GetField("NParams"),il);
			CompileStoreToBoolField(proto.IsVararg,typeof(CompiledPrototype).GetField("IsVararg"),il);
			CompileStoreToIntField(proto.MaxStack,typeof(CompiledPrototype).GetField("MaxStack"),il);
			il.Emit(OpCodes.Ret);
			ConstructorBuilder classCons=compiledProto.DefineTypeInitializer();
			il=classCons.GetILGenerator();
			il.Emit(OpCodes.Newobj,cons);
			il.Emit(OpCodes.Stsfld,instance);
			il.Emit(OpCodes.Ret);
		}

		public void CompileConstants(Prototype proto,ILGenerator il) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldc_I4,proto.Constants.Length);
			il.Emit(OpCodes.Newarr,typeof(LuaValue));
			for(int i=0;i<proto.Constants.Length;i++) 
			{
				il.Emit(OpCodes.Dup);
				il.Emit(OpCodes.Ldc_I4,i);
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
				if(proto.Constants[i].IsReference) 
				{
					if(proto.Constants[i].O==NilClass.Instance) 
					{
						il.Emit(OpCodes.Ldsfld,typeof(NilClass).GetField("Instance"));
						il.Emit(OpCodes.Newobj,typeof(LuaValue).GetConstructor(new Type[] {typeof(LuaReference)}));
					} 
					else 
					{
						il.Emit(OpCodes.Ldstr,(string)proto.Constants[i].O.CLRObject);
						il.Emit(OpCodes.Newobj,typeof(LuaValue).GetConstructor(new Type[] {typeof(string)}));
					}
				} 
				else 
				{
					il.Emit(OpCodes.Ldc_R8,proto.Constants[i].N);
					il.Emit(OpCodes.Newobj,typeof(LuaValue).GetConstructor(new Type[] {typeof(double)}));
				}
				il.Emit(OpCodes.Stobj,typeof(LuaValue));
			}
			il.Emit(OpCodes.Stfld,typeof(CompiledPrototype).GetField("Constants"));
		}

		public void CompileConstants(ILGenerator il,Prototype proto,FieldInfo[] fields) 
		{
			for(int i=0;i<proto.Constants.Length;i++) 
			{
				il.Emit(OpCodes.Ldsflda,fields[i]);
				if(proto.Constants[i].IsReference) 
				{
					if(proto.Constants[i].O==NilClass.Instance) 
					{
						il.Emit(OpCodes.Ldsfld,typeof(NilClass).GetField("Instance"));
						il.Emit(OpCodes.Newobj,typeof(LuaValue).GetConstructor(new Type[] {typeof(LuaReference)}));
					} 
					else 
					{
						il.Emit(OpCodes.Ldstr,(string)proto.Constants[i].O.CLRObject);
						il.Emit(OpCodes.Newobj,typeof(LuaValue).GetConstructor(new Type[] {typeof(string)}));
					}
				} 
				else 
				{
					il.Emit(OpCodes.Ldc_R8,proto.Constants[i].N);
					il.Emit(OpCodes.Newobj,typeof(LuaValue).GetConstructor(new Type[] {typeof(double)}));
				}
				il.Emit(OpCodes.Stobj,typeof(LuaValue));
			}
			il.Emit(OpCodes.Ret);
		}

		public void CompileLineInfo(Prototype proto,ILGenerator il) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldc_I4,proto.LineInfo.Length);
			il.Emit(OpCodes.Newarr,typeof(int));
			for(int i=0;i<proto.LineInfo.Length;i++) 
			{
				il.Emit(OpCodes.Dup);
				il.Emit(OpCodes.Ldc_I4,i);
				il.Emit(OpCodes.Ldc_I4,proto.LineInfo[i]);
				il.Emit(OpCodes.Stelem_I4);
			}
			il.Emit(OpCodes.Stfld,typeof(CompiledPrototype).GetField("LineInfo"));
		}

		public void CompileUpvalues(Prototype proto,ILGenerator il) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldc_I4,proto.Upvalues.Length);
			il.Emit(OpCodes.Newarr,typeof(string));
			for(int i=0;i<proto.Upvalues.Length;i++) 
			{
				il.Emit(OpCodes.Dup);
				il.Emit(OpCodes.Ldc_I4,i);
				il.Emit(OpCodes.Ldstr,proto.Upvalues[i]);
				il.Emit(OpCodes.Stelem_Ref);
			}
			il.Emit(OpCodes.Stfld,typeof(CompiledPrototype).GetField("Upvalues"));
		}

		public void CompileStoreToStrField(string val,FieldInfo field,ILGenerator il) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldstr,val);
			il.Emit(OpCodes.Stfld,field);
		}

		public void CompileStoreToIntField(int val,FieldInfo field,ILGenerator il) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldc_I4,val);
			il.Emit(OpCodes.Stfld,field);
		}

		public void CompileStoreToBoolField(bool val,FieldInfo field,ILGenerator il) 
		{
			il.Emit(OpCodes.Ldarg_0);
			if(val)
				il.Emit(OpCodes.Ldc_I4_1);
			else
				il.Emit(OpCodes.Ldc_I4_0);
			il.Emit(OpCodes.Stfld,field);
		}

		public Hashtable CompileJumpPoints(ILGenerator il,Instruction[] code) 
		{
			Hashtable labels=new Hashtable();
			for(int pc=0;pc<code.Length;pc++) 
			{
				LuaOpCode op=code[pc].OpCode;
				if((op==LuaOpCode.OP_JMP || op==LuaOpCode.OP_FORLOOP ||
					op==LuaOpCode.OP_TFORPREP) && labels[pc+code[pc].ArgsBx]==null) 
					labels[pc+code[pc].ArgsBx]=il.DefineLabel();
			}
			return labels;
		}

		public void CompileBody(Prototype proto,FieldInfo[] constants,Type[] subFunctions,TypeBuilder type,ISymbolDocumentWriter sourceDoc) 
		{
			MethodBuilder method=type.DefineMethod("Call",MethodAttributes.HideBySig | MethodAttributes.Virtual |
				MethodAttributes.Public,typeof(int),new Type[] { typeof(LuaState),typeof(int),typeof(int) });
			ILGenerator il=method.GetILGenerator();
			FieldBuilder[] upvals=new FieldBuilder[proto.NUpvalues];
			for(int i=0;i<proto.NUpvalues;i++)
				upvals[i]=type.DefineField("upvalue"+i,typeof(UpValue),
					FieldAttributes.Public);
			ArrayList tailCalls=new ArrayList();
			CompileDeclarations(il);
			CompilePreamble(il,proto);
			Label leave=il.BeginExceptionBlock();
			LocalBuilder retVal=il.DeclareLocal(typeof(int));
			doubleTemp1 = il.DeclareLocal(typeof(double));
			doubleTemp2 = il.DeclareLocal(typeof(double));
			doubleTemp3 = il.DeclareLocal(typeof(double));
			Hashtable labels=CompileJumpPoints(il,proto.Code);
			int pc=0;
			do {
				il.MarkSequencePoint(sourceDoc,proto.LineInfo[pc],0,proto.LineInfo[pc]+1,0);
//				il.MarkSequencePoint(sourceDoc,pc+1,0,pc+2,0);
				Label lab;
				switch(proto.Code[pc].OpCode) 
				{
					case LuaOpCode.OP_MOVE:
						CompileOpMove(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB);
						break;
					case LuaOpCode.OP_LOADK:
						CompileOpLoadK(il,constants,proto.Code[pc].ArgA,proto.Code[pc].ArgBx);
						break;
					case LuaOpCode.OP_LOADBOOL:
						CompileAB(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,"OpLoadBool");
						if(proto.Code[pc].ArgC!=0) 
						{
							if(labels[pc+1]==null) 
							{
								lab=il.DefineLabel();
								labels[pc+1]=lab;
							} 
							else lab=(Label)labels[pc+1];
							il.Emit(OpCodes.Br,lab);
						}
						break;
					case LuaOpCode.OP_LOADNIL:
						CompileAB(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,"OpLoadNil");
						break;
					case LuaOpCode.OP_GETUPVAL:
						CompileOpGetUpVal(il,upvals,proto.Code[pc].ArgA,proto.Code[pc].ArgB);
						//CompileAB(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,"OpGetUpVal");
						break;
					case LuaOpCode.OP_GETGLOBAL:
						CompileOpGetGlobal(il,constants,proto.Code[pc].ArgA,proto.Code[pc].ArgBx);
						break;
					case LuaOpCode.OP_GETTABLE:
						CompileOpGetTable(il,constants,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC);
						break;
					case LuaOpCode.OP_SETUPVAL:
						CompileOpSetUpVal(il,upvals,proto.Code[pc].ArgA,proto.Code[pc].ArgB);
						//CompileAB(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,"OpSetUpVal");
						break;
					case LuaOpCode.OP_SETGLOBAL:
						CompileOpSetGlobal(il,constants,proto.Code[pc].ArgA,proto.Code[pc].ArgBx);
						break;
					case LuaOpCode.OP_SETTABLE:
						CompileOpSetTable(il,constants,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC);
						break;
					case LuaOpCode.OP_NEWTABLE:
						CompileOpNewTable(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC);
						break;
					case LuaOpCode.OP_SELF:
						CompileOpSelf(il,constants,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC);
						break;
					case LuaOpCode.OP_ADD:
						CompileArith(il,constants,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC,LuaOpCode.OP_ADD);
						break;
					case LuaOpCode.OP_SUB:
						CompileArith(il,constants,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC,LuaOpCode.OP_SUB);
						break;
					case LuaOpCode.OP_MUL:
						CompileArith(il,constants,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC,LuaOpCode.OP_MUL);
						break;
					case LuaOpCode.OP_DIV:
						CompileArith(il,constants,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC,LuaOpCode.OP_DIV);
						break;
					case LuaOpCode.OP_POW:
						CompileArith(il,constants,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC,LuaOpCode.OP_POW);
						break;
					case LuaOpCode.OP_UNM:
						CompileAB(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,"OpUnm");
						break;
					case LuaOpCode.OP_NOT:
						CompileAB(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,"OpNot");
						break;
					case LuaOpCode.OP_CONCAT:
						CompileABC(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC,"OpConcat");
						break;
					case LuaOpCode.OP_JMP:
						lab=(Label)labels[pc+proto.Code[pc].ArgsBx];
						il.Emit(OpCodes.Br,lab);
						break;
					case LuaOpCode.OP_EQ:
						CompileComp(il,constants,pc,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC,LuaOpCode.OP_EQ,labels);
						break;
					case LuaOpCode.OP_LT:
						CompileComp(il,constants,pc,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC,LuaOpCode.OP_LT,labels);
						break;
					case LuaOpCode.OP_LE:
						CompileComp(il,constants,pc,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC,LuaOpCode.OP_LE,labels);
						break;
					case LuaOpCode.OP_TEST:
						CompileOpTest(il,pc,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC,labels);
						//CompileABCSkip(il,pc,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC,"OpTest",labels);
						break;
					case LuaOpCode.OP_CALL:
						CompileOpCall(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC);
						break;
					case LuaOpCode.OP_TAILCALL:
						tailCalls.Add(new TailCall(this,il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC));
//						CompileOpCall(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,proto.Code[pc].ArgC);
						break;
					case LuaOpCode.OP_RETURN:
						CompileReturn(il,proto.Code[pc].ArgA,proto.Code[pc].ArgB,leave,retVal);
						break;
					case LuaOpCode.OP_FORLOOP:
						CompileForLoop(il,pc,proto.Code[pc].ArgA,proto.Code[pc].ArgsBx,labels);
						break;
					case LuaOpCode.OP_TFORLOOP:
						CompileTForLoop(il,pc,proto.Code[pc].ArgA,proto.Code[pc].ArgsBx,proto.Code[pc].ArgC,labels);
						break;
					case LuaOpCode.OP_TFORPREP:
						CompileTForPrep(il,pc,proto.Code[pc].ArgA,proto.Code[pc].ArgsBx,labels);
						break;
					case LuaOpCode.OP_SETLIST:
						CompileSetList(il,proto.Code[pc].ArgA,proto.Code[pc].ArgBx);
						break;
					case LuaOpCode.OP_SETLISTO:
						CompileSetListO(il,proto.Code[pc].ArgA,proto.Code[pc].ArgBx);
						break;
					case LuaOpCode.OP_CLOSE:
						CompileA(il,proto.Code[pc].ArgA,"OpClose");
						break;
					case LuaOpCode.OP_CLOSURE:
						pc=CompileOpClosure(il,proto,upvals,proto.Code[pc].ArgA,pc,subFunctions[proto.Code[pc].ArgBx]);
						break;
					default:
						// Error
						break;
				}
				if(labels[pc]!=null)
					il.MarkLabel((Label)labels[pc]);
				pc++;
			} while(pc<proto.Code.Length);
			il.BeginFaultBlock();
			CompileCloseUpvals(il);
			il.EndExceptionBlock();
			il.Emit(OpCodes.Ldloc,retVal);
			il.Emit(OpCodes.Ret);
			foreach(object o in tailCalls) 
			{
				TailCall t=(TailCall)o;
				t.CompileTailCall(il);
			}
		}

		public void CompileCloseUpvals(ILGenerator il) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4_0);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("OpClose"));
		}

		public void CompileDeclarations(ILGenerator il) 
		{
			il.DeclareLocal(typeof(LuaStack));
			il.DeclareLocal(typeof(int));
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldfld,typeof(LuaState).GetField("Stack"));
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Stloc_0);
			il.Emit(OpCodes.Ldfld,typeof(LuaStack).GetField("Base"));
			il.Emit(OpCodes.Stloc_1);
		}

		public void CompileLoadStack(ILGenerator il) 
		{
			il.Emit(OpCodes.Ldloc_0);
			il.Emit(OpCodes.Ldfld,typeof(LuaStack).GetField("Values"));
			//il.Emit(OpCodes.Ldloc_1);
		}

		public void CompilePreamble(ILGenerator il,Prototype proto) 
		{
			Label whileTest=il.DefineLabel();
			Label whileEnd=il.DefineLabel();
			LocalBuilder lastBase=il.DeclareLocal(typeof(int));
			LocalBuilder top=il.DeclareLocal(typeof(int));
			if(proto.IsVararg) 
			{
				il.Emit(OpCodes.Ldarg_0);
				il.Emit(OpCodes.Ldarg_1);
				il.Emit(OpCodes.Ldarg_2);
				il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("AdjustVarargs"));
			}
			il.Emit(OpCodes.Ldloc_0);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Stloc,lastBase);
			il.Emit(OpCodes.Ldarg_3);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Stloc_1);
			il.Emit(OpCodes.Stfld,typeof(LuaStack).GetField("Base"));
			il.Emit(OpCodes.Ldloc_0);
			il.Emit(OpCodes.Ldc_I4,(int)proto.MaxStack);
			il.Emit(OpCodes.Call,typeof(LuaStack).GetMethod("Check"));
			il.Emit(OpCodes.Ldloc_0);
			il.Emit(OpCodes.Ldfld,typeof(LuaStack).GetField("Top"));
			il.Emit(OpCodes.Stloc,top);
			il.MarkLabel(whileTest);
			il.Emit(OpCodes.Ldloc,top);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,(int)proto.MaxStack);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Bge,whileEnd);
			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc,top);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Ldsfld,typeof(NilClass).GetField("Instance"));
			il.Emit(OpCodes.Stfld,typeof(LuaValue).GetField("O"));
			il.Emit(OpCodes.Ldloc,top);
			il.Emit(OpCodes.Ldc_I4_1);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Stloc,top);
			il.Emit(OpCodes.Br,whileTest);
			il.MarkLabel(whileEnd);
			il.Emit(OpCodes.Ldloc_0);
			il.Emit(OpCodes.Ldloc,top);
			il.Emit(OpCodes.Stfld,typeof(LuaStack).GetField("Top"));
		}

		public void CompileOpMove(ILGenerator il,int RA,int RB) 
		{
			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc_1);
            il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RB);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Ldobj,typeof(LuaValue));
			il.Emit(OpCodes.Stobj,typeof(LuaValue));
		}

		public void CompileOpGetUpVal(ILGenerator il,FieldBuilder[] upvals,int RA,int RB) 
		{
			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldfld,upvals[RB]);
			il.Emit(OpCodes.Call,typeof(UpValue).GetMethod("get_Value"));
			il.Emit(OpCodes.Stobj,typeof(LuaValue));
		}

		public void CompileOpSetUpVal(ILGenerator il,FieldBuilder[] upvals,int RA,int RB) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldfld,upvals[RB]);
			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Ldobj,typeof(LuaValue));
			il.Emit(OpCodes.Call,typeof(UpValue).GetMethod("set_Value"));
		}

		public void CompileOpLoadK(ILGenerator il,FieldInfo[] constants,int RA,int RBx) 
		{
			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			CompileSetConstant(il,constants,RBx);
			il.Emit(OpCodes.Pop);
		}

		public void CompileSetConstant(ILGenerator il,FieldInfo[] constants,int idx) 
		{
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Ldsfld,constants[idx]);
			il.Emit(OpCodes.Stobj,typeof(LuaValue));
		}

		public void CompileLoadConstantA(ILGenerator il,FieldInfo[] constants,int idx) 
		{
			il.Emit(OpCodes.Ldsflda,constants[idx]);
		}

		public void CompileLoadConstant(ILGenerator il,FieldInfo[] constants,int idx) 
		{
			il.Emit(OpCodes.Ldsfld,constants[idx]);
		}

		public void CompileOpGetGlobal(ILGenerator il,FieldInfo[] constants,int RA,int RBx) 
		{
			CompileLoadStack(il);	// L.Stack.Values
			il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);	// L.Stack.Base+RA
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));	// L.Stack.Values[L.Stack.Base+RA]
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldfld,typeof(LuaFunction).GetField("globals"));
			il.Emit(OpCodes.Castclass,typeof(LuaTable));
			il.Emit(OpCodes.Ldarg_1);	// L - first arg
			CompileLoadConstantA(il,constants,RBx);
			//il.Emit(OpCodes.Ldobj,typeof(LuaValue));
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));
			il.Emit(OpCodes.Castclass,typeof(LuaString));
			il.Emit(OpCodes.Call,typeof(LuaTable).GetMethod("get_Item",new Type[] {typeof(LuaState),typeof(LuaString)}));
			il.Emit(OpCodes.Stobj,typeof(LuaValue));
		}

		public void CompileOpGetTable(ILGenerator il,FieldInfo[] constants,int RA,int RB,int RC) 
		{
			CompileLoadStack(il);	// L.Stack.Values
			il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);	// L.Stack.Base+RA
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));	// L.Stack.Values[L.Stack.Base+RA]
			
			CompileLoadStack(il);	// L.Stack.Values
			il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
			il.Emit(OpCodes.Ldc_I4,RB);
			il.Emit(OpCodes.Add);	// L.Stack.Base+RB
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));	
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));	// L.Stack.Values[L.Stack.Base+RB].O - receiver
			il.Emit(OpCodes.Ldarg_1);	// L - first arg
			if(RC<LuaState.MAX_LOCAL_STACK) 
			{
				CompileLoadStack(il);	// L.Stack.Values
				il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
				il.Emit(OpCodes.Ldc_I4,RC);
				il.Emit(OpCodes.Add);	// L.Stack.Base+RC
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
				il.Emit(OpCodes.Ldobj,typeof(LuaValue)); // L.Stack.Values[L.Stack.Base+RC] - index
			} 
			else 
			{
				CompileLoadConstant(il,constants,RC-LuaState.MAX_LOCAL_STACK);
			}
			il.Emit(OpCodes.Callvirt,typeof(LuaReference).GetMethod("get_Item",new Type[] {typeof(LuaState),typeof(LuaValue)}));

			il.Emit(OpCodes.Stobj,typeof(LuaValue)); // Store in L.Stack.Values[L.Stack.Base+RA]
		}

		public void CompileOpSetGlobal(ILGenerator il,FieldInfo[] constants,int RA,int RBx) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldfld,typeof(LuaFunction).GetField("globals"));
			il.Emit(OpCodes.Castclass,typeof(LuaTable));

			il.Emit(OpCodes.Ldarg_1);	// L
			CompileLoadConstant(il,constants,RBx);

			CompileLoadStack(il);	// L.Stack.Values
			il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);	// L.Stack.Base+RA
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Ldobj,typeof(LuaValue));	// L.Stack.Values[L.Stack.Base+RA] - value

			il.Emit(OpCodes.Call,typeof(LuaTable).GetMethod("set_Item",new Type[] {typeof(LuaState),
				typeof(LuaValue),typeof(LuaValue)}));
		}

		public void CompileOpSetTable(ILGenerator il,FieldInfo[] constants,int RA,int RB,int RC) 
		{
			CompileLoadStack(il);	// L.Stack.Values
			il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);	// L.Stack.Base+RA
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));	
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));	// L.Stack.Values[L.Stack.Base+RA].O - receiver

			il.Emit(OpCodes.Ldarg_1);	// L
			if(RB<LuaState.MAX_LOCAL_STACK) 
			{
				CompileLoadStack(il);	// L.Stack.Values
				il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
				il.Emit(OpCodes.Ldc_I4,RB);
				il.Emit(OpCodes.Add);	// L.Stack.Base+RB
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
				il.Emit(OpCodes.Ldobj,typeof(LuaValue));	// L.Stack.Values[L.Stack.Base+RB] - index
			} 
			else 
			{
				CompileLoadConstant(il,constants,RB-LuaState.MAX_LOCAL_STACK);
			}
			if(RC<LuaState.MAX_LOCAL_STACK) 
			{
				CompileLoadStack(il);	// L.Stack.Values
				il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
				il.Emit(OpCodes.Ldc_I4,RC);
				il.Emit(OpCodes.Add);	// L.Stack.Base+RC
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
				il.Emit(OpCodes.Ldobj,typeof(LuaValue));	// L.Stack.Values[L.Stack.Base+RC] - value
			} 
			else 
			{
				CompileLoadConstant(il,constants,RC-LuaState.MAX_LOCAL_STACK);
			}
			il.Emit(OpCodes.Callvirt,typeof(LuaReference).GetMethod("set_Item",new Type[] {typeof(LuaState),
				  typeof(LuaValue),typeof(LuaValue)}));
		}

        public void CompileArith(ILGenerator il, FieldInfo[] constants, int RA, int RB, int RC, LuaOpCode op) 
		{
			Label lab1=il.DefineLabel();
			Label lab2=il.DefineLabel();

			CompileLoadStack(il);	// L.Stack.Values
			il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);	// L.Stack.Base+RA
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Dup);
			
			if(RC<LuaState.MAX_LOCAL_STACK) 
			{
				CompileLoadStack(il);	// L.Stack.Values
				il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
				il.Emit(OpCodes.Ldc_I4,RC);
				il.Emit(OpCodes.Add);	// L.Stack.Base+RB
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			} 
			else 
			{
				CompileLoadConstantA(il,constants,RC-LuaState.MAX_LOCAL_STACK);
			}
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));
			il.Emit(OpCodes.Brtrue,lab1);
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("N"));
			il.Emit(OpCodes.Stloc,doubleTemp1);
			if(RB<LuaState.MAX_LOCAL_STACK) 
			{
				CompileLoadStack(il);	// L.Stack.Values
				il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
				il.Emit(OpCodes.Ldc_I4,RB);
				il.Emit(OpCodes.Add);	// L.Stack.Base+RC
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			} 
			else 
			{
				CompileLoadConstantA(il,constants,RB-LuaState.MAX_LOCAL_STACK);
			}
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));
			il.Emit(OpCodes.Brtrue,lab1);
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("N"));
			il.Emit(OpCodes.Ldloc,doubleTemp1);
			switch(op) 
			{
				case LuaOpCode.OP_ADD:
					il.Emit(OpCodes.Add);
					break;
				case LuaOpCode.OP_SUB:
					il.Emit(OpCodes.Sub);
					break;
				case LuaOpCode.OP_DIV:
					il.Emit(OpCodes.Div);
					break;
				case LuaOpCode.OP_MUL:
					il.Emit(OpCodes.Mul);
					break;
				case LuaOpCode.OP_POW:
					il.Emit(OpCodes.Call,typeof(Math).GetMethod("Pow"));
					break;
				default:
					il.Emit(OpCodes.Ldstr,"Invalid operation to Arith");
					il.Emit(OpCodes.Newobj,
						typeof(RuntimeException).GetConstructor(new Type[] {typeof(string)}));
					il.Emit(OpCodes.Throw);
					break;
			}
			il.Emit(OpCodes.Stfld,typeof(LuaValue).GetField("N"));
			il.Emit(OpCodes.Ldnull);
			il.Emit(OpCodes.Stfld,typeof(LuaValue).GetField("O"));
			il.Emit(OpCodes.Br,lab2);

			il.MarkLabel(lab1);
			il.Emit(OpCodes.Pop);
			il.Emit(OpCodes.Pop);
			il.Emit(OpCodes.Pop);
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			if(RB<LuaState.MAX_LOCAL_STACK) 
			{
				CompileLoadStack(il);
				il.Emit(OpCodes.Ldc_I4,RB);
			}
			else 
			{
				il.Emit(OpCodes.Ldarg_0);
				il.Emit(OpCodes.Ldfld,typeof(LuaClosure).GetField("proto"));
				il.Emit(OpCodes.Ldfld,typeof(CompiledPrototype).GetField("Constants"));
				il.Emit(OpCodes.Ldc_I4,RB-LuaState.MAX_LOCAL_STACK);
			}
			if(RC<LuaState.MAX_LOCAL_STACK) 
			{
				CompileLoadStack(il);
				il.Emit(OpCodes.Ldc_I4,RC);
			}
			else 
			{
				il.Emit(OpCodes.Ldarg_0);
				il.Emit(OpCodes.Ldfld,typeof(LuaClosure).GetField("proto"));
				il.Emit(OpCodes.Ldfld,typeof(CompiledPrototype).GetField("Constants"));
				il.Emit(OpCodes.Ldc_I4,RC-LuaState.MAX_LOCAL_STACK);
			}
			il.Emit(OpCodes.Ldc_I4_S,(int)op);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("Arith"));

			il.MarkLabel(lab2);
		}

        public void CompileComp(ILGenerator il, FieldInfo[] constants, int pc, int RA, int RB, int RC, LuaOpCode op, Hashtable labels) 
		{
			Label lab1=il.DefineLabel();
			Label lab2=il.DefineLabel();

			int firstArg=(op==LuaOpCode.OP_LE ? RC : RB);
			int secondArg=(op==LuaOpCode.OP_LE ? RB : RC);

			if(secondArg<LuaState.MAX_LOCAL_STACK) 
			{
				CompileLoadStack(il);	// L.Stack.Values
				il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
				il.Emit(OpCodes.Ldc_I4,secondArg);
				il.Emit(OpCodes.Add);	// L.Stack.Base+RC
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			} 
			else 
			{
				CompileLoadConstantA(il,constants,secondArg-LuaState.MAX_LOCAL_STACK);
			}
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));
			il.Emit(OpCodes.Brtrue,lab1);
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("N"));
			il.Emit(OpCodes.Stloc,doubleTemp1);
			if(firstArg<LuaState.MAX_LOCAL_STACK) 
			{
				CompileLoadStack(il);	// L.Stack.Values
				il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
				il.Emit(OpCodes.Ldc_I4,firstArg);
				il.Emit(OpCodes.Add);	// L.Stack.Base+RB
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			} 
			else 
			{
				CompileLoadConstantA(il,constants,firstArg-LuaState.MAX_LOCAL_STACK);
			}
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));
			il.Emit(OpCodes.Brtrue,lab1);
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("N"));
			il.Emit(OpCodes.Ldloc,doubleTemp1);
			switch(op) 
			{
				case LuaOpCode.OP_EQ:
					il.Emit(OpCodes.Ceq);
					break;
				case LuaOpCode.OP_LT:
					il.Emit(OpCodes.Clt);
					break;
				case LuaOpCode.OP_LE:
					il.Emit(OpCodes.Cgt);
					break;
				default:
					il.Emit(OpCodes.Ldstr,"Invalid comparison operation");
					il.Emit(OpCodes.Newobj,
						typeof(RuntimeException).GetConstructor(new Type[] {typeof(string)}));
					il.Emit(OpCodes.Throw);
					break;
			}
			il.Emit(OpCodes.Br,lab2);

			il.MarkLabel(lab1);
			il.Emit(OpCodes.Pop);
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			if(RB<LuaState.MAX_LOCAL_STACK) 
			{
				CompileLoadStack(il);	// L.Stack.Values
				il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
				il.Emit(OpCodes.Ldc_I4,RB);
				il.Emit(OpCodes.Add);	// L.Stack.Base+RB
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			}
			else 
			{
				CompileLoadConstantA(il,constants,RB-LuaState.MAX_LOCAL_STACK);
			}
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));
			if(RC<LuaState.MAX_LOCAL_STACK) 
			{
				CompileLoadStack(il);	// L.Stack.Values
				il.Emit(OpCodes.Ldloc_1);	// L.Stack
				il.Emit(OpCodes.Ldc_I4,RC);
				il.Emit(OpCodes.Add);	// L.Stack.Base+RC
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			}
			else 
			{
				CompileLoadConstantA(il,constants,RC-LuaState.MAX_LOCAL_STACK);
			}
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));
			switch(op) 
			{
				case LuaOpCode.OP_EQ:
					il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("EqualVal"));
					break;
				case LuaOpCode.OP_LT:
					il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("LessThanVal"));
					break;
				case LuaOpCode.OP_LE:
					il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("LessEqualVal"));
					break;
				default:
					il.Emit(OpCodes.Ldstr,"Invalid comparison operation");
					il.Emit(OpCodes.Newobj,
						typeof(RuntimeException).GetConstructor(new Type[] {typeof(string)}));
					il.Emit(OpCodes.Throw);
					break;
			}

			il.MarkLabel(lab2);

			Label lab;
			if(labels[pc+1]==null) 
			{
				lab=il.DefineLabel();
				labels[pc+1]=lab;
			} 
			else lab=(Label)labels[pc+1];
			if(RA==0)
				il.Emit(OpCodes.Brtrue,lab);
			else
				il.Emit(OpCodes.Brfalse,lab);
		}

		public void CompileDebugWrite(ILGenerator il) 
		{
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Call,typeof(Console).GetMethod("WriteLine",new Type[] { typeof(double) }));
		}

		public void CompileOpNewTable(ILGenerator il,int RA,int RB,int RC) 
		{
			CompileLoadStack(il);	// L.Stack.Values
			il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);	// L.Stack.Base+RA
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));	
			il.Emit(OpCodes.Newobj,typeof(LuaTable).GetConstructor(new Type[0] {}));
			il.Emit(OpCodes.Stfld,typeof(LuaValue).GetField("O"));
		}

		public void CompileOpSelf(ILGenerator il,FieldInfo[] constants,int RA,int RB,int RC) 
		{
			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RA+1);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RB);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Ldobj,typeof(LuaValue));
			il.Emit(OpCodes.Stobj,typeof(LuaValue));
			CompileOpGetTable(il,constants,RA,RB,RC);
		}

		public void CompileAdjustTop(ILGenerator il,int RA,int RB) 
		{
			if(RB!=0) 
			{
				il.Emit(OpCodes.Ldloc_0);
				il.Emit(OpCodes.Ldloc_1);
				il.Emit(OpCodes.Ldc_I4,RA+RB);
				il.Emit(OpCodes.Add);
				il.Emit(OpCodes.Stfld,typeof(LuaStack).GetField("Top"));
			}
		}

		public void CompileOpCall(ILGenerator il,int RA,int RB,int RC) 
		{
			CompileAdjustTop(il,RA,RB);
			CompileLoadStack(il);	// L.Stack.Values
			il.Emit(OpCodes.Ldloc_1);	// L.Stack.Base
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);	// L.Stack.Base+RA
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));	
			il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));	// L.Stack.Values[L.Stack.Base+RA].O - receiver
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RC-1);
			il.Emit(OpCodes.Ldc_I4,RA+1);
			il.Emit(OpCodes.Callvirt,typeof(LuaReference).GetMethod("Call",new Type[] {
				typeof(LuaState),typeof(int),typeof(int)}));
			il.Emit(OpCodes.Pop);
		}

		public void CompileA(ILGenerator il,int RA,string op) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod(op));
		}

		public void CompileAB(ILGenerator il,int RA,int RB,string op) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Ldc_I4,RB);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod(op));
		}

		public void CompileReturn(ILGenerator il,int RA,int RB,Label leave,LocalBuilder retVal) 
		{
			Label posCall=il.DefineLabel();
			if(RB!=0) 
			{
				il.Emit(OpCodes.Ldloc_0);
				il.Emit(OpCodes.Ldloc_1);
				il.Emit(OpCodes.Ldc_I4,RA+RB-1);
				il.Emit(OpCodes.Add);
				il.Emit(OpCodes.Stfld,typeof(LuaStack).GetField("Top"));
			}
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldfld,typeof(LuaState).GetField("OpenUpvalues"));
			il.Emit(OpCodes.Brfalse,posCall);
			CompileCloseUpvals(il);
			il.MarkLabel(posCall);
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldarg_2);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldloc_2);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("PosCall"));
			il.Emit(OpCodes.Ldc_I4,RB-1);
			il.Emit(OpCodes.Stloc,retVal);
			il.Emit(OpCodes.Leave,leave);
			il.Emit(OpCodes.Nop);
		}

		public void CompileABC(ILGenerator il,int RA,int RB,int RC,string op) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Ldc_I4,RB);
			il.Emit(OpCodes.Ldc_I4,RC);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod(op));
		}

		public void CompileOpTest(ILGenerator il,int pc,int RA,int RB,int RC,Hashtable labels) 
		{
			Label jump1=il.DefineLabel();
			Label jump2=il.DefineLabel();
			Label skip1=il.DefineLabel();
			Label skip2=il.DefineLabel();
			if(RC==0) 
			{
				CompileLoadStack(il);
				il.Emit(OpCodes.Ldloc_1);
				il.Emit(OpCodes.Ldc_I4,RB);
				il.Emit(OpCodes.Add);
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
				il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));
				il.Emit(OpCodes.Dup);
				il.Emit(OpCodes.Ldsfld,typeof(NilClass).GetField("Instance"));
				il.Emit(OpCodes.Beq,jump1);
				il.Emit(OpCodes.Ldsfld,typeof(FalseClass).GetField("Instance"));
				il.Emit(OpCodes.Beq,jump2);
				CompileSkip(il,pc+1,labels);
			} 
			else 
			{
				CompileLoadStack(il);
				il.Emit(OpCodes.Ldloc_1);
				il.Emit(OpCodes.Ldc_I4,RB);
				il.Emit(OpCodes.Add);
				il.Emit(OpCodes.Ldelema,typeof(LuaValue));
				il.Emit(OpCodes.Ldfld,typeof(LuaValue).GetField("O"));
				il.Emit(OpCodes.Dup);
				il.Emit(OpCodes.Ldsfld,typeof(NilClass).GetField("Instance"));
				il.Emit(OpCodes.Beq,skip1);
				il.Emit(OpCodes.Ldsfld,typeof(FalseClass).GetField("Instance"));
				il.Emit(OpCodes.Beq,skip2);
				il.Emit(OpCodes.Br,jump2);
				il.MarkLabel(skip1);
				il.Emit(OpCodes.Pop);
				il.MarkLabel(skip2);
				CompileSkip(il,pc+1,labels);
			}
			il.MarkLabel(jump1);
			il.Emit(OpCodes.Pop);
			il.MarkLabel(jump2);
			CompileOpMove(il,RA,RB);
		}

		public void CompileSkip(ILGenerator il,int dest,Hashtable labels) 
		{
			Label lab;
			if(labels[dest]==null) 
			{
				lab=il.DefineLabel();
				labels[dest]=lab;
			} 
			else lab=(Label)labels[dest];
			il.Emit(OpCodes.Br,lab);
		}
	
		public void CompileABCSkip(ILGenerator il,int pc,int RA,int RB,int RC,string op,Hashtable labels) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Ldc_I4,RB);
			il.Emit(OpCodes.Ldc_I4,RC);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod(op));
			Label lab;
			if(labels[pc+1]==null) 
			{
				lab=il.DefineLabel();
				labels[pc+1]=lab;
			} else lab=(Label)labels[pc+1];
			il.Emit(OpCodes.Brtrue,lab);
		}

		public void CompileForLoop(ILGenerator il,int pc,int RA,int RBx,Hashtable labels) 
		{
			LocalBuilder step=il.DeclareLocal(typeof(double));
			LocalBuilder idx=il.DeclareLocal(typeof(double));
			LocalBuilder limit=il.DeclareLocal(typeof(double));

			Label test = il.DefineLabel();

            il.Emit(OpCodes.Ldloc,step);
			il.Emit(OpCodes.Ldc_R8,(double)0);
			il.Emit(OpCodes.Bne_Un,test);

			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RA+1);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Call,typeof(LuaValue).GetMethod("ToNumber"));
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Stloc,limit);
			il.Emit(OpCodes.Ldc_R8,(double)double.NaN);
			Label lab2=il.DefineLabel();
			il.Emit(OpCodes.Beq,lab2);

			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RA+2);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Call,typeof(LuaValue).GetMethod("ToNumber"));
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Stloc,step);
			il.Emit(OpCodes.Ldc_R8,(double)double.NaN);
			Label lab3=il.DefineLabel();
			il.Emit(OpCodes.Beq,lab3);

			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Call,typeof(LuaValue).GetMethod("ToNumber"));
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Stloc,idx);
			il.Emit(OpCodes.Ldc_R8,(double)double.NaN);
			Label lab1=il.DefineLabel();
			il.Emit(OpCodes.Beq,lab1);

			il.MarkLabel(test);
			CompileLoadStack(il);
			il.Emit(OpCodes.Ldloc_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Ldelema,typeof(LuaValue));
			il.Emit(OpCodes.Ldloc,idx);
			il.Emit(OpCodes.Ldloc,step);
			il.Emit(OpCodes.Add);
			il.Emit(OpCodes.Dup);
			il.Emit(OpCodes.Stloc,idx);
			il.Emit(OpCodes.Stfld,typeof(LuaValue).GetField("N"));
			Label lab=(Label)labels[pc+RBx];

			Label lab4=il.DefineLabel();
			Label jump=il.DefineLabel();
			Label end=il.DefineLabel();
			il.Emit(OpCodes.Ldloc,idx);
			il.Emit(OpCodes.Ldloc,step);
			il.Emit(OpCodes.Ldc_R8,(double)0);
			il.Emit(OpCodes.Bgt,lab4);
			il.Emit(OpCodes.Ldloc,limit);
			il.Emit(OpCodes.Bge,lab);
			il.Emit(OpCodes.Br,end);
			il.MarkLabel(lab4);
			il.Emit(OpCodes.Ldloc,limit);
			il.Emit(OpCodes.Ble,lab);
			il.Emit(OpCodes.Br,end);

			/*il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("OpForLoop"));
			Label lab=(Label)labels[pc+RBx];
			il.Emit(OpCodes.Brtrue,lab);*/

			il.MarkLabel(lab1);
			il.Emit(OpCodes.Ldstr,"'for' initial value must be a number");
			il.Emit(OpCodes.Newobj,
				typeof(RuntimeException).GetConstructor(new Type[] {typeof(string)}));
			il.Emit(OpCodes.Throw);
			il.MarkLabel(lab2);
			il.Emit(OpCodes.Ldstr,"'for' limit value must be a number");
			il.Emit(OpCodes.Newobj,
				typeof(RuntimeException).GetConstructor(new Type[] {typeof(string)}));
			il.Emit(OpCodes.Throw);
			il.MarkLabel(lab3);
			il.Emit(OpCodes.Ldstr,"'for' step value must be a number");
			il.Emit(OpCodes.Newobj,
				typeof(RuntimeException).GetConstructor(new Type[] {typeof(string)}));
			il.Emit(OpCodes.Throw);

			il.MarkLabel(end);
			il.Emit(OpCodes.Ldc_R8,(double)0);
			il.Emit(OpCodes.Stloc,step);
		}

		public void CompileTForLoop(ILGenerator il,int pc,int RA,int RBx,int RC,Hashtable labels) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Ldc_I4,RC);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("OpTForLoop"));
			Label lab;
			if(labels[pc+1]==null) 
			{
				lab=il.DefineLabel();
				labels[pc+1]=lab;
			} 
			else lab=(Label)labels[pc+1];
			il.Emit(OpCodes.Brtrue,lab);
		}

		public void CompileTForPrep(ILGenerator il,int pc,int RA,int RBx,Hashtable labels) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("OpTForPrep"));
			Label lab=(Label)labels[pc+RBx];
			il.Emit(OpCodes.Br,lab);
		}

		public void CompileABx(ILGenerator il,int RA,int RBx,string op) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Ldc_I4,RBx);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod(op));
		}

		public void CompileSetList(ILGenerator il,int RA,int RBx) 
		{
			int n=(RBx&(LuaState.LFIELDS_PER_FLUSH-1))+1;
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Ldc_I4,RBx);
			il.Emit(OpCodes.Ldc_I4,n);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("OpSetList"));
		}

		public void CompileSetListO(ILGenerator il,int RA,int RBx) 
		{
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Ldc_I4,RBx);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldfld,typeof(LuaState).GetField("Stack"));
			il.Emit(OpCodes.Ldfld,typeof(LuaStack).GetField("Top"));
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldfld,typeof(LuaState).GetField("Stack"));
			il.Emit(OpCodes.Ldfld,typeof(LuaStack).GetField("Base"));
			il.Emit(OpCodes.Sub);
			il.Emit(OpCodes.Ldc_I4,RA+1);
			il.Emit(OpCodes.Sub);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldfld,typeof(LuaState).GetField("Stack"));
			il.Emit(OpCodes.Ldarg_2);
			il.Emit(OpCodes.Ldfld,typeof(CallInfo).GetField("MaxTop"));
			il.Emit(OpCodes.Stfld,typeof(LuaStack).GetField("Top"));
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("OpSetList"));
		}

		public int CompileOpClosure(ILGenerator il,Prototype proto,FieldBuilder[] upvals,int RA,int pc,Type callee)
		{
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldfld,typeof(LuaState).GetField("Globals"));

            //Type t = Type.GetType("Lua2IL.LuaReference");
            //ConstructorInfo ci = callee.GetConstructor(new Type[] { t });
            ConstructorInfo ci = callee.GetConstructor(new Type[] { typeof(LuaReference) });
            try
            {
                il.Emit(OpCodes.Newobj, ci);
            }
            catch( Exception)
            {
                    System.Diagnostics.Debugger.Launch();
            }

			LocalBuilder local=il.DeclareLocal(callee);
			il.Emit(OpCodes.Stloc,local);
			int nups=proto.Functions[proto.Code[pc].ArgBx].NUpvalues;
			for(int i=0;i<nups;i++,pc++) 
			{
				if(proto.Code[pc+1].OpCode==LuaOpCode.OP_GETUPVAL) 
				{
					il.Emit(OpCodes.Ldloc,local);
					il.Emit(OpCodes.Ldarg_0);
					il.Emit(OpCodes.Ldfld,upvals[proto.Code[pc+1].ArgB]);
					il.Emit(OpCodes.Stfld,callee.GetField("upvalue"+i));
				} 
				else 
				{
					il.Emit(OpCodes.Ldloc,local);
					il.Emit(OpCodes.Ldarg_1);
					il.Emit(OpCodes.Ldc_I4,proto.Code[pc+1].ArgB);
					il.Emit(OpCodes.Call,typeof(LuaState).GetMethod("FindUpVal"));
					il.Emit(OpCodes.Stfld,callee.GetField("upvalue"+i));
				}
			}
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Ldc_I4,RA);
			il.Emit(OpCodes.Ldloc,local);
			il.Emit(OpCodes.Call,typeof(LuaClosure).GetMethod("OpMoveClosure"));
			return pc;
		}

		public ConstructorInfo CompileConstructor(FieldInfo compiledProto,TypeBuilder type) 
		{
			ConstructorBuilder cons=type.DefineConstructor(MethodAttributes.Public |
				MethodAttributes.HideBySig,CallingConventions.Standard,new Type[] { typeof(LuaReference) });
			ILGenerator il=cons.GetILGenerator();
			il.DeclareLocal(typeof(CompiledPrototype));
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_1);
			il.Emit(OpCodes.Call,type.BaseType.GetConstructor(new Type[] { typeof(LuaReference) }));
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldsfld,compiledProto);
			il.Emit(OpCodes.Stfld,type.BaseType.GetField("proto"));
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldarg_0);
			il.Emit(OpCodes.Ldfld,type.BaseType.GetField("proto"));
			il.Emit(OpCodes.Ldfld,typeof(CompiledPrototype).GetField("NUpvalues"));
			il.Emit(OpCodes.Newarr,typeof(UpValue));
			il.Emit(OpCodes.Stfld,type.BaseType.GetField("UpVals"));
			il.Emit(OpCodes.Ret);
			return cons;
		}

		public string FunctionName(string source,string name) 
		{
			int nameLength=source.IndexOf(".lua")-1;
			if(nameLength<=0) nameLength=source.Length-1;
			string nameSpace=source.Substring(1,nameLength);
			if(name==null) 
			{
				functionNumber++;
				return nameSpace+".Function"+functionNumber;
			}
			else
				return nameSpace+"."+name;
		}

		public Type[] CompileSubFunctions(Prototype proto,ModuleBuilder module,ISymbolDocumentWriter sourceDoc) 
		{
			Type[] subFunctions=new Type[proto.Functions.Length];
			for(int i=0;i<proto.Code.Length-1;i++) 
			{
				if(proto.Code[i].OpCode==LuaOpCode.OP_CLOSURE) 
				{
					int func=proto.Code[i].ArgBx;
					int name=proto.Code[i+1].ArgBx;
					if(proto.Code[i+1].OpCode==LuaOpCode.OP_SETGLOBAL) 
						subFunctions[func]=CompileFunction(proto.Functions[func],proto.Constants[name].ToString(),module,sourceDoc);
					else
						subFunctions[func]=CompileFunction(proto.Functions[func],null,module,sourceDoc);
				}
			}
			return subFunctions;
		}

		public static void CompileFile(string name) 
		{
			Stream src=new FileStream(name+".luac",FileMode.Open);
			Loader loader=new Loader(name+".luac",src);
            XNuaCompiler compiler = new XNuaCompiler();
			Prototype proto=loader.LoadChunk();
			AssemblyName assemblyName=new AssemblyName();
			assemblyName.Name=name;
			AssemblyBuilder assembly=AppDomain.CurrentDomain.DefineDynamicAssembly(assemblyName,AssemblyBuilderAccess.Save);
			assembly.SetCustomAttribute(new CustomAttributeBuilder(
				typeof(DebuggableAttribute).GetConstructor(new Type[] {typeof(bool),typeof(bool)}),
				new Object[] {false,false}));
			ModuleBuilder module=assembly.DefineDynamicModule(name,name+".dll",true);
            ISymbolDocumentWriter sourceDoc = module.DefineDocument(name + ".lua", XNuaCompiler.LuaGUID,
                XNuaCompiler.PucGUID, SymDocumentType.Text);
			compiler.CompileFunction(proto,"MainFunction",module,sourceDoc);
			assembly.Save(name+".dll");
			src.Close();
		}

        public static void ProduceAssembly(Stream inLuaByteCode, string path, string name, string ext)
        {
            Loader loader = new Loader(name, inLuaByteCode);
            XNuaCompiler compiler = new XNuaCompiler();
            Prototype proto = loader.LoadChunk();
            AssemblyName assemblyName = new AssemblyName();
            assemblyName.Name = name;
            AssemblyBuilder assembly = AppDomain.CurrentDomain.DefineDynamicAssembly(assemblyName,
                                        AssemblyBuilderAccess.Save, path);
            assembly.SetCustomAttribute(new CustomAttributeBuilder(
                typeof(DebuggableAttribute).GetConstructor(new Type[] { typeof(bool), typeof(bool) }),
                new Object[] { false, false }));
            ModuleBuilder module = assembly.DefineDynamicModule(name, name + ext, true);
            ISymbolDocumentWriter sourceDoc = module.DefineDocument(name + ".lua", XNuaCompiler.LuaGUID,
                XNuaCompiler.PucGUID, SymDocumentType.Text);
            compiler.CompileFunction(proto, "MainFunction", module, sourceDoc);
            assembly.Save(name + ext);
        }
	}

}