/*
 * XNuaCompiler 0.1 Massively based on Lua2IL
 * Lua2IL 0.5.0
 * LuaLoad.cs - Loader for Lua compiled chunks
 * Copyright 2003-2005 Fabio Mascarenhas
 *                2007 Dean Calver
 * 
 */


using System;
using System.IO;
using System.Text;

namespace XNua 
{
	public struct LocalVar 
	{
		public string Name;
		public int StartPC,EndPC;

		public LocalVar(string name,int startPC,int endPC) 
		{
			this.Name=name;
			this.StartPC=startPC;
			this.EndPC=endPC;
		}
	}

	public struct Instruction 
	{
		uint inst;

		public Instruction(uint inst) 
		{
			this.inst=inst;
		}

		public LuaOpCode OpCode 
		{
			get 
			{
				return (LuaOpCode)(inst&OpSizes.MASK_OP);
			}
		}
		public int ArgA 
		{
			get 
			{
				return (int)inst>>OpSizes.POS_A;
			}
		}
		public int ArgB
		{
			get
			{
				return (int)(inst>>OpSizes.POS_B)&OpSizes.MASK_B;
			}
		}
		public int ArgC
		{
			get 
			{
				return (int)(inst>>OpSizes.POS_C)&OpSizes.MASK_C;
			}
		}
		public int ArgBx
		{
			get 
			{
				return (int)(inst>>OpSizes.POS_Bx)&OpSizes.MASK_Bx;
			}
		}
		public int ArgsBx
		{
			get 
			{
				return (int)ArgBx-(((1<<OpSizes.SIZE_Bx)-1)>>1);
			}
		}
	}

	public class Loader 
	{
		public string Name;
		Stream src;

		public static byte[] LUA_SIGNATURE=new byte[] { 27,(byte)'L',(byte)'u',(byte)'a' };
		public static byte VERSION_FORMAT=0x50;
		public static byte VERSION_MAJOR=0x50;
		public static double TEST_NUMBER=(double)3.14159265358979323846E7;

		public Loader(string name, Stream src) 
		{
			this.Name=name;
			this.src=src;
		}

		public Loader(Stream src) : this(null,src)
		{
		}

		public string LoadString() 
		{
			int size=LoadInt();
			if(size==0)
				return null;
			else 
			{
				byte[] buf=LoadBlock(size);
				StringBuilder str=new StringBuilder(size-1);
				for(int i=0;i<buf.Length-1;i++)
					str.Append(Convert.ToChar(buf[i]));
				return str.ToString();
			}
		}

		public int LoadInt() 
		{
			int val=0;
			for(int i=0;i<4;i++) 
			{
				val=val|(LoadByte()<<(i*8));
			}
			if(val<0) LoadError("bad integer in "+Name);
			return val;
		}

		public double LoadNumber() 
		{
			byte[] buf=new byte[8];
			for(int i=0;i<8;i++) 
			{
				buf[i]=LoadByte();
			}
			return new BinaryReader(new MemoryStream(buf)).ReadDouble();
		}

		public byte LoadByte() 
		{
			int val=src.ReadByte();
			if(val==-1) LoadError("unexpected EOF");
			return (byte)val;
		}

		public byte[] LoadBlock(int size) 
		{
			byte[] buf=new byte[size];
			if(src.Read(buf,0,size)!=size) LoadError("unexpected EOF");
			return buf;
		}

		public void LoadHeader()
		{
			LoadSignature();
			byte version=LoadByte();
			if(version>Loader.VERSION_FORMAT)
				LoadError(Name+" is too new: read version "+version/16+"."+version%16+
					      "; expected at most "+Loader.VERSION_FORMAT/16+"."+Loader.VERSION_FORMAT%16);
			if(version<Loader.VERSION_MAJOR)
				LoadError(Name+" is too old: read version "+version/16+"."+version%16+
					"; expected at most "+Loader.VERSION_MAJOR/16+"."+Loader.VERSION_MAJOR%16);
			LoadByte();
			TestSize(4,"int");
			TestSize(4,"size_t");
			TestSize(4,"Instruction");
			TestSize(OpSizes.SIZE_OP,"OP");
			TestSize(OpSizes.SIZE_A,"A");
			TestSize(OpSizes.SIZE_B,"B");
			TestSize(OpSizes.SIZE_C,"C");
			TestSize(8,"number");
			double x=LoadNumber();
			if((long)x!=(long)Loader.TEST_NUMBER)
				LoadError("unknown number format in "+Name);
		}

        public void LoadSignature()
		{
			for(int i=0;i<Loader.LUA_SIGNATURE.Length;i++)
				if(Loader.LUA_SIGNATURE[i]!=LoadByte())
					LoadError("bad signature in "+Name);
		}

		public Prototype LoadChunk() 
		{
			LoadHeader();
			return Prototype.Load(this);
		}

		public void LoadError(string err) 
		{
			throw new LoadException(err);
		}

		void TestSize(byte size, string what) 
		{
			byte num=LoadByte();
			if(num!=size)
				LoadError("virtual machine mismatch in "+Name+": size of "+what+" is "+size+" but read "+num);
		}
	}

	public class Prototype 
	{
		public LuaValue[] Constants;
		public Instruction[] Code;
		public Prototype[] Functions;
		public int[] LineInfo;
		public LocalVar[] Locals;
		public string[] Upvalues;
		public string Source;
		public int LineDefined;
		public byte NUpvalues;
		public byte NParams;
		public bool IsVararg;
		public byte MaxStack;

		public static Prototype Load(Loader loader) 
		{
			return Load(loader,null);
		}

		public static Prototype Load(Loader loader, string source) 
		{
			Prototype f=new Prototype();
			f.LoadHeader(loader,source);
			f.LoadLines(loader);
			f.LoadLocals(loader);
			f.LoadUpvalues(loader);
			f.LoadConstants(loader);
			f.LoadFunctions(loader);
			f.LoadCode(loader);
			return f;
		}

		public void LoadHeader(Loader loader,string source) 
		{
			Source=loader.LoadString();
			if(Source==null) Source=source;
			LineDefined=loader.LoadInt();
			NUpvalues=loader.LoadByte();
			NParams=loader.LoadByte();
			IsVararg=(loader.LoadByte()!=0);
			MaxStack=loader.LoadByte();
		}

		public void LoadLines(Loader loader)
		{
			int nLines=loader.LoadInt();
			LineInfo=new int[nLines];
			for(int i=0;i<nLines;i++)
				LineInfo[i]=loader.LoadInt();
		}

		public void LoadLocals(Loader loader)
		{
			int nLocals=loader.LoadInt();
			Locals=new LocalVar[nLocals];
			for(int i=0;i<nLocals;i++)
				Locals[i]=new LocalVar(loader.LoadString(),
									   loader.LoadInt(),
									   loader.LoadInt());
		}

		public void LoadUpvalues(Loader loader)
		{
			int nUps=loader.LoadInt();
			if(nUps!=0 && nUps!=NUpvalues)
				loader.LoadError("bad nupvalues in "+loader.Name+
								 ": read "+nUps+"; expected "+NUpvalues);
			Upvalues=new string[nUps];
			for(int i=0;i<nUps;i++) Upvalues[i]=loader.LoadString();
		}

		public void LoadConstants(Loader loader) 
		{
			int nConst=loader.LoadInt();
			Constants=new LuaValue[nConst];
			for(int i=0;i<nConst;i++) 
			{
				LuaTypes type=(LuaTypes)loader.LoadByte();
				switch(type) 
				{
					case LuaTypes.LUA_TNUMBER:
						Constants[i]=new LuaValue(loader.LoadNumber());
						break;
					case LuaTypes.LUA_TSTRING:
						Constants[i]=new LuaValue(loader.LoadString());
						break;
					case LuaTypes.LUA_TNIL:
						Constants[i]=new LuaValue(NilClass.Instance);
						break;
					default:
						loader.LoadError("bad constant type ("+type+") in "+loader.Name);
						break;
				}
			}
		}

		void LoadFunctions(Loader loader) 
		{
			int nFuncs=loader.LoadInt();
			Functions=new Prototype[nFuncs];
			for(int i=0;i<nFuncs;i++) 
				Functions[i]=Prototype.Load(loader,Source);
		}

		public void LoadCode(Loader loader)
		{
			int nOps=loader.LoadInt();
			Code=new Instruction[nOps];
			for(int i=0;i<nOps;i++) 
				Code[i]=new Instruction((uint)loader.LoadInt());
		}
	}
}
