// This is the main DLL file.

#include "stdafx.h"

#include "lua.h"
#include "lauxlib.h"
#include "lfunc.h"
#include "lmem.h"
#include "lobject.h"
#include "lopcodes.h"
#include "lstring.h"
#include "lundump.h"

#include "XNuaLuaParser.h"

using namespace System;
using namespace System::IO;
using namespace System::Runtime::InteropServices;

#define __GCHANDLE_TO_VOIDPTR(x) ((GCHandle::operator System::IntPtr(x)).ToPointer())
#define __VOIDPTR_TO_GCHANDLE(x) (GCHandle::operator GCHandle(System::IntPtr(x)))

namespace XNua
{
	namespace LuaParser 
	{
		/// used but the reader as a proxy file handle for a member thingy
		struct LoadFx
		{
			char* ptr;
			int sizeofptr;
			bool done;
		};

		// this is a helper callback function, that lets the unmanaged Lua parser
		// read from a chunk of managed memory safely
		static const char *reader (lua_State *L, void *u, size_t *size) 
		{
			LoadFx *lf = (LoadFx *)u;
			(void)L;
			if( lf->done == false )
			{
			  lf->done = true;
			  *size = lf->sizeofptr;
			  return (*size > 0) ? lf->ptr : NULL;
			} else
			{
			  return 0;
			}
		}
		// this is a helper callback function, that lets the unmanaged Lua binary bytecode
		// dumper output into a chunk of managed memory safely
		static int writer(lua_State* L, const void* p, size_t size, void* u)
		{
			 UNUSED(L);

			 Stream^ stream = static_cast<Stream^>(__VOIDPTR_TO_GCHANDLE(u).Target);

			 array<Byte>^ tmp = gcnew array<Byte>(size);
			 IntPtr ptr((void*)p);
			 Marshal::Copy( ptr, tmp, 0, size );
			 stream->Write( tmp, 0, size);
			 return 1;
		}

		Parser::Parser()
		{
			ownState = true;
			L = lua_open();
		}
		Parser::Parser(lua_State *l)
		{
			ownState = false;
			L = l;
		}
		Parser::~Parser()
		{
			if( ownState )
			{
				lua_close( L );
			}
		}

		ProtoWrapper^ Parser::ParseString( String^ text, String^ filename )
		{
			char* data = (char*)Marshal::StringToHGlobalAnsi(text).ToPointer();

			LoadFx lf;
			lf.ptr = data;
			lf.sizeofptr = text->Length;
			lf.done = false;

			if( filename == nullptr )
			{
				filename = gcnew String("Nothing");
			}
			// push the filename to produce source level debugging info
			char* fnptr = (char*)Marshal::StringToHGlobalAnsi(Path::GetFileName(filename)).ToPointer();
			int fnameindex = lua_gettop(L) + 1;  /* index of filename on the stack */
		    lua_pushfstring(L, "@%s", fnptr );

			// this 'loads' from memory thro Lua, parsing into a closure
			int status = lua_load(L, reader, &lf, lua_tostring(L, -1));		

			// clean filename of the lua stack
			lua_remove(L, fnameindex);	

			// lets see if it parsed okay or wether we had an error
			if( status != 0 )
			{
				// an error, so lets extract the info from Lua and throw
				// an exception

				const char* pErr = lua_tostring(L,-1);
				String^ errStr = Marshal::PtrToStringAnsi( IntPtr((char*)pErr) );

				// clean up, so we don't leak
				Marshal::FreeHGlobal( IntPtr(data) );
				Marshal::FreeHGlobal( IntPtr(fnptr) );

				// create a useful exception from the lua error
				int indexOfLineNumber = errStr->IndexOf( ':' );
				String^ source = errStr->Substring(0, indexOfLineNumber);
				int indexOfHelpText = errStr->IndexOf( ':', indexOfLineNumber+1 );
				String^ lineNum = errStr->Substring(indexOfLineNumber+1, indexOfHelpText-(indexOfLineNumber+1));
				String^ messageText = errStr->Substring(indexOfHelpText+1);

				throw gcnew ParseException( source, lineNum, messageText );
			}
			// get the closure and convert to a Proto
			const Closure* c =(const Closure*)lua_topointer(L,-1);
			Proto* f = c->l.p;

			Marshal::FreeHGlobal( IntPtr(data) );
			Marshal::FreeHGlobal( IntPtr(fnptr) );

			return gcnew ProtoWrapper(f);
		}

		// helper loads a file and call parse string on its contents
		ProtoWrapper^ Parser::ParseFile( String^ filename )
		{
			String^ filetext = File::ReadAllText( filename );
			char* filedata = (char*)Marshal::StringToHGlobalAnsi(filetext).ToPointer();
			return ParseString( filetext, filename );

		}

		void Parser::WriteByteCode(ProtoWrapper ^proto, Stream ^output)
		{
			GCHandle han = GCHandle::Alloc(output);
			luaU_dump(L,proto->Proto,writer, __GCHANDLE_TO_VOIDPTR(han) );
			han.Free();
		}

		void Parser::ByteCodeToIL(Stream ^input, System::String ^filepath)
		{
			String^ filename = Path::GetFileNameWithoutExtension(filepath);
			String^ extension = Path::GetExtension(filepath);
			String^ path = Path::GetDirectoryName(filepath);
			XNua::XNuaCompiler::ProduceAssembly( input, path, filename, extension );
		}
	}
}