package jp.ne.motoki.android.fuelchecker;

import static jp.ne.motoki.android.fuelchecker.Utils.log_d;
import static jp.ne.motoki.android.fuelchecker.Utils.toDateRangeFloat;

import java.util.Calendar;
import java.util.List;

import jp.ne.motoki.android.chartexperthelper.ChartExpertHelper;
import jp.ne.motoki.android.chartexperthelper.ChartObject;
import jp.ne.motoki.android.chartexperthelper.ChartObject.Type;
import jp.ne.motoki.android.chartexperthelper.DataObject;
import jp.ne.motoki.android.fuelchecker.DatePickerDialogPreference.OnDateChangedListener;
import jp.ne.motoki.android.fuelchecker.NumericPickerDialogPreference.OnNumericChangedListener;
import jp.ne.motoki.android.fuelchecker.TimePickerDialogPreference.OnTimeChangedListener;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.DialogInterface.OnClickListener;
import android.content.Intent;
import android.os.Bundle;
import android.preference.PreferenceActivity;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.widget.Toast;

public class FuelCheckerActivity extends PreferenceActivity {
    
    private static final String PREF_KEY_DATE = "key_date";
    private static final String PREF_KEY_TIME = "key_time";
    private static final String PREF_KEY_QUANTITY = "key_quantity";
    private static final String PREF_KEY_DISTANCE = "key_distance";
    
    private final Intent INTENT_CALL_LIST = new Intent();
    
    private final FuelData FUEL_DATA = new FuelData();
    
    private DatePickerDialogPreference datePreference = null;
    private TimePickerDialogPreference timePreference = null;
    private NumericPickerDialogPreference quantityPreference = null;
    private NumericPickerDialogPreference distancePreference = null;
    
    private final OnDateChangedListener DATE_LISTENER =
            new OnDateChangedListener() {
        @Override
        public void onDateChanged(int year, int monthOfYear, int dayOfMonth) {
            log_d(String.format(
                    "onDateChanged : year = %d, monthOfYear = %d, dayOfMonth = %d",
                    year, monthOfYear, dayOfMonth));
            FUEL_DATA.setDate(year, monthOfYear, dayOfMonth);
        }
    };
    
    private final OnTimeChangedListener TIME_LISTENER =
            new OnTimeChangedListener() {
        @Override
        public void onTimeChanged(int hour, int minute) {
            log_d(String.format(
                    "onDateChanged : hour = %d, minute = %d", hour, minute));
            FUEL_DATA.setTime(hour, minute);
        }
    };
    
    private final OnNumericChangedListener QUANTITY_LISTENER =
            new OnNumericChangedListener() {
        @Override
        public void onNumericChanged(int integralValue, int decimalValue) {
            log_d(String.format(
                    "onDateChanged : integralValue = %d, decimalValue = %d",
                    integralValue, decimalValue));
            FUEL_DATA.setQuantity(integralValue, decimalValue);
        }
    };
    
    private final OnNumericChangedListener DISTANCE_LISTENER =
            new OnNumericChangedListener() {
        @Override
        public void onNumericChanged(int integralValue, int decimalValue) {
            log_d(String.format(
                    "onDateChanged : integralValue = %d, decimalValue = %d",
                    integralValue, decimalValue));
            FUEL_DATA.setDistance(integralValue, decimalValue);
        }
    };
    
    private final OnClickListener CONFIRM_LISTENER = new OnClickListener() {
        @Override
        public void onClick(DialogInterface dialog, int which) {
            log_d("onClick : datetime = " + FUEL_DATA.getDatetime().getTime().toString());
            insertFuelData();
            Toast.makeText(
                    FuelCheckerActivity.this,
                    R.string.text_complete_registration,
                    Toast.LENGTH_SHORT).show();
        }
    };
    
    /** Called when the activity is first created. */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.input);
        addPreferencesFromResource(R.xml.preference);

        datePreference =
            (DatePickerDialogPreference) findPreference(PREF_KEY_DATE);
        timePreference =
            (TimePickerDialogPreference) findPreference(PREF_KEY_TIME);
        quantityPreference =
            (NumericPickerDialogPreference) findPreference(PREF_KEY_QUANTITY);
        distancePreference =
            (NumericPickerDialogPreference) findPreference(PREF_KEY_DISTANCE);
        
        INTENT_CALL_LIST.setClass(this, FuelListActivity.class);
    }
    
    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.menu, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        int itemId = item.getItemId();
        switch (itemId) {
            case R.id.id_graph:
                List<FuelData> dataList = FuelData.Factory.createDataListFromDB(this);
                
                if (dataList.isEmpty()) {
                    Toast.makeText(
                            this,
                            getString(R.string.message_no_data),
                            Toast.LENGTH_LONG).show();
                } else {
                    showChart(dataList);
                }
                break;
            case R.id.id_list:
                startActivity(INTENT_CALL_LIST);
                break;
            default:
                throw new AssertionError("itemId = " + itemId);
        }
        return true;
    }

    @Override
    protected void onPause() {
        super.onPause();
        datePreference.setOnDateChangedListener(null);
        timePreference.setOnTimeChangedListener(null);
        quantityPreference.setOnNumericChangedListener(null);
        distancePreference.setOnNumericChangedListener(null);
    }

    @Override
    protected void onResume() {
        super.onResume();
        datePreference.setOnDateChangedListener(DATE_LISTENER);
        timePreference.setOnTimeChangedListener(TIME_LISTENER);
        quantityPreference.setOnNumericChangedListener(QUANTITY_LISTENER);
        distancePreference.setOnNumericChangedListener(DISTANCE_LISTENER);
    }

    public void onClickButton(View view) {
        int id = view.getId();
        switch (id) {
            case R.id.id_button_register:
                if (FUEL_DATA.getQuantity() == 0) {
                    Toast.makeText(
                            this,
                            getString(R.string.message_0_quantity),
                            Toast.LENGTH_LONG).show();
                    return;
                }
                showConfirmDialog();
                break;
            case R.id.id_button_cancel:
                finish();
                break;
            default:
                throw new AssertionError("id = " + id);
        }
    }
    
    private void insertFuelData() {
        FuelDbAdapter adapter =
            new FuelDbAdapter(FuelCheckerActivity.this);
        adapter.open();
        adapter.insert(FUEL_DATA);
        adapter.close();
    }
    
    private void showConfirmDialog() {
        // TODO Ugly code.
        String messageBody = getString(R.string.text_confirm_dialog_body);
        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setTitle(getString(R.string.title_confirm_dialog));
        builder.setMessage(
                getString(R.string.text_confirm_dialog_header) + "\n" +
                String.format(
                        messageBody, getString(R.string.title_date),
                        datePreference.getDisplayText()) + "\n" +
                String.format(
                        messageBody, getString(R.string.title_time),
                        timePreference.getDisplayText()) + "\n" +
                String.format(
                        messageBody, getString(R.string.title_quantity),
                        quantityPreference.getDisplayText()) + "\n" +
                String.format(
                        messageBody, getString(R.string.title_distance),
                        distancePreference.getDisplayText()));
        builder.setPositiveButton("OK", CONFIRM_LISTENER);
        builder.setNegativeButton(
                getString(R.string.text_button_cancel), null);
        builder.create().show();
    }

    private void showChart(List<FuelData> dataList) {
        // TODO Ugly code.
        DataObject dataObject =
            new DataObject(getString(R.string.key_fuel_efficiency));
        for (FuelData fuelData : dataList) {
            addValue(dataObject, fuelData);
        }
        
        Calendar firstDatetime = dataList.get(0).getDatetime();
        Calendar lastDatetime =
            dataList.get(dataList.size() - 1).getDatetime();
        Calendar minXAxis = Utils.floorByMonth(firstDatetime);
        Calendar maxXAxis = Utils.roundByMonth(lastDatetime);
        
        Calendar temp = (Calendar) minXAxis.clone();
        while (temp.before(maxXAxis)) {
            dataObject.addXLabel(Utils.toXLabel(temp));
            temp.add(Calendar.DAY_OF_MONTH, 1);
        }
        
        dataObject.setXAxis(
                Utils.toDateRangeFloat(minXAxis),
                Utils.toDateRangeFloat(maxXAxis),
                1);
        
        ChartObject chartObject = new ChartObject("FuelChart", Type.LINE);
        chartObject.setTitle(getString(R.string.title_chart));
        chartObject.addData(dataObject.getKey());
        
        ChartExpertHelper helper = new ChartExpertHelper("Fuel-Checker");
        helper.addChart(chartObject);
        helper.addData(dataObject);
        helper.startChartExpert(this);
    }
    
    private static void addValue(DataObject dataObject, FuelData fuelData) {
        float datetime = toDateRangeFloat(fuelData.getDatetime());
        dataObject.addValue(datetime, fuelData.getFuelEfficiency());
    }
}