#include "global.h"
#include "preferences.h"
#include "textview.h"

#include <QApplication>
#include <QClipboard>
#include <QDebug>
#include <QPainter>
#include <QPaintEvent>
#include <QScrollArea>
#include <QTextCodec>

const int GAP_LINE_TEXT = 1;
const int MARGIN_RIGHT = 1;

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::TextView
/// \param parent   親ウィジェット
///
/// コンストラクタ
///
TextView::TextView(QScrollArea *parent) :
    QWidget(parent)
{
    m_scrollArea = parent;
    m_scrollArea->setWidget(this);
    setObjectName("textView");
    setCursor(Qt::IBeamCursor);

    resetSelection(0);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::setData
/// \param data バイト列
///
/// データを設定します。
///
void TextView::setData(const QByteArray &data)
{
    m_source = data;

    std::string code = detectCode(data.left(1024));
    convertFrom(code.c_str());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::adjust
///
/// 表示領域を調整します。
///
void TextView::adjust()
{
    // 行数をカウントする
    int lineNum = 1;
    int offset = 0;
    for (; offset < m_data.size(); ++offset) {
        if (m_data[offset] == '\r') {
            if (offset + 1 < m_data.size() && m_data[offset + 1] == '\n') {
                ++offset;
            }
            ++lineNum;
        }
        else if (m_data[offset] == '\n') {
            ++lineNum;
        }
    }

    // 行番号表示に必要な幅を取得する
    int lineNumWidth = m_charWidth * lineNumChars(lineNum);
    // テキストの表示開始位置
    int xPosText = lineNumWidth + GAP_LINE_TEXT * m_charWidth;

    setMinimumWidth(xPosText + m_charWidth * 4);

    // 文字の表示位置を取得する
    m_viewPositions.clear();
    int x = 0;
    int y = m_charHeight;
    int lineWidth = 0;
    int wrapLine = 0;
    const int lineMaxWidth = width() - xPosText - MARGIN_RIGHT * m_charWidth;
    for (lineNum = 1, offset = 0; offset < m_data.size(); ++offset) {
        bool isEOL = false;
        if (m_data[offset] == '\r') {
            isEOL = true;
            if (offset < m_data.size() - 1 && m_data[offset + 1] == '\n') {
                ++offset;
            }
        }
        if (m_data[offset] == '\n') {
            isEOL = true;
        }

        ViewPosition vPos;
        vPos.lineNum = lineNum;
        vPos.offset = offset;

        int charWidth = fontMetrics().width(m_data.mid(offset, 1));
        int tabIndent = 0;
        if (isEOL) {
            charWidth = fontMetrics().width(0x21B5);
        }
        else if (m_data[offset] == '\t') {
            tabIndent = ((lineWidth + m_tabWidth) / m_tabWidth) * m_tabWidth - lineWidth;
            charWidth = fontMetrics().width("^");
        }

        if (x + charWidth >= lineMaxWidth) {
            x = 0;
            ++wrapLine;
        }

        vPos.x = xPosText + x;
        vPos.y = y + wrapLine * m_charHeight;
        m_viewPositions << vPos;

        if (isEOL) {
            lineWidth = 0;
            x = 0;
            y += (wrapLine + 1) * m_charHeight;
            wrapLine = 0;
            ++lineNum;
        }
        else {
            if (m_data[offset] == '\t') {
                x += tabIndent;
                if (x > lineMaxWidth) {
                    x = x - lineMaxWidth;
                    ++wrapLine;
                }
            }
            else {
                x += charWidth;
            }
            lineWidth += charWidth + tabIndent;
        }
    }

    if (!m_viewPositions.isEmpty()) {
        setMinimumHeight(m_viewPositions.last().y + m_charHeight);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::convertFrom
/// \param code 文字コード
///
/// 文字コードを変更します。
///
void TextView::convertFrom(const char *code)
{
    QTextCodec *codec = QTextCodec::codecForName(code);
    m_data = codec->toUnicode(m_source);
    adjust();
    resetSelection(0);
    update();

    emit statusChanged(code);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::cursorPos
/// \param pos  カーソル位置
/// \return カーソル位置に該当するViewPositionのインデックスを返します。
///
int TextView::cursorPos(const QPoint &pos)
{
    int n;
    int y;
    for (n = 0; n < m_viewPositions.size(); n++) {
        if (pos.y() < m_viewPositions[n].y) {
            y = m_viewPositions[n].y;
            break;
        }
    }

    for(; n < m_viewPositions.size(); n++) {
        if (pos.x() < m_viewPositions[n].x) {
            return n - 1;
        }
        if (y < m_viewPositions[n].y) {
            return n;
        }
    }

    return n;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::lineNumChars
/// \param lines    行数
/// \return 行番号の表示に必要な桁数を返します。
///
int TextView::lineNumChars(int lines) const
{
    if (lines == -1) {
        if (m_viewPositions.isEmpty()) {
            return 0;
        }
        lines = m_viewPositions.last().lineNum;
    }
    return QString("%1").arg(lines).size() + 1;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::resetSelection
/// \param index    ViewPositionのインデックス
///
/// 選択領域を初期化します。
///
void TextView::resetSelection(int index)
{
    m_selectionInit = index;
    m_selectionBegin = index;
    m_selectionEnd = index;

    emit copyAvailable(false);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::setSelection
/// \param index    ViewPositionのインデックス
///
/// 選択領域を設定します。
///
void TextView::setSelection(int index)
{
    if (index > m_selectionInit) {
        m_selectionBegin = m_selectionInit;
        m_selectionEnd = index;
    }
    else {
        m_selectionBegin = index;
        m_selectionEnd = m_selectionInit;
    }

    emit copyAvailable(m_selectionBegin != m_selectionEnd);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::onConvertFromEUC
///
/// EUC-JPからUnicodeへ変換します。
///
void TextView::onConvertFromEUC()
{
    convertFrom("EUC-JP");
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::onConvertFromJIS
///
/// ISO-2022-JP(JIS)からUnicodeへ変換します。
///
void TextView::onConvertFromJIS()
{
    convertFrom("ISO 2022-JP");
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::onConvertFromSJIS
///
/// Shift-JIS(SJIS)からUnicodeへ変換します。
///
void TextView::onConvertFromSJIS()
{
    convertFrom("Shift-JIS");
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::onConvertFromUTF8
///
/// UTF-8からUnicodeへ変換します。
///
void TextView::onConvertFromUTF8()
{
    convertFrom("UTF-8");
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::onConvertFromUTF16
///
/// UTF-16からUnicodeへ変換します。
///
void TextView::onConvertFromUTF16()
{
    convertFrom("UTF-16");
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::onConvertFromUTF16BE
///
/// UTF-16BEからUnicodeへ変換します。
///
void TextView::onConvertFromUTF16BE()
{
    convertFrom("UTF-16BE");
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::onConvertFromUTF16LE
///
/// UTF-16LEからUnicodeへ変換します。
///
void TextView::onConvertFromUTF16LE()
{
    convertFrom("UTF-16LE");
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::onCopy
///
/// 選択領域をクリップボードにコピーします。
///
void TextView::onCopy()
{
    QString selected;
    int length = m_viewPositions[m_selectionEnd].offset -
            m_viewPositions[m_selectionBegin].offset;

    selected = m_data.mid(m_viewPositions[m_selectionBegin].offset, length);

    QClipboard *clipboard = qApp->clipboard();
    clipboard->setText(selected);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::onScaleDown
///
/// 文字を小さくします。
///
void TextView::onScaleDown()
{
    Preferences prefs(this);
    QFont font = prefs.getTextViewFont();
    font.setPointSize(font.pointSize() - 1);
    prefs.setTextViewFont(font);

    setVisible(true);
    adjust();
    update();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::onScaleUp
///
/// 文字を大きくします。
///
void TextView::onScaleUp()
{
    Preferences prefs(this);
    QFont font = prefs.getTextViewFont();
    font.setPointSize(font.pointSize() + 1);
    prefs.setTextViewFont(font);

    setVisible(true);
    adjust();
    update();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::onSelectAll
///
/// 全て選択します。
///
void TextView::onSelectAll()
{
    m_selectionInit = 0;
    m_selectionBegin = 0;
    m_selectionEnd = m_viewPositions.size();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::setVisible
/// \param visible  表示(true)/非表示(false)
///
/// 表示時の処理を行います。
///
void TextView::setVisible(bool visible)
{
    if (visible) {
        Preferences prefs(this);
        QPalette pal = this->palette();
        pal.setColor(this->backgroundRole(), prefs.getTextViewBgColor());
        pal.setColor(this->foregroundRole(), prefs.getTextViewFgColor());
        pal.setBrush(QPalette::BrightText, prefs.getTextViewCtrlColor());
        this->setPalette(pal);
        this->setAutoFillBackground(true);
        this->setFont(prefs.getTextViewFont());

        m_charHeight = fontMetrics().height() * prefs.getTextViewLineHeight();
        m_charWidth = fontMetrics().width('9');
        m_tabWidth = 8 * m_charWidth;
    }

    QWidget::setVisible(visible);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::mousePressEvent
/// \param e    マウスイベントオブジェクト
///
/// マウスクリック時の処理を行います。
///
void TextView::mousePressEvent(QMouseEvent *e)
{
    if (e->button() == Qt::LeftButton) {
        int cPos = cursorPos(e->pos());
        resetSelection(cPos);

        update();
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::mouseDoubleClickEvent
/// \param e    マウスイベントオブジェクト
///
/// ダブルクリック時の処理を行います。
///
void TextView::mouseDoubleClickEvent(QMouseEvent *e)
{
    if (e->button() == Qt::LeftButton) {
        int cPos = cursorPos(e->pos());
        if (cPos >= m_viewPositions.size()) {
            cPos = m_viewPositions.size() - 1;
        }
        int y = m_viewPositions[cPos].y;

        int n;
        for (n = cPos; n >= 0; n--) {
            if (m_viewPositions[n].y != y) {
                ++n;
                break;
            }
        }
        resetSelection(n);

        for (n = cPos; n < m_viewPositions.size(); n++) {
            if (m_viewPositions[n].y != y) {
                break;
            }
        }
        setSelection(n - 1);

        update();
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::mouseMoveEvent
/// \param e    マウスイベントオブジェクト
///
/// マウス移動時の処理を行います。
///
void TextView::mouseMoveEvent(QMouseEvent *e)
{
    m_scrollArea->ensureVisible(e->x(), e->y());

    int cPos = cursorPos(e->pos());
    setSelection(cPos);
    update();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::paintEvent
/// \param e    ペイントイベントオブジェクト
///
/// 描画イベントを処理します。
///
void TextView::paintEvent(QPaintEvent *e)
{
    QPainter painter(this);

    // 行番号エリア
    QRect lineNumRect(e->rect());
    lineNumRect.setLeft(0);
    lineNumRect.setWidth(m_charWidth * lineNumChars());
    painter.fillRect(lineNumRect, Qt::gray);

    int prevLine = -1;
    int idx;
    for (idx = 0;
         idx < m_viewPositions.size() &&
            m_viewPositions[idx].y < e->rect().top();
         ++idx)
    {
        prevLine = m_viewPositions[idx].lineNum;
    }

    painter.setPen(this->palette().color(this->foregroundRole()));
    for (;
         idx < m_viewPositions.size() &&
            m_viewPositions[idx].y <= e->rect().bottom() + m_charHeight;
         ++idx)
    {
        const ViewPosition &vPos = m_viewPositions[idx];

        // 行番号
        if (vPos.lineNum != prevLine) {
            prevLine = vPos.lineNum;
            painter.setPen(Qt::black);
            painter.setBackgroundMode(Qt::TransparentMode);
            QString lineNumber = QString("%1").arg(vPos.lineNum);
            painter.drawText(
                        lineNumRect.right() - fontMetrics().width(lineNumber),
                        vPos.y,
                        lineNumber);
        }

        if (m_selectionBegin <= idx && idx < m_selectionEnd) {
            painter.setBackground(this->palette().highlight());
            painter.setBackgroundMode(Qt::OpaqueMode);
            painter.setPen(this->palette().highlightedText().color());
        }
        else {
            painter.setPen(this->palette().color(this->foregroundRole()));
            painter.setBackgroundMode(Qt::TransparentMode);
        }

        QString ch = m_data.mid(vPos.offset, 1);
        if (ch == "\r" || ch == "\n") {
            painter.setPen(this->palette().color(QPalette::BrightText));
            painter.drawText(vPos.x, vPos.y, QChar(0x21B5));
            painter.setPen(this->palette().color(this->foregroundRole()));
        }
        else if (ch == "\t") {
//            painter.setPen(this->palette().color(QPalette::BrightText));
//            painter.drawText(vPos.x, vPos.y, "^");
//            painter.setPen(this->palette().color(this->foregroundRole()));
        }
        else {
            painter.drawText(vPos.x, vPos.y, ch);
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief TextView::resizeEvent
/// \param e    リサイズイベントオブジェクト
///
/// リサイズ時の処理を行います。
///
void TextView::resizeEvent(QResizeEvent *e)
{
    QWidget::resizeEvent(e);

    adjust();
}
