/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.log;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.log.DbChecksumException;
import com.sleepycat.je.log.LogEntryType;
import com.sleepycat.je.log.LogException;
import com.sleepycat.je.log.LogUtils;
import com.sleepycat.je.log.Provisional;
import com.sleepycat.je.log.ReplicationContext;
import com.sleepycat.je.log.entry.LogEntry;
import com.sleepycat.je.utilint.Adler32;
import com.sleepycat.je.utilint.VLSN;
import java.nio.ByteBuffer;
import java.util.zip.Checksum;

public class LogEntryHeader {
    static final int MIN_HEADER_SIZE = 14;
    public static final int MAX_HEADER_SIZE = 22;
    private static final int CHECKSUM_BYTES = 4;
    private static final int ENTRYTYPE_OFFSET = 4;
    private static final int PREV_OFFSET = 6;
    private static final int ITEMSIZE_OFFSET = 10;
    private static final int VLSN_OFFSET = 14;
    private static final byte PROVISIONAL_ALWAYS_MASK = -128;
    private static final byte IGNORE_PROVISIONAL_ALWAYS = 127;
    private static final byte PROVISIONAL_BEFORE_CKPT_END_MASK = 64;
    private static final byte IGNORE_PROVISIONAL_BEFORE_CKPT_END = -65;
    private static final byte REPLICATED_MASK = 32;
    private static final byte IGNORE_REPLICATED = -33;
    private long checksumVal;
    private byte entryType;
    private byte entryVersion;
    private long prevOffset;
    private int itemSize;
    private VLSN vlsn;
    private Provisional provisional;
    private boolean replicated;

    public LogEntryHeader(EnvironmentImpl envImpl, ByteBuffer entryBuffer, boolean anticipateChecksumErrors) throws DbChecksumException {
        this.checksumVal = LogUtils.readUnsignedInt(entryBuffer);
        this.entryType = entryBuffer.get();
        if (!LogEntryType.isValidType(this.entryType)) {
            throw new DbChecksumException(anticipateChecksumErrors ? null : envImpl, "Read invalid log entry type: " + this.entryType);
        }
        this.entryVersion = entryBuffer.get();
        this.prevOffset = LogUtils.readUnsignedInt(entryBuffer);
        this.itemSize = LogUtils.readInt(entryBuffer);
        this.provisional = (this.entryVersion & 0xFFFFFF80) != 0 ? Provisional.YES : ((this.entryVersion & 0x40) != 0 ? Provisional.BEFORE_CKPT_END : Provisional.NO);
        this.replicated = (this.entryVersion & 0x20) != 0;
        this.entryVersion = (byte)(this.entryVersion & 0x7F);
        this.entryVersion = (byte)(this.entryVersion & 0xFFFFFFBF);
        this.entryVersion = (byte)(this.entryVersion & 0xFFFFFFDF);
    }

    public LogEntryHeader(LogEntry entry, Provisional provisional, ReplicationContext repContext) {
        LogEntryType logEntryType = entry.getLogType();
        this.entryType = logEntryType.getTypeNum();
        this.entryVersion = (byte)6;
        this.itemSize = entry.getSize();
        this.provisional = provisional;
        assert (logEntryType.isReplicationPossible() || !repContext.inReplicationStream()) : logEntryType + " should never be replicated.";
        this.replicated = logEntryType.isReplicationPossible() ? repContext.inReplicationStream() : false;
    }

    public long getChecksum() {
        return this.checksumVal;
    }

    public byte getType() {
        return this.entryType;
    }

    public byte getVersion() {
        return this.entryVersion;
    }

    public long getPrevOffset() {
        return this.prevOffset;
    }

    public int getItemSize() {
        return this.itemSize;
    }

    public VLSN getVLSN() {
        return this.vlsn;
    }

    public boolean getReplicated() {
        return this.replicated;
    }

    public Provisional getProvisional() {
        return this.provisional;
    }

    public int getVariablePortionSize() {
        return 8;
    }

    public int getSize() {
        if (this.replicated) {
            return 22;
        }
        return 14;
    }

    int getSizeMinusChecksum() {
        return this.getSize() - 4;
    }

    int getInvariantSizeMinusChecksum() {
        return 10;
    }

    public void readVariablePortion(ByteBuffer entryBuffer) throws LogException {
        if (this.replicated) {
            this.vlsn = new VLSN();
            this.vlsn.readFromLog(entryBuffer, this.entryVersion);
        }
    }

    public void writeToLog(ByteBuffer entryBuffer) {
        entryBuffer.position(4);
        entryBuffer.put(this.entryType);
        byte versionFlags = this.entryVersion;
        if (this.provisional == Provisional.YES) {
            versionFlags = (byte)(versionFlags | 0xFFFFFF80);
        } else if (this.provisional == Provisional.BEFORE_CKPT_END) {
            versionFlags = (byte)(versionFlags | 0x40);
        }
        if (this.replicated) {
            versionFlags = (byte)(versionFlags | 0x20);
        }
        entryBuffer.put(versionFlags);
        entryBuffer.position(10);
        LogUtils.writeInt(entryBuffer, this.itemSize);
        if (this.replicated) {
            entryBuffer.position(entryBuffer.position() + 8);
        }
    }

    public ByteBuffer addPostMarshallingInfo(EnvironmentImpl envImpl, ByteBuffer entryBuffer, long lastOffset, ReplicationContext repContext) {
        this.prevOffset = lastOffset;
        entryBuffer.position(6);
        LogUtils.writeUnsignedInt(entryBuffer, this.prevOffset);
        if (repContext.inReplicationStream()) {
            entryBuffer.position(14);
            this.vlsn = repContext.mustGenerateVLSN() ? envImpl.getReplicator().bumpVLSN() : repContext.getClientVLSN();
            this.vlsn.writeToLog(entryBuffer);
        }
        Checksum checksum = Adler32.makeChecksum();
        checksum.update(entryBuffer.array(), entryBuffer.arrayOffset() + 4, entryBuffer.limit() - 4);
        entryBuffer.position(0);
        this.checksumVal = checksum.getValue();
        LogUtils.writeUnsignedInt(entryBuffer, this.checksumVal);
        entryBuffer.position(0);
        return entryBuffer;
    }

    public void dumpLog(StringBuffer sb, boolean verbose) {
        sb.append("<hdr ");
        this.dumpLogNoTag(sb, verbose);
        sb.append("\"/>");
    }

    void dumpLogNoTag(StringBuffer sb, boolean verbose) {
        LogEntryType lastEntryType = LogEntryType.findType(this.entryType);
        sb.append("type=\"").append(lastEntryType.toStringNoVersion()).append("/").append(this.entryVersion);
        if (this.provisional != Provisional.NO) {
            sb.append("\" prov=\"");
            sb.append((Object)this.provisional);
        }
        if (this.replicated) {
            sb.append("\" rep=\"true");
        }
        if (this.vlsn != null) {
            sb.append("\" ");
            this.vlsn.dumpLog(sb, verbose);
        } else {
            sb.append("\"");
        }
        sb.append(" prev=\"0x").append(Long.toHexString(this.prevOffset));
        if (verbose) {
            sb.append("\" size=\"").append(this.itemSize);
            sb.append("\" cksum=\"").append(this.checksumVal);
        }
    }

    void convertCommitToAbort(ByteBuffer entryBuffer) {
        assert (this.entryType == LogEntryType.LOG_TXN_COMMIT.getTypeNum());
        int itemStart = entryBuffer.position();
        int entryTypePosition = itemStart - (this.getSize() - 4);
        entryBuffer.position(entryTypePosition);
        entryBuffer.put(LogEntryType.LOG_TXN_ABORT.getTypeNum());
        Checksum checksum = Adler32.makeChecksum();
        int checksumSize = this.itemSize + (this.getSize() - 4);
        checksum.update(entryBuffer.array(), entryTypePosition + entryBuffer.arrayOffset(), checksumSize);
        entryBuffer.position(itemStart - this.getSize());
        this.checksumVal = checksum.getValue();
        LogUtils.writeUnsignedInt(entryBuffer, this.checksumVal);
    }

    public String toString() {
        StringBuffer sb = new StringBuffer();
        this.dumpLog(sb, true);
        return sb.toString();
    }

    public boolean logicalEquals(LogEntryHeader other) {
        return this.getType() == other.getType() && this.getVersion() == other.getVersion() && this.getVLSN().equals(other.getVLSN()) && this.getReplicated() == other.getReplicated();
    }

    public static boolean isSyncPoint(ByteBuffer buffer) throws DbChecksumException {
        buffer.mark();
        LogEntryHeader header = new LogEntryHeader(null, buffer, true);
        buffer.reset();
        return LogEntryType.isSyncPoint(header.getType());
    }

    public static VLSN getVLSN(ByteBuffer buffer) throws DatabaseException {
        buffer.mark();
        LogEntryHeader header = new LogEntryHeader(null, buffer, true);
        header.readVariablePortion(buffer);
        buffer.reset();
        return header.getVLSN();
    }
}

