/*
 * Copyright (C) 2008-2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Copyright (C) 2011 OgakiSoft
 * 
 * Added method getAllStrokes(), save_gesture(), load_gesture().
 */

package ogakisoft.android.gesture.reform;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import ogakisoft.android.util.LOG;
import ogakisoft.android.util.Vector;
import android.graphics.RectF;

/**
 * Utility Class for Gesture Sampling
 * 
 * @author The Android Open Source Project
 * @author noritoshi ogaki
 * @version 1.1
 */
public final class GestureUtilities {
	private static final float[] EMPTY_ARRAY = new float[0];
	private static final int TEMPORAL_SAMPLING_RATE = 16;
	private static final String TAG = "GestureUtilities";

	private GestureUtilities() {
	}

	static float[] spatialSampling(Gesture gesture, int sampleMatrixDimension) {
		final float targetPatchSize = sampleMatrixDimension - 1;
		// edge inclusive
		final float[] sample = new float[sampleMatrixDimension
				* sampleMatrixDimension];
		final RectF rect = gesture.getBoundingBox();
		final float sx = targetPatchSize / rect.width();
		final float sy = targetPatchSize / rect.height();
		final float scale = (sx < sy) ? sx : sy;
		final float preDx = -rect.centerX();
		final float preDy = -rect.centerY();
		final float postDx = targetPatchSize / 2;
		final float postDy = targetPatchSize / 2;
		final List<GestureStroke> strokes = gesture.getStrokes();
		final int count = strokes.size();
		int size = 0;
		float xpos = 0;
		float ypos = 0;
		GestureStroke stroke = null;
		float[] strokepoints = null;
		float[] pts = null;
		float segmentEndX = 0f;
		float segmentEndY = 0f;
		float segmentStartX = 0f;
		float segmentStartY = 0f;
		float slope = 0f;
		float invertSlope = 0f;

		Arrays.fill(sample, 0);
		for (int index = 0; index < count; index++) {
			stroke = strokes.get(index);
			strokepoints = stroke.points;
			size = strokepoints.length;
			pts = new float[size];

			for (int i = 0; i < size; i += 2) {
				pts[i] = (strokepoints[i] + preDx) * scale + postDx;
				pts[i + 1] = (strokepoints[i + 1] + preDy) * scale + postDy;
			}
			segmentEndX = -1;
			segmentEndY = -1;

			for (int i = 0; i < size; i += 2) {
				segmentStartX = (pts[i] < 0) ? 0 : pts[i];
				segmentStartY = (pts[i + 1] < 0) ? 0 : pts[i + 1];
				if (segmentStartX > targetPatchSize) {
					segmentStartX = targetPatchSize;
				}
				if (segmentStartY > targetPatchSize) {
					segmentStartY = targetPatchSize;
				}
				plot(segmentStartX, segmentStartY, sample,
						sampleMatrixDimension);
				if (segmentEndX != -1) {
					// evaluate horizontally
					if (segmentEndX > segmentStartX) {
						xpos = (float) Math.ceil(segmentStartX);
						slope = (segmentEndY - segmentStartY)
								/ (segmentEndX - segmentStartX);
						for (; xpos < segmentEndX; xpos++) {
							ypos = slope * (xpos - segmentStartX)
									+ segmentStartY;
							plot(xpos, ypos, sample, sampleMatrixDimension);
						}
					} else if (segmentEndX < segmentStartX) {
						xpos = (float) Math.ceil(segmentEndX);
						slope = (segmentEndY - segmentStartY)
								/ (segmentEndX - segmentStartX);
						for (; xpos < segmentStartX; xpos++) {
							ypos = slope * (xpos - segmentStartX)
									+ segmentStartY;
							plot(xpos, ypos, sample, sampleMatrixDimension);
						}
					}

					// evaluating vertically
					if (segmentEndY > segmentStartY) {
						ypos = (float) Math.ceil(segmentStartY);
						invertSlope = (segmentEndX - segmentStartX)
								/ (segmentEndY - segmentStartY);
						for (; ypos < segmentEndY; ypos++) {
							xpos = invertSlope * (ypos - segmentStartY)
									+ segmentStartX;
							plot(xpos, ypos, sample, sampleMatrixDimension);
						}
					} else if (segmentEndY < segmentStartY) {
						ypos = (float) Math.ceil(segmentEndY);
						invertSlope = (segmentEndX - segmentStartX)
								/ (segmentEndY - segmentStartY);
						for (; ypos < segmentStartY; ypos++) {
							xpos = invertSlope * (ypos - segmentStartY)
									+ segmentStartX;
							plot(xpos, ypos, sample, sampleMatrixDimension);
						}
					}
				}

				segmentEndX = segmentStartX;
				segmentEndY = segmentStartY;
			}
		}
		return sample;
	}

	private static void plot(float x, float y, float[] sample, int sampleSize) {
		final float xx = (x < 0) ? 0 : x;
		final float yy = (y < 0) ? 0 : y;
		final int xFloor = (int) Math.floor(xx);
		final int xCeiling = (int) Math.ceil(xx);
		final int yFloor = (int) Math.floor(yy);
		final int yCeiling = (int) Math.ceil(yy);

		// if it's an integer
		if (xx == xFloor && yy == yFloor) {
			final int index = yCeiling * sampleSize + xCeiling;
			if (sample[index] < 1) {
				sample[index] = 1;
			}
		} else {
			final double topLeft = Math.sqrt(Math.pow(xFloor - xx, 2)
					+ Math.pow(yFloor - yy, 2));
			final double topRight = Math.sqrt(Math.pow(xCeiling - xx, 2)
					+ Math.pow(yFloor - yy, 2));
			final double btmLeft = Math.sqrt(Math.pow(xFloor - xx, 2)
					+ Math.pow(yCeiling - yy, 2));
			final double btmRight = Math.sqrt(Math.pow(xCeiling - xx, 2)
					+ Math.pow(yCeiling - yy, 2));
			final double sum = topLeft + topRight + btmLeft + btmRight;

			double value = topLeft / sum;
			int index = yFloor * sampleSize + xFloor;
			if (value > sample[index]) {
				sample[index] = (float) value;
			}
			value = topRight / sum;
			index = yFloor * sampleSize + xCeiling;
			if (value > sample[index]) {
				sample[index] = (float) value;
			}
			value = btmLeft / sum;
			index = yCeiling * sampleSize + xFloor;
			if (value > sample[index]) {
				sample[index] = (float) value;
			}
			value = btmRight / sum;
			index = yCeiling * sampleSize + xCeiling;
			if (value > sample[index]) {
				sample[index] = (float) value;
			}
		}
	}

	/**
	 * Featurize a stroke into a vector of a given number of elements
	 * 
	 * @param stroke
	 * @param sampleSize
	 * @return a float array
	 */
	public static float[] temporalSampling(GestureStroke stroke, int sampleSize) {
		final float increment = stroke.length / (sampleSize - 1);
		final int vectorLength = sampleSize * 2;
		final float[] vector = new float[vectorLength];
		float distanceSoFar = 0;
		final float[] pts = stroke.points;
		float lstPointX = pts[0];
		float lstPointY = pts[1];
		int index = 0;
		float currentPointX = Float.MIN_VALUE;
		float currentPointY = Float.MIN_VALUE;
		int i = 0;
		int count = 0;
		float deltaX = 0f;
		float deltaY = 0f;
		float distance = 0f;
		float ratio = 0f;
		float nx = 0f;
		float ny = 0f;

		// ogaki-->
		if (null == stroke || stroke.points.length < 2) {
			return EMPTY_ARRAY;
		}
		count = pts.length / 2;
		// <--ogaki
		vector[index] = lstPointX;
		index++;
		vector[index] = lstPointY;
		index++;
		while (i < count) {
			// if (currentPointX == Float.MIN_VALUE) {
			if (0 == (currentPointX - Float.MIN_VALUE)) {
				i++;
				if (i >= count) {
					break;
				}
				currentPointX = pts[i * 2];
				currentPointY = pts[i * 2 + 1];
			}
			deltaX = currentPointX - lstPointX;
			deltaY = currentPointY - lstPointY;
			distance = (float) Math.sqrt(deltaX * deltaX + deltaY * deltaY);
			if (distanceSoFar + distance >= increment) {
				ratio = (increment - distanceSoFar) / distance;
				nx = lstPointX + ratio * deltaX;
				ny = lstPointY + ratio * deltaY;
				vector[index] = nx;
				index++;
				vector[index] = ny;
				index++;
				lstPointX = nx;
				lstPointY = ny;
				distanceSoFar = 0;
			} else {
				lstPointX = currentPointX;
				lstPointY = currentPointY;
				currentPointX = Float.MIN_VALUE;
				currentPointY = Float.MIN_VALUE;
				distanceSoFar += distance;
			}
		}

		for (i = index; i < vectorLength; i += 2) {
			vector[i] = lstPointX;
			vector[i + 1] = lstPointY;
		}
		return vector;
	}

	/**
	 * Calculate the centroid
	 * 
	 * @param points
	 * @return the centroid
	 */
	static float[] computeCentroid(float[] points) {
		float centerX = 0;
		float centerY = 0;
		final float[] center = new float[2];
		final int count = points.length;
		for (int i = 0; i < count; i++) {
			centerX += points[i];
			i++;
			centerY += points[i];
		}
		center[0] = 2 * centerX / count;
		center[1] = 2 * centerY / count;

		return center;
	}

	/**
	 * calculate the variance-covariance matrix, treat each point as a sample
	 * 
	 * @param points
	 * @return the covariance matrix
	 */
	private static double[][] computeCoVariance(float[] points) {
		final double[][] array = new double[2][2];
		final int count = points.length;
		float x = 0f;
		float y = 0f;
		array[0][0] = 0;
		array[0][1] = 0;
		array[1][0] = 0;
		array[1][1] = 0;
		for (int i = 0; i < count; i++) {
			x = points[i];
			i++;
			y = points[i];
			array[0][0] += x * x;
			array[0][1] += x * y;
			array[1][0] = array[0][1];
			array[1][1] += y * y;
		}
		array[0][0] /= (count / 2f);
		array[0][1] /= (count / 2f);
		array[1][0] /= (count / 2f);
		array[1][1] /= (count / 2f);

		return array;
	}

	static float computeTotalLength(float[] points) {
		float sum = 0;
		final int count = points.length - 4;
		float dx = 0f;
		float dy = 0f;
		for (int i = 0; i < count; i += 2) {
			dx = points[i + 2] - points[i];
			dy = points[i + 3] - points[i + 1];
			sum += Math.sqrt(dx * dx + dy * dy);
		}
		return sum;
	}

	static double computeStraightness(float[] points) {
		final float totalLen = computeTotalLength(points);
		final float dx = points[2] - points[0];
		final float dy = points[3] - points[1];
		return Math.sqrt(dx * dx + dy * dy) / totalLen;
	}

	static double computeStraightness(float[] points, float totalLen) {
		final float dx = points[2] - points[0];
		final float dy = points[3] - points[1];
		return Math.sqrt(dx * dx + dy * dy) / totalLen;
	}

	/**
	 * Calculate the squared Euclidean distance between two vectors
	 * 
	 * @param vector1
	 * @param vector2
	 * @return the distance
	 */
	static double squaredEuclideanDistance(float[] vector1, float[] vector2) {
		double squaredDistance = 0;
		final int size = vector1.length;
		float difference = 0f;
		for (int i = 0; i < size; i++) {
			difference = vector1[i] - vector2[i];
			squaredDistance += difference * difference;
		}
		return squaredDistance / size;
	}

	/**
	 * Calculate the cosine distance between two instances
	 * 
	 * @param vector1
	 * @param vector2
	 * @return the distance between 0 and Math.PI
	 */
	static double cosineDistance(float[] vector1, float[] vector2) {
		float sum = 0;
		final int len = vector1.length;
		for (int i = 0; i < len; i++) {
			sum += vector1[i] * vector2[i];
		}
		return Math.acos(sum);
	}

	static OrientedBoundingBox computeOrientedBoundingBox(List<GesturePoint> pts) {
		final GestureStroke stroke = new GestureStroke(pts);
		final float[] points = temporalSampling(stroke, TEMPORAL_SAMPLING_RATE);
		return computeOrientedBoundingBox(points);
	}

	static OrientedBoundingBox computeOrientedBoundingBox(float[] points) {
		final float[] meanVector = computeCentroid(points);
		return computeOrientedBoundingBox(points, meanVector);
	}

	static OrientedBoundingBox computeOrientedBoundingBox(float[] points,
			float[] centroid) {
		final double[][] array = computeCoVariance(points);
		final double[] targetVector = computeOrientation(array);
		float angle = 0f;
		float minx = Float.MAX_VALUE;
		float miny = Float.MAX_VALUE;
		float maxx = Float.MIN_VALUE;
		float maxy = Float.MIN_VALUE;
		final int count = points.length;

		translate(points, -centroid[0], -centroid[1]);
		if (0 == targetVector[0] && 0 == targetVector[1]) {
			angle = (float) -Math.PI / 2;
		} else { // -PI<alpha<PI
			angle = (float) Math.atan2(targetVector[1], targetVector[0]);
			rotate(points, -angle);
		}
		for (int i = 0; i < count; i++) {
			if (points[i] < minx) {
				minx = points[i];
			}
			if (points[i] > maxx) {
				maxx = points[i];
			}
			i++;
			if (points[i] < miny) {
				miny = points[i];
			}
			if (points[i] > maxy) {
				maxy = points[i];
			}
		}

		return new OrientedBoundingBox((float) (angle * 180 / Math.PI),
				centroid[0], centroid[1], maxx - minx, maxy - miny);
	}

	private static double[] computeOrientation(double[][] covarianceMatrix) {
		final double[] targetVector = new double[2];
		final double aa = -covarianceMatrix[0][0] - covarianceMatrix[1][1];
		final double bb = covarianceMatrix[0][0] * covarianceMatrix[1][1]
				- covarianceMatrix[0][1] * covarianceMatrix[1][0];
		double value = aa / 2;
		final double rightside = Math.sqrt(Math.pow(value, 2) - bb);
		final double lambda1 = -value + rightside;
		final double lambda2 = -value - rightside;

		if (0 == covarianceMatrix[0][1] || 0 == covarianceMatrix[1][0]) {
			targetVector[0] = 1;
			targetVector[1] = 0;
		}
		// if (lambda1 == lambda2) {
		if (0 == Math.abs(lambda1 - lambda2)) {
			targetVector[0] = 0;
			targetVector[1] = 0;
		} else {
			final double lambda = (lambda1 > lambda2) ? lambda1 : lambda2;
			targetVector[0] = 1;
			targetVector[1] = (lambda - covarianceMatrix[0][0])
					/ covarianceMatrix[0][1];
		}
		return targetVector;
	}

	static float[] rotate(float[] points, double angle) {
		final double cos = Math.cos(angle);
		final double sin = Math.sin(angle);
		final int size = points.length;
		float x = 0f;
		float y = 0f;
		for (int i = 0; i < size; i += 2) {
			x = (float) (points[i] * cos - points[i + 1] * sin);
			y = (float) (points[i] * sin + points[i + 1] * cos);
			points[i] = x;
			points[i + 1] = y;
		}
		return points;
	}

	static float[] translate(float[] points, float dx, float dy) {
		final int size = points.length;
		for (int i = 0; i < size; i += 2) {
			points[i] += dx;
			points[i + 1] += dy;
		}
		return points;
	}

	static float[] scale(float[] points, float sx, float sy) {
		final int size = points.length;
		for (int i = 0; i < size; i += 2) {
			points[i] *= sx;
			points[i + 1] *= sy;
		}
		return points;
	}

	/**
	 * Get all strokes as one stroke
	 * 
	 * @param gesture
	 * @return GestureStroke
	 */
	public static GestureStroke getAllStroke(Gesture gesture) {
		final List<GesturePoint> points = new ArrayList<GesturePoint>();
		if (null != gesture && null != gesture.getStrokes()) {
			final List<GestureStroke> strokes = gesture.getStrokes();
			final int count = strokes.size();
			int size = 0;
			GestureStroke stroke = null;
			float[] strokepoints = null;
			for (int index = 0; index < count; index++) {
				stroke = strokes.get(index);
				strokepoints = stroke.points;
				size = strokepoints.length;
				for (int i = 0; i < size; i += 2) {
					points.add(new GesturePoint(strokepoints[i],
							strokepoints[i + 1], 0));
				}
			}
		}
		return new GestureStroke(points);
	}

	public static Feature computeFeature(Gesture gesture) {
		List<GestureStroke> strokes = gesture.getStrokes();
		int count = strokes.size();
		Feature f = new Feature();
		// float angle = 0f;
		// float[] points;
		// float xdiff = 0;
		// float ydiff = 0;
		f.strokes_count = count;
		for (int i = 0; i < count; i++) {
			Vector.straightness(strokes.get(i).points);

			// angle = Vector.angle(strokes.get(i).points);
			// if (angle > 300) {
			// f.sum_circle++;
			// } else if (80 < angle && angle < 100) {
			// // f.sum_straight++;
			// points = strokes.get(i).points;
			// ydiff = points[1] - points[points.length - 1];
			// // y1 < y2
			// if (ydiff < Feature.STRAIGHT_THRESHOLD * (-1)) {
			// f.sum_vertical_down++;
			// } else if (ydiff > Feature.STRAIGHT_THRESHOLD) {
			// f.sum_vertical_up++;
			// }
			// } else if (0 < angle && angle < 30) {
			// // f.sum_straight++;
			// points = strokes.get(i).points;
			// xdiff = points[0] - points[points.length - 2];
			// // x1 < x2
			// if (xdiff < Feature.STRAIGHT_THRESHOLD * (-1)) {
			// f.sum_horizontal_right++;
			// } else if (xdiff > Feature.STRAIGHT_THRESHOLD) {
			// f.sum_horizontal_left++;
			// }
			// }
			// LOG.d(TAG, "computeFeature: angle:{0,number,#},"
			// + "v-down:{1,number,#}, v-up:{2,number,#},"
			// + "h-right:{3,number,#}, h-left:{4,number,#}", angle,
			// f.sum_vertical_down, f.sum_vertical_up,
			// f.sum_horizontal_right, f.sum_horizontal_left);
		}
		return f;
	}

	// private static float computeAngle(GestureStroke s1) {
	// int length = s1.points.length;
	// float sum_angle = 0f;
	// double angle = 0;
	// Vector a1, a2, b1, b2;
	// for (int i = 0; (i < length) && (length - i) > 6; i += 6) {
	// a1 = new Vector(s1.points[i], s1.points[i + 1]);
	// a2 = new Vector(s1.points[i + 2], s1.points[i + 3]);
	// b1 = new Vector(s1.points[i + 2], s1.points[i + 3]);
	// b2 = new Vector(s1.points[i + 4], s1.points[i + 5]);
	// angle = Vector.angle(a1, a2, b1, b2);
	// // LOG.d(TAG, "computeAngle angle:{0,number,#}", angle);
	// sum_angle += angle;
	// }
	// return sum_angle;
	// }

	public static List<GestureStroke> scaling(Gesture gesture, int width1,
			int height1, int width2, int height2) {
		LOG.d(TAG, "scaling: {0,number,#},{1,number,#}"
				+ " -> {2,number,#},{3,number,#}", width1, height1, width2,
				height2);
		final List<GestureStroke> strokes = gesture.getStrokes();
		final int count_strokes = strokes.size();
		int count_points;
		GestureStroke stroke;
		float x, y;
		RectF box = computeBoundingBox(strokes);
		LOG.d(TAG, "scaling: w={0,number,#}, h={1,number,#}", box.width(),
				box.height());
		LOG.d(TAG,
				"scaling: left={0,number,#}, right={1,number,#}, top={2,number,#}, bottom={3,number,#}",
				box.left, box.right, box.top, box.bottom);
		// scaling
		// make magnifying-power by small of width or height.
		final float scale = (width2 < height2) ? width2: height2;
		final float scaleX = scale / (box.right - box.left);
		final float scaleY = scale / (box.bottom - box.top);
//		final float scaleX = width2 / (box.right - box.left);
//		final float scaleY = height2 / (box.bottom - box.top);
		for (int i = 0; i < count_strokes; i++) {
			stroke = strokes.get(i);
			stroke.clearPath();
			count_points = stroke.points.length;
			for (int j = 0; j < count_points; j += 2) {
				x = (float) (stroke.points[j] * scaleX);
				y = (float) (stroke.points[j + 1] * scaleY);
//				if (x < 0 || y < 0 || x > width2 || y > height2) {
//					LOG.e(TAG, "scaling: ({0,number,#},{1,number,#})"
//							+ " -> ({2,number,#},{3,number,#})",
//							stroke.points[j], stroke.points[j + 1], x, y);
//				}
				stroke.points[j] = x;
				stroke.points[j + 1] = y;
			}
		}
		box = computeBoundingBox(strokes);
		LOG.d(TAG, "scaling: centering w={0,number,#}, h={1,number,#}",
				box.width(), box.height());
		LOG.d(TAG,
				"scaling: centering left={0,number,#}, right={1,number,#}, top={2,number,#}, bottom={3,number,#}",
				box.left, box.right, box.top, box.bottom);
		// centering
		float shiftX = (width2 - (box.right - box.left)) / 2;
		float shiftY = (height2 - (box.bottom - box.top)) / 2;
		shiftX = (shiftX < 0) ? 0 : shiftX;
		shiftY = (shiftY < 0) ? 0 : shiftY;
		LOG.d(TAG, "centering: shiftX={0,number,#},shiftY={1,number,#}",
				shiftX, shiftY);
		for (int i = 0; i < count_strokes; i++) {
			stroke = strokes.get(i);
			count_points = stroke.points.length;
			for (int j = 0; j < count_points; j += 2) {
				x = stroke.points[j] - shiftX;
				y = stroke.points[j + 1] - shiftY;
//				if (x < 0 || y < 0 || x > width2 || y > height2) {
//					LOG.e(TAG, "centering: ({0,number,#},{1,number,#})"
//							+ " -> ({2,number,#},{3,number,#})",
//							stroke.points[j], stroke.points[j + 1], x, y);
//				}
				stroke.points[j] = x;
				stroke.points[j + 1] = y;
			}
		}
		return strokes;
	}

	private static RectF computeBoundingBox(List<GestureStroke> strokes) {
		int count_strokes = strokes.size();
		RectF boundingBox = new RectF();
		RectF box;
		for (int i = 0; i < count_strokes; i++) {
			box = strokes.get(i).boundingBox;
			boundingBox.union(box);
		}
		return boundingBox;
	}
}
