/*
 * Copyright (C) 2008-2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Copyright (C) 2011 OgakiSoft
 * 
 * Added a duplicate key check in addGesture().
 * Added constructor argument, method hasLoaded().
 * Moved a variable definition in the method top by refactoring.
 */

package ogakisoft.android.gesture.reform;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ogakisoft.android.gestureime.NamedGesture;
import ogakisoft.android.util.LOG;
import ogakisoft.android.util.Utils;

/**
 * Maintains gesture-data
 * 
 * @author The Android Open Source Project
 * @author noritoshi ogaki
 * @version 1.0
 */

//
// File format for GestureStore:
// -----------------------------------
// Header
// 2 bytes short File format version number
// 4 bytes int Number of entries
// Entry
// X bytes UTF String Entry name
// 4 bytes int Number of gestures
// Gesture
// 8 bytes long Gesture ID
// 4 bytes int Number of strokes
// Stroke
// 4 bytes int Number of points
// Point
// 4 bytes float X coordinate of the point
// 4 bytes float Y coordinate of the point
// 8 bytes long Time stamp
//
public class GestureStore {
	private static final String TAG = "GestureStore";
	public static final int SEQUENCE_INVARIANT = 1;
	public static final int SEQUENCE_SENSITIVE = 2;
	// ORIENTATION_SENSITIVE and ORIENTATION_INVARIANT are only for
	// SEQUENCE_SENSITIVE gestures
	public static final int ORIENTATION_INVARIANT = 1;
	public static final int ORIENTATION_SENSITIVE = 2;
	private static final short FILE_FORMAT_VERSION = 1;
	private int mSequenceType = SEQUENCE_SENSITIVE;
	private int mOrientationStyle = ORIENTATION_SENSITIVE;
	private final Map<String, List<Gesture>> mNamedGestures;
	private final Learner mLearner;
	private boolean mChanged = false;
	private boolean mLoaded = false;
	private static final int IO_BUFFER_SIZE = 32 * 1024; // 32K

	// private final int mFileNo;

	public GestureStore(int fileno) {
		// mFileNo = fileno;
		mLearner = new Learner(fileno);
		mNamedGestures = Collections
				.synchronizedMap(new HashMap<String, List<Gesture>>());
	}

	public Learner getLearner() {
		return mLearner;
	}

	/**
	 * Specify how the gesture library will handle orientation. Use
	 * ORIENTATION_INVARIANT or ORIENTATION_SENSITIVE
	 * 
	 * @param style
	 */
	public void setOrientationStyle(int style) {
		mOrientationStyle = style;
	}

	public int getOrientationStyle() {
		return mOrientationStyle;
	}

	/**
	 * @param type
	 *            SEQUENCE_INVARIANT or SEQUENCE_SENSITIVE
	 */
	public void setSequenceType(int type) {
		mSequenceType = type;
	}

	/**
	 * @return SEQUENCE_INVARIANT or SEQUENCE_SENSITIVE
	 */
	public int getSequenceType() {
		return mSequenceType;
	}

	/**
	 * Get all gestures in the library
	 * 
	 * @return NamedGesture
	 */
	public Set<String> getGestureEntries() {
		return mNamedGestures.keySet();
	}

	// public Map<String, List<Gesture>> getGestureEntries() {
	// return mNamedGestures;
	// }

	/**
	 * Recognize a gesture
	 * 
	 * @param query
	 * @return a list of predictions of possible entries for a given gesture
	 */
	public List<Prediction> recognize(Gesture query, Feature feature) {
		// LOG.d(TAG, "recognize fileno={0,number,#}", mFileNo);
		if (null == query) {
			LOG.d(TAG, "recognize: query-gesture is null");
			return new ArrayList<Prediction>(0);
		}
		final Instance instance = Instance.createInstance(mSequenceType,
				mOrientationStyle, query, null, feature);
		if (null == instance) {
			LOG.d(TAG, "recognize: instance is null");
			return new ArrayList<Prediction>(0);
		}
		return mLearner.classify(mSequenceType, instance.vector, feature);
	}

	/**
	 * Add a gesture for the entry
	 * 
	 * @param label
	 * @param gesture
	 */
	public void addGesture(String label, Gesture gesture) {
		if (null == label || 0 == label.length()) {
			LOG.e(TAG, "addGesture: label is null");
			return;
		}
		final long id = gesture.getId();
		List<Gesture> gestures = mNamedGestures.get(label);
		boolean dup = false;
		if (null == gestures) {
			gestures = new ArrayList<Gesture>();
			mNamedGestures.put(label, gestures);
		} else {
			// duplicate check
			final int count = gestures.size();
			for (int i = 0; i < count && !dup; i++) {
				dup = duplicate(gestures.get(i), gesture);
			}
		}
		if (!dup) {
			// store to mNamedGestures
			gestures.add(gesture);
			LOG.d(TAG, "addGesture: label={0}, id={1,number,#}", label,
					gesture.getId());
			Feature feature = GestureUtilities.computeFeature(gesture);
			mLearner.addInstance(Instance.createInstance(mSequenceType,
					mOrientationStyle, gesture, label, feature));
			mChanged = true;
		} else {
			LOG.e(TAG, "addGesture: duplicate id={0,number,#}, label={1}", id,
					label);
		}
	}

	private boolean duplicate(Gesture g1, Gesture g2) {
		boolean result = false;
		if (g1.getId() == g2.getId()) {
			return true;
		}
		List<GestureStroke> strokes1 = g1.getStrokes();
		List<GestureStroke> strokes2 = g2.getStrokes();
		int strokes1_count = strokes1.size();
		int strokes2_count = strokes2.size();
		if (strokes1_count == strokes2_count) {
			for (int i = 0; i < strokes1_count; i++) {
				if (strokes1.get(i).equals(strokes2.get(i))) {
					result = true;
				} else {
					result = false;
					break;
				}
			}
		}
		return result;
	}

	/**
	 * Remove a gesture from the library. If there are no more gestures for the
	 * given entry, the gesture entry will be removed.
	 * 
	 * @param label
	 * @param gesture
	 */
	public void removeGesture(String label, Gesture gesture) {
		if (null == label || 0 == label.length()) {
			LOG.e(TAG, "removeGesture: label is null");
			return;
		}
		final List<Gesture> gestures = mNamedGestures.get(label);
		if (null == gestures) {
			return;
		}
		final boolean exist = gestures.remove(gesture);
		// if there are no more samples, remove the entry automatically
		if (gestures.isEmpty()) {
			mNamedGestures.remove(label);
		}
		if (exist) {
			final long id = gesture.getId();
			mLearner.removeInstance(id);
			mChanged = true;
			LOG.d(TAG, "removeGesture: label={0}, id={1,number,#}", label, id);
		}
	}

	/**
	 * Remove all entry of gestures
	 * 
	 */
	public void removeAll() {
		mNamedGestures.clear();
		mLearner.removeAll();
		mLoaded = false;
	}

	/**
	 * Get all the gestures of an entry
	 * 
	 * @param label
	 * @return the list of gestures that is under this name
	 */
	public List<Gesture> getGestures(String label) {
		// final List<Gesture> gestures = mNamedGestures.get(label);
		// if (null != gestures) {
		// return new ArrayList<Gesture>(gestures);
		// } else {
		// return null;
		// }
		return mNamedGestures.get(label);
	}

	/**
	 * Find gesture by gesture_id
	 * 
	 * @param id
	 * @return Gesture
	 */
	public Gesture getGesture(long id) {
		Gesture result = null;
		synchronized (mNamedGestures) {
			final Map<String, List<Gesture>> map = mNamedGestures;
			Map.Entry<String, List<Gesture>> entry = null;
			out: for (Iterator<Map.Entry<String, List<Gesture>>> it = map
					.entrySet().iterator(); it.hasNext() && (result == null);) {
				entry = it.next();
				for (Gesture g : entry.getValue()) {
					if (g.getId() == id) {
						result = g;
						break out;
					}
				}
			}
		}
		return result;
	}

	public NamedGesture getNamedGesture(long id) {
		NamedGesture result = null;
		synchronized (mNamedGestures) {
			final Map<String, List<Gesture>> map = mNamedGestures;
			Map.Entry<String, List<Gesture>> entry = null;
			out: for (Iterator<Map.Entry<String, List<Gesture>>> it = map
					.entrySet().iterator(); it.hasNext() && (result == null);) {
				entry = it.next();
				for (Gesture g : entry.getValue()) {
					if (g.getId() == id) {
						LOG.d(TAG, "getNamedGesture: found id={0,number,#}", id);
						result = new NamedGesture(g, entry.getKey());
						break out;
					}
				}
			}
		}
		return result;
	}

	// public Map<String, List<Gesture>> getNamedGestures() {
	// return mNamedGestures;
	// }

	public boolean hasChanged() {
		return mChanged;
	}

	/**
	 * Save the gestures
	 */
	public void save(OutputStream stream) throws IOException {
		final Map<String, List<Gesture>> maps = mNamedGestures;
		final long start = Utils.currentTime(); // SystemClock.elapsedRealtime();
		DataOutputStream out = null;
		try {
			out = new DataOutputStream(new BufferedOutputStream(stream,
					IO_BUFFER_SIZE));
			// Write version number
			out.writeShort(FILE_FORMAT_VERSION);
			// Write number of entries
			out.writeInt(maps.size());

			for (Map.Entry<String, List<Gesture>> entry : maps.entrySet()) {
				final String key = entry.getKey();
				final List<Gesture> examples = entry.getValue();
				final int count = examples.size();
				// Write entry name
				out.writeUTF(key);
				// Write number of examples for this entry
				out.writeInt(count);
				for (int i = 0; i < count; i++) {
					examples.get(i).serialize(out);
				}
			}
			out.flush();
			LOG.d(TAG, "save: time = {0,number,#} ms",
					(Utils.currentTime() - start));

			mChanged = false;
		} finally {
			if (null != out) {
				out.close();
			}
		}
	}

	/**
	 * Load the gestures
	 */
	public void load(InputStream stream) throws IOException {
		DataInputStream in = null;
		final long start = Utils.currentTime(); // SystemClock.elapsedRealtime();
		short versionNumber = 0;
		try {
			in = new DataInputStream(new BufferedInputStream(stream,
					IO_BUFFER_SIZE));
			// Read file format version number
			versionNumber = in.readShort();
			switch (versionNumber) {
			case 1:
				readFormatV1(in);
				break;
			default:
				break;
			}
			mLoaded = true;
			LOG.d(TAG, "load: readFormatV1() time = {0,number,#} ms",
					(Utils.currentTime() - start));
		} finally {
			if (null != in) {
				in.close();
			}
		}
	}

	private void readFormatV1(DataInputStream in) throws IOException {
		final Learner classifier = mLearner;
		final Map<String, List<Gesture>> namedGestures = mNamedGestures;
		int entriesCount = 0;
		int gestureCount = 0;
		List<Gesture> gestures = null;
		Gesture gesture = null;
		String name = null;
		Feature feature = null;

		removeAll();
		entriesCount = in.readInt();
		for (int i = 0; i < entriesCount; i++) {
			// Entry name
			name = in.readUTF();
			// Number of gestures
			gestureCount = in.readInt();
			gestures = new ArrayList<Gesture>(gestureCount);
			for (int j = 0; j < gestureCount; j++) {
				gesture = Gesture.deserialize(in);
				gestures.add(gesture);
				feature = GestureUtilities.computeFeature(gesture);
				classifier.addInstance(Instance.createInstance(mSequenceType,
						mOrientationStyle, gesture, name, feature));
				// debugSave(Utils.concat(String.valueOf(gesture.getId()), " ",
				// name, "\n"));
			}
			namedGestures.put(name, gestures);
		}
	}

	public boolean hasLoaded() {
		return mLoaded;
	}

	// public static final void debugSave(String str) {
	// final File f = new File(GestureLibrary.DataDir.getAbsolutePath(),
	// "debug.txt");
	// OutputStreamWriter out = null;
	// try {
	// out = new OutputStreamWriter(new FileOutputStream(f, true),
	// Charset.forName("UTF-8"));
	// out.write(str);
	// } catch (IOException e) {
	// e.printStackTrace();
	// } finally {
	// try {
	// out.close();
	// } catch (IOException e) {
	// e.printStackTrace();
	// }
	// }
	// }
}
