/*
 * Copyright (C) 2008-2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Copyright (C) 2011 OgakiSoft
 * 
 * Supported multiple strokes.
 */

package ogakisoft.android.gesture.reform;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import ogakisoft.android.util.LOG;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.RectF;
import android.os.Parcel;
import android.os.Parcelable;

/**
 * A gesture can have a multiple strokes
 * 
 * @author The Android Open Source Project
 * @author noritoshi ogaki
 * @version 1.0
 */
public class Gesture implements Parcelable {
	private static final String TAG = "Gesture";
	private static final long GESTURE_ID_BASE = System.currentTimeMillis();
	private static final int BITMAP_RENDERING_WIDTH = 2;
	private static final boolean BITMAP_RENDERING_ANTIALIAS = true;
	private static final boolean BITMAP_RENDERING_DITHER = true;
	private static final AtomicInteger gestureCount = new AtomicInteger(0);
	private final RectF mBoundingBox = new RectF();
	private long mGestureId;
	public final List<GestureStroke> mStrokes = new ArrayList<GestureStroke>();
	private static final int IO_BUFFER_SIZE = 32 * 1024; // 32K

	public Gesture() {
		mGestureId = GESTURE_ID_BASE + gestureCount.incrementAndGet();
	}

	/**
	 * @return all the strokes of the gesture
	 */
	public List<GestureStroke> getStrokes() {
		List<GestureStroke> clone = new ArrayList<GestureStroke>();
		// clone.addAll(mStrokes);
		int strokes_count = mStrokes.size();
		int points_count;
		GestureStroke stroke;
		List<GesturePoint> points;
		for (int i = 0; i < strokes_count; i++) {
			stroke = mStrokes.get(i);
			points_count = stroke.points.length;
			points = new ArrayList<GesturePoint>(points_count / 2);
			for (int j = 0; j < points_count; j += 2) {
				points.add(new GesturePoint(stroke.points[j],
						stroke.points[j + 1], 0));
			}
			clone.add(new GestureStroke(points));
		}
		return clone;
	}

	public void setStrokes(List<GestureStroke> strokes) {
		mStrokes.clear();
		GestureStroke stroke;
		List<GesturePoint> points;
		int strokes_count = strokes.size();
		int points_count;
		for (int i = 0; i < strokes_count; i++) {
			stroke = strokes.get(i);
			points_count = stroke.points.length;
			points = new ArrayList<GesturePoint>(points_count / 2);
			for (int j = 0; j < points_count; j += 2) {
				points.add(new GesturePoint(stroke.points[j],
						stroke.points[j + 1], 0));
			}
			addStroke(new GestureStroke(points));
		}
	}

	/**
	 * @return the number of strokes included by this gesture
	 */
	public int getStrokesCount() {
		return mStrokes.size();
	}

	/**
	 * Add a stroke to the gesture
	 * 
	 * @param stroke
	 */
	public void addStroke(GestureStroke stroke) {
		mStrokes.add(stroke);
		mBoundingBox.union(stroke.boundingBox);
	}

	/**
	 * Get the total length of the gesture. When there are multiple strokes in
	 * the gesture, this returns the sum of the lengths of all the strokes
	 * 
	 * @return the length of the gesture
	 */
	public float getLength() {
		int len = 0;
		final List<GestureStroke> strokes = mStrokes;
		final int count = strokes.size();
		for (int i = 0; i < count; i++) {
			len += strokes.get(i).length;
		}
		return len;
	}

	/**
	 * @return the bounding box of the gesture
	 */
	public RectF getBoundingBox() {
		return mBoundingBox;
	}

	public Path toPath() {
		return toPath(null);
	}

	public Path toPath(Path path) {
		final List<GestureStroke> strokes = mStrokes;
		final int count = strokes.size();
		if (null == path) {
			path = new Path();
		}
		for (int i = 0; i < count; i++) {
			path.addPath(strokes.get(i).getPath());
		}
		return path;
	}

	public Path toPath(int width, int height, int edge, int numSample) {
		return toPath(null, width, height, edge, numSample);
	}

	public Path toPath(Path path, int width, int height, int edge, int numSample) {
		final List<GestureStroke> strokes = mStrokes;
		final int count = strokes.size();
		if (null == path) {
			path = new Path();
		}
		for (int i = 0; i < count; i++) {
			path.addPath(strokes.get(i).toPath(width - 2 * edge,
					height - 2 * edge, numSample));
		}
		return path;
	}

	/**
	 * Set the id of the gesture
	 * 
	 * @param id
	 */
	void setId(long id) {
		mGestureId = id;
	}

	/**
	 * @return the id of the gesture
	 */
	public long getId() {
		return mGestureId;
	}

	/**
	 * Create a bitmap of the gesture with a transparent background
	 * 
	 * @param width
	 *            width of the target bitmap
	 * @param height
	 *            height of the target bitmap
	 * @param edge
	 *            the edge
	 * @param numSample
	 * @param color
	 * @return the bitmap
	 */
	public Bitmap toBitmap(int width, int height, int edge, int numSample,
			int color) {
		final Bitmap bitmap = Bitmap.createBitmap(width, height,
				Bitmap.Config.ARGB_8888);
		final Canvas canvas = new Canvas(bitmap);
		canvas.translate(edge, edge);
		final Paint paint = new Paint();
		paint.setAntiAlias(BITMAP_RENDERING_ANTIALIAS);
		paint.setDither(BITMAP_RENDERING_DITHER);
		paint.setColor(color);
		paint.setStyle(Paint.Style.STROKE);
		paint.setStrokeJoin(Paint.Join.ROUND);
		paint.setStrokeCap(Paint.Cap.ROUND);
		paint.setStrokeWidth(BITMAP_RENDERING_WIDTH);
		final List<GestureStroke> strokes = mStrokes;
		final int count = strokes.size();
		Path path;
		for (int i = 0; i < count; i++) {
			path = strokes.get(i).toPath(width - 2 * edge, height - 2 * edge,
					numSample);
			canvas.drawPath(path, paint);
		}
		return bitmap;
	}

	/**
	 * Create a bitmap of the gesture with a transparent background
	 * 
	 * @param width
	 * @param height
	 * @param inset
	 * @param color
	 * @return the bitmap
	 */
	public Bitmap toBitmap(int width, int height, int inset, int color) {
		final Bitmap bitmap = Bitmap.createBitmap(width, height,
				Bitmap.Config.ARGB_8888);
		final Canvas canvas = new Canvas(bitmap);
		final Paint paint = new Paint();
		paint.setAntiAlias(BITMAP_RENDERING_ANTIALIAS);
		paint.setDither(BITMAP_RENDERING_DITHER);
		paint.setColor(color);
		paint.setStyle(Paint.Style.STROKE);
		paint.setStrokeJoin(Paint.Join.ROUND);
		paint.setStrokeCap(Paint.Cap.ROUND);
		paint.setStrokeWidth(BITMAP_RENDERING_WIDTH);
		final Path path = toPath();
		final RectF bounds = new RectF();
		path.computeBounds(bounds, true);
		final float sx = (width - 2 * inset) / bounds.width();
		final float sy = (height - 2 * inset) / bounds.height();
		final float scale = (sx > sy) ? sy : sx;
		paint.setStrokeWidth(2.0f / scale);
		path.offset(-bounds.left + (width - bounds.width() * scale) / 2.0f,
				-bounds.top + (height - bounds.height() * scale) / 2.0f);
		canvas.translate(inset, inset);
		canvas.scale(scale, scale);
		canvas.drawPath(path, paint);
		return bitmap;
	}

	public void serialize(DataOutputStream out) throws IOException {
		final List<GestureStroke> strokes = mStrokes;
		final int count = strokes.size();

		// Write gesture ID
		out.writeLong(mGestureId);
		// Write number of strokes
		out.writeInt(count);

		for (int i = 0; i < count; i++) {
			strokes.get(i).serialize(out);
		}
	}

	public static Gesture deserialize(DataInputStream in) throws IOException {
		final Gesture gesture = new Gesture();
		int count = 0;
		// Gesture ID
		gesture.mGestureId = in.readLong();
		// Number of strokes
		count = in.readInt();
		for (int i = 0; i < count; i++) {
			gesture.addStroke(GestureStroke.deserialize(in));
		}
		return gesture;
	}

	public static final Parcelable.Creator<Gesture> CREATOR = new Parcelable.Creator<Gesture>() {
		public Gesture createFromParcel(Parcel in) {
			Gesture gesture = null;
			final long gestureID = in.readLong();
			final DataInputStream inStream = new DataInputStream(
					new ByteArrayInputStream(in.createByteArray()));
			try {
				gesture = deserialize(inStream);
			} catch (IOException e) {
				LOG.e(TAG, "createFromParcel: {0}", e.getMessage());
			} finally {
				// GestureUtilities.closeStream(inStream);
				if (null != inStream) {
					try {
						inStream.close();
					} catch (IOException e) {
						LOG.e(TAG, "createFromParcel: {0}", e.getMessage());
					}
				}
			}
			if (null != gesture) {
				gesture.mGestureId = gestureID;
			}
			return gesture;
		}

		public Gesture[] newArray(int size) {
			return new Gesture[size];
		}
	};

	public void writeToParcel(Parcel out, int flags) {
		out.writeLong(mGestureId);
		boolean result = false;
		final ByteArrayOutputStream byteStream = new ByteArrayOutputStream(
				IO_BUFFER_SIZE);
		final DataOutputStream outStream = new DataOutputStream(byteStream);
		try {
			serialize(outStream);
			result = true;
		} catch (IOException e) {
			LOG.e(TAG, "writeToParcel: {0}", e);
		} finally {
			try {
				outStream.close();
				byteStream.close();
			} catch (IOException e) {
				LOG.e(TAG, "writeToParcel: {0}", e);
			}
		}
		if (result) {
			out.writeByteArray(byteStream.toByteArray());
		}
	}

	public int describeContents() {
		return 0;
	}
}
