#ifndef GIKOMONA_CORE_DATABASE_HPP
#define GIKOMONA_CORE_DATABASE_HPP

#include <memory>

#include <boost/filesystem.hpp>

#include "GikoMona.hpp"

namespace monazilla { namespace GikoMona { namespace core {

class database final {
public:
    typedef database self_type;

    database() : database("", boost::system::error_code()) {}
    
    /// @brief すでに存在するデータベースファイルを開く。まだファイルが存在しない場合は、新たに作成する。
    database(const boost::filesystem::path& db_path,
             boost::system::error_code& ec) : db(db_path.c_str(), ec) {}
    
    database(const self_type& other) = delete;
    
    ~database() { db.close(); }
    
    sqlite::connection& get_connection() { return db; }

    /// @brief データベースの最適化を行う。
    /**
     * @note 内部では以下のコマンドが実行され、データベース中の不要なデータの圧縮と再構築が行われる。
     *       VACUUM; REINDEX;
     */
    void optimize(boost::system::error_code& ec) noexcept {
        sqlite::execute_statement<>(db, u8"VACUUM;", ec);
        sqlite::execute_statement<>(db, u8"REINDEX;", ec);
    }
    
    void begin_sql_statement() noexcept {
        sqlite::execute_statement<>(db, u8"BEGIN;", transaction_ec);
        
        if(transaction_ec) { /* logger */ }
    }
    
    void end_sql_statement() noexcept {
        if(!transaction_ec) {
            sqlite::execute_statement<>(db, "END;");
        } else {
            // logger
        }
    }
    
private:
    sqlite::connection db;
    boost::system::error_code transaction_ec;
};

struct sql_transaction final {
    sql_transaction(database& db_) : db(db_) { db.begin_sql_statement(); }
    ~sql_transaction() { db.end_sql_statement(); }
private:
    database db;
};

} } }

#endif
