#ifndef GIKOMONA_CORE_MODEL_HPP
#define GIKOMONA_CORE_MODEL_HPP

#include <unistd.h>
#include <memory>

#include <boost/fusion/include/vector.hpp>
#include <boost/any.hpp>
#include <boost/lockfree/queue.hpp>
#include <boost/optional.hpp>

#include "GikoMona.hpp"
#include "database.hpp"
#include "config.hpp"

namespace monazilla { namespace GikoMona { namespace core {

class model final {
public:
    typedef model self_type;
    typedef std::tuple<mona_string, boost::any> inserted_value_type;
    typedef inserted_value_type *inserted_value_triv_copyable_type;
    
    model() noexcept;
    ~model() {}
    
    static model *get_instance() {
        if(!instance) {
            /* error? */
        }
        return instance;
    }

    template <typename T, typename ...ValueType>
    bool insert(const mona_string& into,
                const boost::fusion::vector<T, ValueType...>& value) {
        return insert(into, boost::any(value));
    }

    template <typename T>
    boost::optional<T>
    select(const mona_string& from) const noexcept {
        try {
            return boost::any_cast<T>(select(from, true));
        } catch(boost::bad_any_cast&) {
            return boost::none;
        }
    }

    void execute_accumulated_query();
    
    bool load_file(const boost::filesystem::path& file_path);
    
    bool save_to_file(const boost::filesystem::path& file_path);

private:
    static self_type *instance = nullptr;
    
    struct model_pimpl;
    std::shared_ptr<model_pimpl> pimpl;
    
    bool insert(const mona_string& into, const boost::any& value) {
        return insertion_query_queue.push(new inserted_value_type(into, value));
    }
    
    boost::any select(const mona_string& from, bool tag) const noexcept;
    
    std::vector<mona_string> analyze_query(const mona_string& path) const;
    
    boost::lockfree::queue<inserted_value_triv_copyable_type> insertion_query_queue;
};

void execute_accumulated_query_in_model() {
    model::get_instance()->execute_accumulated_query();
}

} } }

#endif 
