
#ifndef GIKOMONA_CORE_STRING_HPP
#define GIKOMONA_CORE_STRING_HPP

#include <string>

#include <boost/utility/string_ref.hpp>

#include <wx/string.h>

// このコンパイラの char16_t、char32_t は UTF-16/32 を表現するか？
#if !(__STDC_UTF_16__)
    #error "[GikoMona.libcore @pp-time] : sorry, this compiler does not support utf-16."
#endif

#if !(__STDC_UTF_32__)
    #error "[GikoMona.libcore @pp-time] : sorry, this compiler does not support utf-32."
#endif

namespace monazilla { namespace GikoMona {
    
template <typename CharT>
using basic_string_ref = boost::basic_string_ref<CharT, std::char_traits<CharT>>;
    
typedef char u8_char;
typedef std::basic_string<u8_char> u8_string;
typedef basic_string_ref<u8_char> u8_string_ref;
    
typedef u8_string mona_string;
typedef basic_string_ref<u8_char> mona_string_ref;

typedef char sjis_char;
typedef std::basic_string<sjis_char> sjis_string;
typedef basic_string_ref<sjis_char> sjis_string_ref;

// to...
u8_string convert_to_u8(const mona_string_ref& src);
sjis_string convert_to_sjis(const mona_string_ref& src);

std::u16string convert_to_u16(const mona_string_ref& src);
std::u32string convert_to_u32(const mona_string_ref& src);
std::string convert_to_std_str(const mona_string_ref& src);
std::wstring convert_to_std_wstr(const mona_string_ref& src);

wxString convert_to_wx(const mona_string_ref& src) {
    return wxString(src.data());
}

// from...
mona_string convert_from(const u8_string_ref& src);
mona_string convert_from(const sjis_string_ref& src);
    
mona_string convert_from(const basic_string_ref<char16_t>& src);
mona_string convert_from(const basic_string_ref<char32_t>& src);
mona_string convert_from(const basic_string_ref<std::string::value_type>& src);
mona_string convert_from(const basic_string_ref<std::wstring::value_type>& src);
    
mona_string convert_from(const wxString& src);

// converter
template <typename Other>
class converter_impl {
public:
    typedef typename Other::value_type char_type;
    typedef basic_string_ref<char_type> string_ref_type;
    
    mona_string from(const string_ref_type& src) const { return convert_from(src); }
};

template <typename Other, bool U8_is_Sjis = std::is_same<u8_string, sjis_string>::value>
class converter {};

template <>
class converter <u8_string, false>  : public converter_impl<u8_string> {
public:
    char_type to(const mona_string_ref& src) const { return convert_to_u8(src); }
};

template <>
class converter <sjis_string, false> : public converter_impl<sjis_string> {
public:
    char_type to(const mona_string_ref& src) const { return convert_to_sjis(src); }
};

template <>
class converter <u8_string, true> : public converter_impl<u8_string> {
public:
    char_type to_u8(const mona_string_ref& src) const { return convert_to_u8(src); }
    char_type to_sjis(const mona_string_ref& src) const { return convert_to_sjis(src); }
    char_type to(const mona_string_ref&) const {
        static_assert(false, "std::is_same<u8_string, sjis_string>::value == true.");
    }
};

template <bool Value>
class converter <std::u16string, Value> : public converter_impl<std::u16string> {
public:
    char_type to(const mona_string_ref& src) const { return convert_to_u16(src); }
};

template <bool Value>
class converter <std::u32string, Value> : public converter_impl<std::u32string> {
public:
    char_type to(const mona_string_ref& src) const { return convert_to_u32(src); }
};

}
}
#endif
