#ifndef GINTENLIB_INCLUDED_DEEP_PTR_HPP_
#define GINTENLIB_INCLUDED_DEEP_PTR_HPP_

/*

      <gintenlib/deep_ptr.hpp>

  deep_ptr ： 深いコピーをするスマートポインタ

  宣言：
    template<typename T>
    struct deep_ptr;
    
    // 基本的なメソッドはスマートポインタと同じなので省略

  機能：
    コピーするときに、オブジェクトの「深いコピー」を製作するポインタ。
    このクラスは基本的に boost::optional で代用できます。
    そのため、あえて deep_ptr を使うメリットは多くありません。
    
    ・new されたものを直接受け取りたい場合
    ・変数自体のサイズが気になる場合
    ・未定義の型を持たせたい場合（クラス内部で、そのクラスの値を持ちたい場合など
    
    せいぜいがこの程度です。その中で比較的重要なのが三番目であり、
    boost::optional の場合、boost::optional<T> 型の変数を定義した時点で T の型の定義が決まっていないといけません（ sizeof(T) が必要なので）。
    しかし deep_ptr なら、実際にコピーされる時まで、対象の型の定義を伸ばせます（ sizeof(T*) は常に一定なので）。
    というわけで、new のコストが軽いオブジェクト（メモリプールを使った operator new が実装されている場合とか）ならば意外と使い道はあるのかもしれません。
    ちなみにこのポインタは、バグ防止のため同じ型同士（ const 等の修飾子は除く）でしかコピーできません。

*/

#include <boost/scoped_ptr.hpp>

#include "shared_ptr.hpp"
#include "pointer_facade.hpp"
#include "is_same_class.hpp"

namespace gintenlib
{
  namespace detail_{ struct deep_ptr_tag_{}; }
  
  template< typename T >
  struct deep_ptr
    : pointer_facade< deep_ptr<T>, T, detail_::deep_ptr_tag_ >
  {
    template<typename U>
    friend class deep_ptr;
    
    typedef typename boost::remove_cv<T>::type object_type;
    
    // デフォルトコンストラクタ
    deep_ptr() : p() {}

    // ポインタから作るコンストラクタ
    template<typename U>
    explicit deep_ptr( U* p_, 
      typename enable_if_same_class<T, U>::type* = 0 )
       : p( p_ ) {}
    
    // コピーコンストラクタ
    deep_ptr( const deep_ptr& src )
       : p( src.make_clone_() ) {}
    
    // 他の種類の deep_ptr からの変換は、const や volatile 修飾しか異ならない場合のみOK
    template< typename U >
    deep_ptr( const deep_ptr<U>& src,
      typename enable_if_same_class<T, U>::type* = 0 )
       : p( src.make_clone_() ) {}
    
    // デストラクタ
    // scoped_ptr() のおかげで何もしなくていい
    // ~deep_ptr(){}
    
    // nothrow swap ( for operator= )
    void swap( deep_ptr& other )
    {
      p.swap( other.p );
    }
    friend void swap( deep_ptr& one, deep_ptr& another )
    {
      one.swap( another );
    }
    
    // 代入演算は copy して swap
    deep_ptr& operator=( const deep_ptr& src )
    {
      deep_ptr( src ).swap( *this );
      return *this;
    }
    template< typename U >
    deep_ptr& operator=( const deep_ptr<U>& src )
    {
      deep_ptr( src ).swap( *this );
      return *this;
    }
    
    // get pointer
    // これと pointer_facade からいろいろと自動生成
    T* get() const { return p.get(); }
    
    // 中身を切り替える
    void reset()
    {
      deep_ptr().swap( *this );
    }
    template<typename U>
    void reset( U* ptr )
    {
      deep_ptr( ptr ).swap( *this );
    }
    
    // shared_ptr に変換する
    shared_ptr<T> to_shared() const
    {
      return shared_ptr<T>( make_clone_() );
    }
    // friend 版
    friend shared_ptr<T> to_shared( const deep_ptr& target )
    {
      return target.to_shared();
    }
  
   private:
    boost::scoped_ptr<T> p;
    
    // p に保持されてるオブジェクトの複製を作る
    object_type* make_clone_() const
    {
      return p ? new object_type(*p) : 0;
    }
  
  };  // class deep_ptr<T>
  
}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_DEEP_PTR_HPP_ 
