#ifndef GINTENLIB_INCLUDED_DELETER_HPP_
#define GINTENLIB_INCLUDED_DELETER_HPP_

/*

      <gintenlib/deleter.hpp>

  deleter ： 単純な delete 演算子と等価なファンクタ
  typed_deleter ： 型の指定された deleter
  dummy_deleter ： 何もしないファンクタ
  typed_dummy_deleter ： 型が指定されてるけど何もしないファンクタ

  機能：
    ポインタ delete 用の関数オブジェクトです。それ以外の何物でもありません。
    deleter は boost::checked_delete を呼んでますが、この中では単に delete 演算子を呼んでいるだけ。
    dummy_deleter は delete 演算子すら呼ばない、単なるプレースホルダーとして機能します。
    それぞれ型付バージョンもあり、それらは unary_function の条件を満たしています。

*/

#include <functional>
#include <boost/checked_delete.hpp>

namespace gintenlib
{
 namespace deleters_  // ADL 対策
 {
  // ただ単に delete するだけのファンクタ
  struct deleter
  {
    typedef void result_type;
    
    template<typename T>
    result_type operator()( T* p ) const throw()
    {
      boost::checked_delete( p );
    }
    
  };  // struct deleter
  
  // 型の束縛された deleter
  template<typename T>
  struct typed_deleter : std::unary_function<T*, void>
  {
    void operator()( T* p ) const throw()
    {
      boost::checked_delete( p );
    }
  
  };  // struct typed_deleter<T>
  
  // 何もしない deleter
  struct dummy_deleter
  {
    typedef void result_type;
    
    template<typename T>
    result_type operator()( T* ) const throw() {}
    
  };  // struct dummy_deleter
  
  // 型束縛されてるけど何もしない deleter
  template<typename T>
  struct typed_dummy_deleter : std::unary_function<T*, void>
  {
    void operator()( T* ) const throw() {}
  
  };  // struct typed_dummy_deleter<T>
  
 }  // namespace deleters_
 
 // 名前を引き込む
 using namespace deleters_;

}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_DELETER_HPP_
