#ifndef GINTENLIB_INCLUDED_FACTORIZE_HPP_
#define GINTENLIB_INCLUDED_FACTORIZE_HPP_

/*
      <gintenlib/factorize.hpp>

  factorize ： 整数の因数分解

  宣言：
    template<typename Int, typename OutIte>
    OutIte factorize( Int n, OutIte out );

  機能：
    整数 n の素因数を、小さい順に出力イテレータ out に出力します。
    n は 0 より大きくなければいけません（ 0 以下にならないよう事前にチェックすべきです）。
    「整数」として使える型は、整数型から暗黙変換でき、四則演算および大小比較が可能な型です。

  使用例：
    // 143 = 11 * 13
    // '11 13 'と表示される
    gintenlib::factorize( 143, std::ostream_iterator<int>( std::cout, " " ) );

  補足：
    ・単純なテンプレートなので、型を指定したい場合は factorize<short>( 143, out ) のように書けます。
    　が、他のテンプレートでは上手くいかないこともあるため、統一性の為にも gintenlib::cast を用いた方がよいかもしれません。
    　その場合は factorize( cast<short>(143), out ) のように書きます（ gintenlib:: は略）

*/

#include <cassert>
#include <stdexcept>

namespace gintenlib
{
  // 素因数分解して out に出力
  template<typename Int, typename OutIte>
  OutIte factorize( Int n, OutIte out )
  {
    // 引数チェック
    if( n <= 0 )
    {
      throw std::invalid_argument( "gintenlib.factorize: n should be greater than 0" );
    }
    
    // 補助ファンクタ
    // 同じ記述を何回も書くのは面倒なので
    struct helper_
    {
      Int& n; OutIte& out;
      helper_( Int& n_, OutIte& out_ )
        : n(n_), out(out_) {}
      
      bool operator()( Int i ) const
      {
        using namespace std;
        assert( i > 1 && n > 0 );
        
        while( n % i == 0 )
        {
          n /= i;
          *out = i;
          ++out;
        }
        
        if( i * i > n )
        {
          if( n > 1 )
          {
            *out = n;
            ++out;
            n = 1;
          }
          return false;
        }
        
        return true;
      }
    } doit( n, out );
    
    // 実際の処理
    // まず 2 と 3、これは i も含めてインライン展開されるといいな
    if( !doit(2) ){ return out; }
    if( !doit(3) ){ return out; }
    
    // 次は汎用だが、2 と 3 の倍数は素数ではないので省略する
    // そのためには、6n - 1 と 6n + 1 のみ検索すればよい
    Int i = 5;
    for(;;)
    {
      // 6n - 1
      if( !doit(i) ){ return out; }
      i += 2;
      
      // 6n + 1
      if( !doit(i) ){ return out; }
      i += 4;
    }
  }
  
}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_FACTORIZE_HPP_
