#ifndef GINTENLIB_INCLUDED_LEXICAL_CONVERT_HPP_
#define GINTENLIB_INCLUDED_LEXICAL_CONVERT_HPP_

/*

      <gintenlib/lexical_convert.hpp>

  lexical_convert ： Cライクに処理できる boost::lexical_cast

  宣言：
    template< typename Target, typename Source >
    bool lexical_convert( Target& dest, const Source& src );

  機能：
    第二引数を boost::lexical_cast によって変換し、その結果を第一引数に代入します。
    正常に変換が行われた場合、戻り値として true が返されます。
    変換が失敗した場合には、第一引数は変化せず、戻り値は false になります。

  使用例：
    // hoge は数値を表しているかもしれない「文字列に変換できる何か」
    int i;
    if( gintenlib::lexical_convert( i, hoge ) )
    {
      // hoge が数値を表していた場合の処理
    }
    else
    {
      // hoge が数値を表していなかった場合の処理
    }

  補足：
    ・基本的に boost::lexical_cast の例外処理が面倒な場合に使います。
    ・使用例のソースコードは、boost::lexical_cast のみを使って次のように書く事も出来ます：
      int i;
      try
      {
        i = boost::lexical_cast<int>(hoge);

        // hoge が数値を表していた場合の処理
      }
      catch( boost::bad_lexical_cast )
      {
        // hoge が数値を表していなかった場合の処理
      }

*/

#include <boost/lexical_cast.hpp>

namespace gintenlib
{
  // src を文字列に変換し、その文字列を Target 型に変換して dest に代入
  // 成功した場合には true が、失敗した場合には false が返る
  template< typename Target, typename Source >
  bool lexical_convert( Target& dest, const Source& src )
  {
    using namespace boost;

    try
    {
      dest = lexical_cast<Target>(src);
      return true;
    }
    catch( bad_lexical_cast )
    {
      return false;
    }
    
  }   // lexical_convert

}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_LEXICAL_CONVERT_HPP_
