#ifndef GINTENLIB_INCLUDED_MOVE_PTR_HPP_
#define GINTENLIB_INCLUDED_MOVE_PTR_HPP_

/*

      <gintenlib/move_ptr.hpp>

  move_ptr ： 所有権移動式スマートポインタ

  宣言：
    template<typename T, typename Deleter>
    struct move_ptr
    {
      // 基本的なメソッドは auto_ptr と同じなので省略
      
      // デリータの型
      typedef Deleter deleter;
      
      // ポインタ（とデリータ）から作るコンストラクタ
      explicit move_ptr( T* p_, const Deleter& del_ = Deleter() );
      
      // release すると同時に deleter を取得する
      // 通常の release 関数も当然あります
      T* release( Deleter& del_ );
      
      // deleter を取得する
      Deleter get_deleter() const;
      
    };

  機能：
    std::auto_ptr を、任意のデリータを指定できるよう拡張した所有権移動式ポインタです。
    それと同時に swap 呼び出しも出来るようになっています。
    それ以外は、細かいところまで std::auto_ptr と同じように振舞うよう工夫しています。
    が、デリータが任意になった都合上、例外安全性では若干脆弱になってしまっています。
    なお、デリータとして使えるのは、「デフォルト構築可能」で「コピーできる」「ファンクタ」。
    例外安全性を考えると「デフォルト構築時」「 swap 時」に例外を投げない、という条件も必要です。
    また、デフォルト構築されたデリータに対して関数呼出しが行われることは有りません。
    なので、関数ポインタをデリータとして使うことも普通にＯＫ。

*/

#include <algorithm>
#include <cassert>
#include <boost/compressed_pair.hpp>
#include <boost/utility/value_init.hpp>
#include "deleter.hpp"
#include "bool_comparable.hpp"

namespace gintenlib
{
 namespace detail
 {
  // 関数の値返しのための、ちょっとした工夫
  template< typename T, typename Deleter >
  struct move_ptr_ref_
  {
    typedef boost::compressed_pair<T*, Deleter> pair_type;
    pair_type pair_;
    
    template< typename U, typename D >
    move_ptr_ref_( const move_ptr_ref_<U, D>& src )
       : pair_( src.pair_.first(), src.pair_.second() ) {}
    
    template< typename U, typename D >
    explicit move_ptr_ref_( const boost::compressed_pair<U, D>& src )
       : pair_( src.first(), src.second() ) {}
    
    template< typename U, typename D >
    move_ptr_ref_( const U& first, const D& second ) : pair_( first, second ) {}
    
  };
  
 }  // namespace detail
  
  // 本体
  template< typename T, typename Deleter = ::gintenlib::deleter >
  struct move_ptr : bool_comparable< move_ptr<T, Deleter> >
  {
    template<typename U, typename D>
    friend class move_ptr;
    
    typedef T element_type;
    typedef T value_type;
    typedef T* pointer;
    typedef T& reference;
    typedef Deleter deleter;
    
    // デフォルトコンストラクタ
    move_ptr() : pair_( 0, deleter() ) {}

    // ポインタ（とデリータ）から作るコンストラクタ
    explicit move_ptr( pointer p_, const deleter& del_ = deleter() )
      : pair_( p_, del_ ) {}
    
    
    // 所有権移動コンストラクタ
    move_ptr( move_ptr& src ) : pair_( 0, deleter() )
    {
      swap(src);
    }
    // 他の move_ptr からの所有権移動
    // 型をまたいだ swap が不可能な以上、微妙に例外安全じゃないので注意
    template< typename U, typename D >
    move_ptr( move_ptr<U, D>& src ) : pair_( src.p(), src.del() )
    {
      src.release();
    }
    
    // デストラクタ
    ~move_ptr()
    {
      if( p() )
      {
        del()( p() );
      }
    }
    
    
    // swap ( for operator= )
    void swap( move_ptr& other )
    {
      pair_.swap( other.pair_ );
    }
    friend void swap( move_ptr& one, move_ptr& another )  // never throws
    {
      one.swap( another );
    }
    
    // move assignment
    move_ptr& operator=( move_ptr& src )
    {
      move_ptr temp(src);
      this->swap( temp );
      return *this;
    }
    template< typename U, typename D >
    move_ptr& operator=( move_ptr<U, D>& src )
    {
      move_ptr temp(src);
      this->swap( temp );
      return *this;
    }
    
    reference operator*() const
    {
      using namespace std;
      assert( p() != 0 );
      return *p();
    }
    pointer operator->() const
    {
      using namespace std;
      assert( p() != 0 );
      return p();
    }
    
    // get pointer
    pointer get() const { return p(); }
    // mem_fn support
    friend T* get_pointer( const move_ptr& ptr )
    {
      return ptr.get();
    }
    
    // release pointer
    // ポインタをリリースし、デリータを初期化します。
    // deleter の swap が例外を投げなければ例外安全です。
    pointer release()
    {
      pointer ptr = 0;
      boost::value_initialized<deleter> d_;
      deleter& d = d_;
      
      using std::swap;
      swap( p(), ptr );
      swap( del(), d );
      
      return ptr;
    }
    // ポインタをリリースし、デリータも取得。
    // 渡される参照に何が代入されていても構いません。
    // 例外安全性を考慮し swap による実装を行っています。
    pointer release( deleter& del_ )
    {
      pointer ptr = 0;
      boost::value_initialized<deleter> d_;
      deleter& d = d_;
      
      using std::swap;
      swap( del(), del_ );
      
      swap( p(), ptr );
      swap( del(), d );
      
      return ptr;
      /*
      del_ = del();
      
      return release();
      */
    }
    
    deleter get_deleter() const
    {
      return del();
    }
    friend deleter get_deleter( const move_ptr& target )
    {
      return target.get_deleter();
    }
    
    void reset()
    {
      move_ptr temp;
      this->swap(temp);
    }
    void reset( T* p_, const deleter& del_ = deleter() )
    {
      move_ptr temp( p_, del_ );
      this->swap( temp );
    }
    
    
    bool operator!() const  // never throws
    {
      return !p();
    }
    
    bool operator==( const move_ptr& other ) const  // never throws
    {
      return p() == other.p();
    }
    bool operator!=( const move_ptr& other ) const  // never throws
    {
      return p() != other.p();
    }

    template<typename U, typename D>
    bool operator==( const move_ptr<U, D>& other ) const  // never throws
    {
      return p() == other.p();
    }
    template<typename U, typename D>
    bool operator!=( const move_ptr<U, D>& other ) const  // never throws
    {
      return p() != other.p();
    }
    
   private:
    typedef typename detail::move_ptr_ref_<T, Deleter>::pair_type pair_type;
    pair_type pair_;
    
    pointer& p()
    {
      return pair_.first();
    }
    const pointer& p() const
    {
      return pair_.first();
    }
  
    deleter& del()
    {
      return pair_.second();
    }
    const deleter& del() const
    {
      return pair_.second();
    }
    
   public:
    // 値での受け渡しのための、ちょっとしたテク
    move_ptr( const detail::move_ptr_ref_<T, Deleter>& src ) : pair_(src.pair_) {} 

    move_ptr& operator=( const detail::move_ptr_ref_<T, Deleter>& src )
    {
      move_ptr temp(src);
      this->swap( temp );
      return *this;
    }
    
    template<typename U, typename D>
    operator detail::move_ptr_ref_<U, D>()
    { 
      detail::move_ptr_ref_<U, D> temp( pair_ );
      release();
      return temp; 
    }
  
    // pointer casts
    template<typename U>
    friend move_ptr<U, deleter> static_pointer_cast( move_ptr& t )
    {
      move_ptr<U, deleter> temp( static_cast<U*>( t.p() ), t.del() );
      t.release();
      return temp;
    }
    template<typename U>
    friend move_ptr<U, deleter> dynamic_pointer_cast( move_ptr& t )
    {
      move_ptr<U, deleter> temp( dynamic_cast<U*>( t.p() ), t.del() );
      t.release();
      return temp;
    }
    template<typename U>
    friend move_ptr<U, deleter> const_pointer_cast( move_ptr& t )
    {
      move_ptr<U, deleter> temp( const_cast<U*>( t.p() ), t.del() );
      t.release();
      return temp;
    }
    
  };  // class move_ptr<T>
  
}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_MOVE_PTR_HPP_ 
