#ifndef GINTENLIB_PREPROCESSOR_INCLUDED_DEQUOTE_HPP
#define GINTENLIB_PREPROCESSOR_INCLUDED_DEQUOTE_HPP

/*

      <gintenlib/preprocessor/dequote.hpp>

  dequote ： マクロの引数にテンプレートを使う場合のユーティリティ

  宣言：
    namespace gintenlib
    {
     namespace detail_
     {
      template<typename T>
      struct dequote_helper{};
      
      template<typename Arg>
      struct dequote_helper<void (*)(Arg)>
      {
        typedef Arg type;
      };
     
     }
    }
    
    #define GINTENLIB_DEQUOTE( QUOTED_TYPE ) \
      ::gintenlib::detail_::dequote_helper<void (*)QUOTED_TYPE>::type
    
  使用例：
    // 例として、与えられた型にたいし、その型二つのペアの型を返すマクロを定義してみる
    // テンプレートへの別名付けが C++ 標準でサポートされていれば不要ですが、現状無理なので
    
    // まずダメな例
    // #define PAIR_OF( type ) std::pair<type, type>
    // これだと、 PAIR_OF( std::pair<int, double> ) のような
    // 複数の引数を持つテンプレート型を、マクロの引数として渡すことが出来ない
    // （プリプロセッサはテンプレートの三角括弧を無視するため、二引数マクロと勘違いされる）
    
    // そこで、こう書く
    #define PAIR_OF( quoted_type )                  \
      std::pair<                                    \
        typename GINTENLIB_DEQUOTE( quoted_type ),  \
        typename GINTENLIB_DEQUOTE( quoted_type )   \
      >
    // GINTENLIB_DEQUOTE は なんたら::type というふうに展開されるので
    // 必要に応じて typename キーワードを付けなければいけない
    // 大体の場合はとりあえず typename を付ければＯＫ
    
    // 使用するときは、
    PAIR_OF(( std::pair<int, double> )) hoge;
    // このように型名を二重括弧でくるむ

  解説：
    このマクロが何をしているかは、例を見れば分かると思うので、分かってください。
    簡単に言えば、「 ( 型名 ) 」というシンボル列を受け取り、括弧を外すマクロです。
    実装方法ですが、実は「 ( なんたら ) 」の括弧を直接外す方法は（知りえる範囲では）存在しないので、
    「 void (*)( 型名 ) 」という表現に直して、関数の引数、という形で型名を分離しています。
    そういう実装の都合上、関数型をテンプレートの引数として特殊化できるコンパイラでしか動作しません。
    また、型名自身に展開されるわけでもないので、必要に応じて typename 修飾する必要があります。

*/

namespace gintenlib
{
 namespace detail_
 {
  template<typename T>
  struct dequote_helper {};
  
  template<typename Arg>
  struct dequote_helper<void (*)(Arg)>
  {
    typedef Arg type;
  };
  
  // 本体
  // QUOTED_TYPE は ( DEQUOTED_TYPE ) の形で指定される
  // dequote_helper<void (*)( DEQUOTED_TYPE )>::type は DEQUOTED_TYPE 型となる
  #define GINTENLIB_DEQUOTE( QUOTED_TYPE ) \
    ::gintenlib::detail_::dequote_helper<void (*)QUOTED_TYPE>::type
  
 }  // namespace detail_
}   // namespace gintenlib


#endif  // #ifndef GINTENLIB_PREPROCESSOR_INCLUDED_DEQUOTE_HPP
