#ifndef GINTENLIB_INCLUDED_REFERENCE_COUNTER_HPP_
#define GINTENLIB_INCLUDED_REFERENCE_COUNTER_HPP_

/*

      <gintenlib/reference_counter.hpp>

  reference_counter ： boost::intrusive_ptr 用オブジェクト生成
  
  【注】同様の機能を提供するライブラリに、 intrusive_ptr_hook があります。
        intrusive_ptr_hook は削除子の指定が可能となっている改良であり、
        基本的には intrusive_ptr_hook を使うことを推奨します。

  宣言：
    template<typename Derived, bool multiplely_successable = false >
    class reference_counter
    {
     public:
      // 参照を増やす
      void AddRef() const;
      // 参照を減らす。ゼロになったら自動解放
      void Release() const;
      
     protected:
      // 初期化
      reference_counter();
      // 初期カウントを指定して初期化
      reference_counter( int initial_count );
      // コピーは一応可能にする
      // カウントは動かない
      reference_counter( const reference_counter& );
      // 解放
      ~reference_counter();
      
     private:
      // 代入は出来ない
      void operator=( const reference_counter& );
      
    };
    
    template< typename T, bool b >
    inline void intrusive_ptr_add_ref( const reference_counter<T, b>* ptr );
    template< typename T, bool b >
    inline void intrusive_ptr_release( const reference_counter<T, b>* ptr );
    
  使用法：
    class hoge : public gintenlib::reference_counter<hoge>
    {
      // 適当に中身を記述
    };
    
    boost::intrusive_ptr<hoge> p = new hoge;
    // or gintenlib::com_ptr<hoge> p( new hoge );
    
    // 多重継承も出来ます。その場合はテンプレート第二引数に true を渡してください
    struct base1 : gintenlib::reference_counter<base1, true>
    {
      // 中身
    };
    struct base2 : gintenlib::reference_counter<base2, true>
    {
      // 中身
    };
    struct derived : base1, base2
    {
      // 中身
    };
    
  機能：
    使用法を見れば一目瞭然でしょう。テンプレート引数に自身を渡して継承すればOKです。
    第二テンプレート引数は、多重継承が出来るか否か。trueを渡せば多重継承できるようになります。
    デフォルトでは false です。この場合、仮想関数テーブルは使用していません。
    よって、非多重継承版で継承をしたい場合、デストラクタを virtual 指定する必要があります。
    多重継承版では、そういった気遣いをする必要はありません。が、その分コストが高いです。
    
  備考：
    このクラスは、当初は noncopyable として製作されていました。
    しかし、派生先でコピーコンストラクタを自動生成してくれないのは面倒なので、コピーに限り可能になっています。
    この際、参照カウントに関しては、コピー元には無関係に 0 に初期化されます。
    また代入演算は「 copy して swap 」という例外安全な方法があるので、そちらを推奨する意味で禁止になっています。

*/

#include <cassert>
#include <algorithm>

namespace gintenlib
{
  // 非仮想バージョン。サイズや速度の面で最適化されますが、多重継承は出来ません
  template<typename Derived, bool multiplely_successable = false >
  struct reference_counter
  {
    void AddRef() const
    {
      using namespace std;
      assert( count >= 0 );
      
      ++count;
    }
    void Release() const
    {
      using namespace std;
      assert( count > 0 );
      
      if( --count == 0 )
      {
        delete static_cast<const Derived*>(this);
      }
    }
    
    // 参照カウントを取得
    int use_count() const { return count; }
    
   protected:
    reference_counter() : count(0) {}
    reference_counter( int initial_count ) : count(initial_count) {}
    ~reference_counter()
    {
      using namespace std;
      assert( count == 0 );
    }
    
    // コピーは一応可能にする
    reference_counter( const reference_counter& ) : count(0) {}
    
   private:
    mutable int count;
    // 代入は出来ない
    void operator=( const reference_counter& );
    
  };
  
  // boost::intrusive_ptr 用のインターフェイス
  // reference_counter<T> は reference_counter<T, false> の意味なので
  // 仮想継承版とは競合しません
  template< typename T >
  inline void intrusive_ptr_add_ref( const reference_counter<T>* ptr )
  {
    ptr->AddRef();
  }
  template< typename T >
  inline void intrusive_ptr_release( const reference_counter<T>* ptr )
  {
    ptr->Release();
  }
  
  
  // 仮想継承版のベースクラス
  struct reference_counter_base
  {
    void AddRef() const
    {
      using namespace std;
      assert( count >= 0 );
      
      ++count;
    }
    void Release() const
    {
      using namespace std;
      assert( count > 0 );
      
      if( --count == 0 )
      {
        delete this;
      }
    }
    // 参照カウントを取得
    int use_count() const { return count; }
    
   protected:
    reference_counter_base() : count(0) {}
    reference_counter_base( int initial_count ) : count(initial_count) {}
    virtual ~reference_counter_base()
    {
      using namespace std;
      assert( count == 0 );
    }
    
    // コピーは一応可能にする
    reference_counter_base( const reference_counter_base& ) : count(0) {}
        
   private:
    mutable int count;
    // 代入は出来ない
    void operator=( const reference_counter_base& );
    
  };
  
  // 仮想継承版本体。コストは高いですがダイアモンド継承できます
  template<typename Derived>
  struct reference_counter<Derived, true>
    : virtual reference_counter_base
  {
   protected:
    reference_counter()
      : reference_counter_base() {}
    reference_counter( int initial_count )
      : reference_counter_base(initial_count) {}
    virtual ~reference_counter(){}
    
  };
  
  // 仮想継承版では reference_counter_base に対し add_ref/release を定義することで
  // 衝突が起こらないように工夫されています
  inline void intrusive_ptr_add_ref( const reference_counter_base* ptr )
  {
    ptr->AddRef();
  }
  inline void intrusive_ptr_release( const reference_counter_base* ptr )
  {
    ptr->Release();
  }
  
  
}   // namespace gintenlib


#endif  // #ifndef GINTENLIB_INCLUDED_REFERENCE_COUNTER_HPP_
