#ifndef GINTENLIB_INCLUDED_STORAGE_HPP_
#define GINTENLIB_INCLUDED_STORAGE_HPP_

/*

      <gintenlib/storage.hpp>

  storage ： メモリ領域確保用クラス

  宣言：
    template< typename T >
    struct storage_of
    {
      typedef boost::aligned_storage< sizeof(T), boost::alignment_of<T>::value >::type type;
    };
    
    template<typename T>
    struct storage
    {
      // noncopyable にしてる都合上、必要
      storage() {}
      
      // メタ関数
      // 内部に格納されてるストレージの型
      typedef typename storage_of<T>::type storage_type;
      // メタ関数としても使えるように（互換性の為）。
      typedef storage_type type;
      
      // アドレス取得
      void* address();
      void const* address() const;
      
     private:
      // noncopyable
      storage( storage const& );
      storage& operator=( storage const& );
      
    };

  機能：
    T 型のクラスを格納できるだけのメモリ領域を確保したい場合、
    安直に char[sizeof(T)] を使うと、アライメントの問題が発生します（詳しくは調べてください）。
    それを避ける為の部品が、 boost にある alignment_storage クラスであり、
    boost::aligned_storage< sizeof(T), boost::alignment_of<T>::value >
    という型を使うことで、 T 型のクラスを格納できるメモリ領域を確保することが出来ます。
    しかしこれには弱点があります。この表現では直感的に「 T 型を入れるための場所」と分かりません。
    そこで、もっと直感的な表現として用意されたのが gintenlib::storage<T> です。
    意味はそのまま「 T 型のオブジェクトを格納できるメモリ領域」。
    その領域のアドレスを取得したい場合は address() 関数を呼び出すことで使えます。

*/

#include <boost/aligned_storage.hpp>
#include <boost/type_traits/alignment_of.hpp>

namespace gintenlib
{
  // storage_of<T>::type は T 型を格納できる POD 型
  template<typename T>
  struct storage_of
  {
    typedef typename boost::aligned_storage<
      sizeof(T), boost::alignment_of<T>::value >::type type;
  };
  
  // T 型を収められるメモリ領域
  // noncopyable で has_trivial_destructor
  // boost::noncopyable を継承すると has_trivial_destructor にならないので
  // 自前で代入とコピーを殺している
  template<typename T>
  struct storage
  {
    // noncopyable にしてる都合上、必要
    // = default 指定が欲しい。
    storage() {}
    
    // メタ関数
    // 内部に格納されてるストレージの型
    typedef typename storage_of<T>::type storage_type;
    // メタ関数としても使えるように（互換性の為）。
    typedef storage_type type;
    
    // アドレス取得
    void* address()             { return &data_; }
    void const* address() const { return &data_; }
    
   private:
    storage_type data_;
    
    // noncopyable
    storage( storage const& );
    storage& operator=( storage const& );
    
  };
  
}   // namespace gintenlib


#endif  // #ifndef GINTENLIB_INCLUDED_STORAGE_HPP_
