#ifndef GINTENLIB_INCLUDED_TO_SHARED_HPP_
#define GINTENLIB_INCLUDED_TO_SHARED_HPP_

/*

      <gintenlib/to_shared.hpp>

  to_shared ： 任意のポインタを shared_ptr に変換

  宣言：
    // 任意ポインタ
    template<typename P>
    shared_ptr<typename P::element_type> to_shared( const P& pt );
    
    // std::auto_ptr
    template<typename T>
    shared_ptr<T> to_shared( std::auto_ptr<T> p );
    
    // trivial ですが shared_ptr からも
    template<typename T>
    shared_ptr<T> to_shared( const shared_ptr<T>& p );
    
    // これは intrusive_to_shared.hpp 内で定義
    template<typename T>
    shared_ptr<T> to_shared( const boost::intrusive_ptr<T>& p );
    
    // そのほか、銀天ライブラリの各スマートポインタに対し適切な to_shared が定義されている

  機能：
    各種スマートポインタを shared_ptr に変換します。
    名前空間の自動照会は行わない事になりました。代わりに using gintenlib::to_shared としてください。
  
  要求：
    一般の P に対する要求は、以下の通り：
      element_type を持つ。
      P::element_type get() const; メンバ関数を持つ。
      デフォルトコンストラクタを呼び出せる。例外を投げない。
      copy constructable である。その際、内部に格納されているポインタのアドレスが変化してもいい。
      swap( p1, p2 ) が例外を投げない。

*/

#include "shared_ptr.hpp"
#include "intrusive_to_shared.hpp"

#include <cassert>
#include <memory>

namespace gintenlib
{
  // std::auto_ptr version
  // 実際のところ必要ないけど
  template<typename T>
  inline shared_ptr<T> to_shared( std::auto_ptr<T> pt )
  {
    return shared_ptr<T>( pt.release() );
  }
  
  // trivial version
  template<typename T>
  inline shared_ptr<T> to_shared( const shared_ptr<T>& pt )
  {
    return pt;
  }

  // 汎用バージョン
  
  // いわゆる「スマートポインタ」を保持するホルダ
  // shared_ptr のコンストラクタの第二引数として渡す削除子として使える
  template<typename Pointer>
  struct to_shared_holder
  {
    typedef typename Pointer::element_type T;
    typedef shared_ptr<T> shared_t;
    
    // 構築
    to_shared_holder()
      : pt() {}
    to_shared_holder( const Pointer& pt_ )
      : pt( pt_ ) {}
    
    // shared_ptr にする
    static shared_t to_shared( const Pointer& pt_ )
    {
      // まず null pointer に対する最適化
      T* p = pt_.get();
      if( !p ){ return shared_t(); }
      
      // 本体は転送
      return to_shared_holder(pt_).to_shared();
    }
    // 本体
    // これを実行すると、このオブジェクトは NULL に再設定される
    shared_t to_shared()
    {
      using namespace std;  // for assert, swap
      T* p = pt.get();
      
      // いったん仮の to_shared_holder を使って shared_ptr を作る
      // ここで例外が投げられた場合、何も行われず抜けるので安全
      shared_t temp( p, to_shared_holder() );
      
      // 作られた shared_ptr の中身をいじる
      // ポイントは、ここから仮の to_shared_holder が正しく設定されるまで、例外が投げられないこと
      
      // これは例外を投げない
      to_shared_holder* holder = boost::get_deleter< to_shared_holder >( temp );
      // 作ったばかりの holder の型が不一致なわけないが、一応チェック
      assert( holder != 0 );
      
      // これも大丈夫
      Pointer& pt_ = holder->pt;
      // これが一番の山場。普通は swap は nothrow と信じる
      swap( pt, pt_ );
      
      // これで to_shared_pointer の中身が正しく設定された
      
      // 一応確認する
      assert( pt.get() == 0 && pt_.get() == p );
      
      // 何故こんな面倒なことをしているかというと、
      // 深いコピーを行うタイプのスマートポインタも問題なく保持できるようにしたいため。
      // Pointer p2 = p1; の処理をした後に、 p1.get() == p2.get() である保証はないのです
      
      // 完成した shared_ptr を返す
      return temp;
    }
    
    // 削除関数
    typedef void result_type;
    void operator()( T* )
    {
      // ポインタを破棄する
      Pointer p0;
      
      using namespace std;
      swap( pt, p0 );
    }
    
   private:
    // メンバとして Pointer を保持する
    Pointer pt;
  
  };  // to_shared_holder<Pointer>
  
  // 汎用版の本体は単なるラッパ
  template<typename P>
  inline shared_ptr<typename P::element_type> to_shared( const P& pt )
  {
    return to_shared_holder<P>::to_shared( pt );
  }
  
}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_TO_SHARED_HPP_ 
