#ifndef GINTENLIB_INCLUDED_VALUE_SAVER_HPP_
#define GINTENLIB_INCLUDED_VALUE_SAVER_HPP_

/*

      <gintenlib/value_saver.hpp>

  value_saver ： デストラクタでの値の復帰

  宣言：
    class value_saver : boost::noncopyable
    {
     public:
      template< typename T > explicit value_saver( T& t );
      ~value_saver();
      
      // ともに typed_saver と同じ
      void invoke();
      void release();
      void reset();
    };
    
  機能：
    コンストラクタで受け取った変数の値を記録し、デストラクタで書き戻します。
    typed_saver は変数の型を指定する必要がありますが高速に動作し、
    value_saver は変数の型に依らない動作をしますが、その分のコストがかかります。
    
    どちらも、release() 関数を呼び出すことでデストラクタでの書き戻しを抑制できます。
    また、デストラクタ前でも invoke() 関数により強制的に書き戻すことが可能です。
    なお、単に invoke しただけでは、その後のデストラクタでの書き戻しは抑制されません。
    デストラクタ呼び出し前に書き戻しを行い、かつ release するには、 reset() を呼びます。
  
  補足事項：
    typed_saver, value_saver を利用できるオブジェクトの条件は、
    ・CopyConstructible
    ・Assignable
    の２点です（version1.0.0で変更）。
    また、デストラクタによる書き戻しにおいては、 swap 関数を利用して無駄なコピーを省いています。
    例外を投げない swap() 関数呼び出しが存在すれば、例外安全です。


*/

#include <algorithm>

#include <boost/scoped_ptr.hpp>
#include <boost/noncopyable.hpp>
#include <boost/utility/addressof.hpp>

namespace gintenlib
{
  // 万能セーバ
  struct value_saver
    : private boost::noncopyable
  {
    // t の値を記録し、デストラクタで巻き戻す
    template< typename T >
    explicit value_saver( T& t )
      : p( new holder<T>(t) ) {}
    
    // 復帰
    ~value_saver() { reset(); }
    
    // 明示的な復帰
    // この呼出が行われたとしても、デストラクタでは再度 値の復帰が行われる
    void invoke()
    {
      if(p)
      {
        p->invoke();
      }
    }
    // 復帰抑止
    void release()
    {
      p.reset();
    }
    
    // invoke して release する
    void reset()
    {
      if(p)
      {
        p->reset();
        p.reset();
      }
    }

   private:
    struct basic_holder
      : private boost::noncopyable
    {
      virtual ~basic_holder(){}
      virtual void invoke() = 0;
      virtual void reset() = 0;
    };
    boost::scoped_ptr<basic_holder> p;
    
    template< typename T >
    struct holder
      : basic_holder
    {
      holder( T& t )
        : target_(t), saver_(t) {}
      
      virtual void invoke()
      {
        target_ = saver_;
      }
      virtual void reset()
      {
        using std::swap;
        swap( target_, saver_ );
      }

     private:
      T& target_;
      T saver_;

    };  // class holder<T>

  };  // class value_saver

}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_VALUE_SAVER_HPP_
