// ヘッダのインクルード
#include "../gintenlib/bool_comparable.hpp"

// boost の単体テストフレームワーク
#include <boost/test/minimal.hpp>

// まず普通のクラスとして bool_comparable.hpp 中の例文

#include <boost/scoped_ptr.hpp>

template<typename T>
class my_optional
  : public gintenlib::bool_comparable< my_optional<T> > // このように使う
{
  // 本体部分はなんら気にせず書いてよい
 public:
  typedef  T  value_type;
  
  typedef       T&       reference;
  typedef const T& const_reference;
  
  my_optional() : p() {}
  my_optional( const T& x ) : p( new T(x) ) {}
  
  reference        operator*()       { return *p; }
  const_reference  operator*() const { return *p; }
  
  // operator!() を確実に実装することを忘れなければ。
  bool operator!() const { return !p; }
  // これにより、bool比較も出来るようになる
  
  // ちなみに operator!() じゃなくて
  // bool boolean_test() const { return p.get(); }
  // これでもいい（その場合、operator! は自動定義）
  
  // そのほか、コピーコンストラクタとか代入演算子とかは面倒だから省略
  
 private:
  // 面倒なポインタ管理とかしたくないので scoped_ptr を使う
  boost::scoped_ptr<T> p;
  
};

// テストケース
void my_optional_test()
{
  my_optional<int> a, b(1), c(0);
  
  // 基本的なチェック
  
  // if の条件部に直接使う
  if(  a  )       { BOOST_ERROR( "a is empty." ); }
  if(  b  ); else { BOOST_ERROR( "b is not empty." ); }
  // 当然だけど ! 演算子もテストする
  if( !c  )       { BOOST_ERROR( "c is not empty." ); }
  
  // || や &&
  // || や && は直接多重定義してないので、short circuit になるはず
  BOOST_CHECK( b || a );
  BOOST_CHECK( b && c );
  
  // 0 との比較
  BOOST_CHECK( a == 0 );
  BOOST_CHECK( 0 == a );  // 逆もテスト
  BOOST_CHECK( b != 0 );  // !=
  BOOST_CHECK( 0 != c );  // != の逆
  
  // true, false との比較
  BOOST_CHECK( b == true  );  // これはＯＫ
  BOOST_CHECK( false == a );  // ＯＫ
  BOOST_CHECK( c != false );  // ＯＫ
  BOOST_CHECK( true  != a );  // ＯＫ
  // BOOST_CHECK( b == 1 );   // コレはダメ
  
  // bool へのキャスト
  BOOST_CHECK( static_cast<bool>(a) != static_cast<bool>(b) );
  
  // 直接比較はダメ
  // BOOST_CHECK( a != b );
  // if( b == c ){ BOOST_ERROR( "1 equals to 0 !?" ); }
}


// 次、比較も出来るクラス
#include <boost/operators.hpp>
#include <string>
using namespace std;

struct eq_cmp
  : gintenlib::bool_comparable<eq_cmp>,
    private boost::equality_comparable<eq_cmp>
{
  string value;
  eq_cmp( string x = "" ) : value(x) {}
  
  // これさえ実装すればＯＫ
  bool operator!() const { return value.empty(); }
  
  // equal_to
  friend bool operator==( const eq_cmp& lhs, const eq_cmp& rhs )
  {
    return lhs.value == rhs.value;
  }

  // テストコード
  static void test()
  {
    eq_cmp x, y("hoge");
    
    // この辺は適当に流す
    BOOST_CHECK( !x );
    BOOST_CHECK(  y );
    
    BOOST_CHECK( x != true );
    BOOST_CHECK( y == true );
    BOOST_CHECK( x == 0 );
    BOOST_CHECK( y != 0 );
    
    // 当然ダメ
    // BOOST_CHECK( x != 1 );
    // BOOST_CHECK( y == 1 );
    
    // これらは比較関数を定義してるのでＯＫ
    BOOST_CHECK( x == x );
    BOOST_CHECK( x != y );
    // 暗黙変換でこれもＯＫ
    BOOST_CHECK( x == string("") );
    BOOST_CHECK( y != string("") ); 
    
    // 逆方向でも基本ＯＫ
    BOOST_CHECK( ( x == true ) == ( true == x ) );
    BOOST_CHECK( ( x != true ) == ( true != x ) );
    BOOST_CHECK( ( x == 0 ) == ( 0 == x ) );
    BOOST_CHECK( ( x != 0 ) == ( 0 != x ) );
    BOOST_CHECK( ( x == y ) == ( y == x ) );
    BOOST_CHECK( string("hoge") != x );
    BOOST_CHECK( string("hoge") == y ); 
    
    // 逆方向でも当然ダメ
    // BOOST_CHECK( 1 != x );
    // BOOST_CHECK( 1 == y );
  }

};


// int から暗黙変換できる場合、ちょっと特殊

struct convertible_from_int
  : gintenlib::bool_comparable<convertible_from_int>,
    private boost::equality_comparable<convertible_from_int>
{
  int value;
  convertible_from_int( int x = 0 ) : value(x) {} // int から暗黙変換できると・・・
  
  bool operator!() const { return value == 0; }
  
  // operator== による比較が出来る場合
  friend bool operator==( const convertible_from_int& lhs, const convertible_from_int& rhs )
  {
    return lhs.value == rhs.value;
  }
 
  static void test()
  {
    convertible_from_int x, y = 2;
    BOOST_CHECK( !x ); // こういうのは普通に出来るが
    BOOST_CHECK( y );
    BOOST_CHECK( x == false );
    BOOST_CHECK( y == true );
    
    // BOOST_CHECK( x == 0 ); // コレが実行できない（曖昧になってしまう）
    // BOOST_CHECK( y != 0 ); // 当然これもダメ
    
    BOOST_CHECK( y == 2 ); // でもコレはＯＫ
    BOOST_CHECK( x != 1 ); // コレもＯＫ
  }
};

// その場合、int との比較を別個定義してやればＯＫ
struct convertible_from_int2
  : gintenlib::bool_comparable<convertible_from_int2>,
    private boost::equality_comparable<convertible_from_int2>,
    private boost::equality_comparable<convertible_from_int2, int>
{
  int value;
  convertible_from_int2( int x = 0 ) : value(x) {} // int から暗黙変換できると・・・
  
  // こっそり boolean_test を使う。特に意味は無い
  bool operator!() const { return value == 0; }
  
  // 通常の == 比較に加えて
  friend bool operator==( const convertible_from_int2& lhs, const convertible_from_int2& rhs )
  {
    return lhs.value == rhs.value;
  }
  // int との比較を明示的に定義さえすれば
  friend bool operator==( const convertible_from_int2& lhs, int rhs )
  {
    return lhs.value == rhs;
  }
 
  static void test()
  {
    convertible_from_int2 x, y = 2;
    
    BOOST_CHECK( x == 0 ); // コレも実行できる！
    BOOST_CHECK( y != 0 );
    
    BOOST_CHECK( y == 2 ); // 当然コレもＯＫ
    BOOST_CHECK( x != 1 );
    
    BOOST_CHECK( !x );  // ちなみに boolean_test() があれば operator !() も定義される
                        // 今回のテストにはあまり関係ないが
    BOOST_CHECK( y );
    BOOST_CHECK( x == false );
    BOOST_CHECK( y == true );
  }
};


// テストメイン
int test_main( int, char** )
{
  my_optional_test();
  eq_cmp::test();
  convertible_from_int::test();
  convertible_from_int2::test();
  
  return 0;
}
