package jp.sourceforge.glad.calendar.text;

import java.text.DateFormat;
import java.text.SimpleDateFormat;

import jp.sourceforge.glad.calendar.CalendarConsts;
import jp.sourceforge.glad.calendar.ISOCalendar;
import junit.framework.TestCase;

/**
 * {@link ISODateFormatSymbols} の単体テスト。
 * 
 * @author GLAD!!
 */
public class ISODateFormatSymbolsTest extends TestCase {

    ISODateFormatSymbols symbols;
    ISOCalendar calendar;

    protected void setUp() throws Exception {
        symbols = new ISODateFormatSymbols();
        calendar = new ISOCalendar(2009, 1, 23, 1, 12, 23, 123);
    }

    protected void tearDown() throws Exception {
        symbols = null;
        calendar = null;
    }

    public void testISO8601Basic() {
        DateFormat df = new SimpleDateFormat(
                "yyyyMMdd'T'HHmmss.SSSz", symbols);
        assertEquals("20090123T011223.123+0900",
                df.format(calendar.toDate()));
        df.setTimeZone(CalendarConsts.UTC);
        assertEquals("20090122T161223.123Z",
                df.format(calendar.toDate()));
    }

    public void testISO8601Extended() {
        DateFormat df = new SimpleDateFormat(
                "yyyy-MM-dd'T'HH:mm:ss.SSSzzzz", symbols);
        assertEquals("2009-01-23T01:12:23.123+09:00",
                df.format(calendar.toDate()));
        df.setTimeZone(CalendarConsts.UTC);
        assertEquals("2009-01-22T16:12:23.123Z",
                df.format(calendar.toDate()));
    }

    public void testISO8601OrdinalDateBasic() {
        DateFormat df = new SimpleDateFormat("yyyyDDD", symbols);
        assertEquals("2009023", df.format(calendar.toDate()));
        df.setTimeZone(CalendarConsts.UTC);
        assertEquals("2009022", df.format(calendar.toDate()));
    }

    public void testISO8601OrdinalDateExtended() {
        DateFormat df = new SimpleDateFormat("yyyy-DDD", symbols);
        assertEquals("2009-023", df.format(calendar.toDate()));
        df.setTimeZone(CalendarConsts.UTC);
        assertEquals("2009-022", df.format(calendar.toDate()));
    }

    public void testISO8601WeekAndDayBasic() {
        DateFormat df = new SimpleDateFormat("yyyy'W'wwE", symbols);
        assertEquals("2009W045", df.format(calendar.toDate()));
        df.setTimeZone(CalendarConsts.UTC);
        assertEquals("2009W044", df.format(calendar.toDate()));
    }

    public void testISO8601WeekAndDayExtended() {
        DateFormat df = new SimpleDateFormat("yyyy-'W'ww-E", symbols);
        assertEquals("2009-W04-5", df.format(calendar.toDate()));
        df.setTimeZone(CalendarConsts.UTC);
        assertEquals("2009-W04-4", df.format(calendar.toDate()));
    }

    public void testRFC2822() {
        DateFormat df = new SimpleDateFormat(
                "EEEE, dd MMM yyyy HH:mm:ss Z", symbols);
        assertEquals("Fri, 23 Jan 2009 01:12:23 +0900",
                df.format(calendar.toDate()));
        df.setTimeZone(CalendarConsts.UTC);
        assertEquals("Thu, 22 Jan 2009 16:12:23 +0000",
                df.format(calendar.toDate()));
    }

}
