#
# test_LinearGraphTemplate.rb
#
# Copyright (C) 2010 GLAD!! (ITO Yoshiichi)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied. See the License for the specific language
# governing permissions and limitations under the License.
#

require 'test/unit'
require 'logger'
require 'LinearGraphTemplate'

class LinearGraphTemplateTest < Test::Unit::TestCase

  @@log = Logger.new(STDOUT)
  @@log.level = Logger::WARN
  @@log.formatter = proc {|s, d, p, m| sprintf("\n%-5s %s: %s", s, p, m) }

  def test_L4
    template = LinearGraphTemplate.new(4)
    @@log.debug('L4') { template }
    assert_equal(3, template.graphs.size)
    assert_equal(LinearGraph.new([1]), template.graphs[0])
    assert_equal(LinearGraph.new([2]), template.graphs[1])
    assert_equal(LinearGraph.new([3]), template.graphs[2])
    assert_equal([1, 1, 1], template.make_row(Array.new(3, 1)))
    assert_equal('L4: 2^3', template.summary)
    assert_equal('L4: [1], [2], [3]', template.to_s)
  end

  def test_L4_2
    template = LinearGraphTemplate.new(4, '2')
    @@log.debug('L4_2') { template }
    assert_equal(3, template.graphs.size)
    assert_equal(LinearGraph.new([1]), template.graphs[0])
    assert_equal(LinearGraph.new([2]), template.graphs[1])
    assert_equal(LinearGraph.new([3]), template.graphs[2])
    assert_equal([1, 1, 1], template.make_row(Array.new(3, 1)))
    assert_equal('L4: 2^3', template.summary)
    assert_equal('L4: [1], [2], [3]', template.to_s)
  end

  def test_L4_4
    template = LinearGraphTemplate.new(4, '4')
    @@log.debug('L4_4') { template }
    assert_equal(1, template.graphs.size)
    assert_equal(LinearGraph.new([1, 2]), template.graphs[0])
    assert_equal([3], template.make_row(Array.new(3, 1)))
    assert_equal('L4: 4^1', template.summary)
    assert_equal('L4: [1, 2; 3]', template.to_s)
  end

  def test_L8
    template = LinearGraphTemplate.new(8)
    @@log.debug('L8') { template }
    assert_equal(5, template.graphs.size)
    assert_equal(5, template.graphs.size)
    assert_equal(LinearGraph.new([1, 2]), template.graphs[0])
    assert_equal(LinearGraph.new([4]), template.graphs[1])
    assert_equal(LinearGraph.new([5]), template.graphs[2])
    assert_equal(LinearGraph.new([6]), template.graphs[3])
    assert_equal(LinearGraph.new([7]), template.graphs[4])
    assert_equal([3, 1, 1, 1, 1], template.make_row(Array.new(7, 1)))
    assert_equal('L8: 4^1, 2^4', template.summary)
    assert_equal('L8: [1, 2; 3], [4], [5], [6], [7]', template.to_s)
  end

  def test_L8_2
    template = LinearGraphTemplate.new(8, '2')
    @@log.debug('L8_2') { template }
    assert_equal(7, template.graphs.size)
    assert_equal(LinearGraph.new([1]), template.graphs[0])
    assert_equal(LinearGraph.new([2]), template.graphs[1])
    assert_equal(LinearGraph.new([3]), template.graphs[2])
    assert_equal(LinearGraph.new([4]), template.graphs[3])
    assert_equal(LinearGraph.new([5]), template.graphs[4])
    assert_equal(LinearGraph.new([6]), template.graphs[5])
    assert_equal(LinearGraph.new([7]), template.graphs[6])
    assert_equal([1, 1, 1, 1, 1, 1, 1], template.make_row(Array.new(7, 1)))
    assert_equal('L8: 2^7', template.summary)
    assert_equal('L8: [1], [2], [3], [4], [5], [6], [7]', template.to_s)
  end

  def test_L8_4
    template = LinearGraphTemplate.new(8, '4')
    @@log.debug('L8_4') { template }
    assert_equal(5, template.graphs.size)
    assert_equal(LinearGraph.new([1, 2]), template.graphs[0])
    assert_equal(LinearGraph.new([4]), template.graphs[1])
    assert_equal(LinearGraph.new([5]), template.graphs[2])
    assert_equal(LinearGraph.new([6]), template.graphs[3])
    assert_equal(LinearGraph.new([7]), template.graphs[4])
    assert_equal([3, 1, 1, 1, 1], template.make_row(Array.new(7, 1)))
    assert_equal('L8: 4^1, 2^4', template.summary)
    assert_equal('L8: [1, 2; 3], [4], [5], [6], [7]', template.to_s)
  end

  def test_L8_8
    template = LinearGraphTemplate.new(8, '8')
    @@log.debug('L8_8') { template }
    assert_equal(1, template.graphs.size)
    assert_equal(LinearGraph.new([1, 2, 4]), template.graphs[0])
    assert_equal([7], template.make_row(Array.new(7, 1)))
    assert_equal('L8: 8^1', template.summary)
    assert_equal('L8: [1, 2, 4; 3, 5, 6, 7]', template.to_s)
  end

  def test_L16
    template = LinearGraphTemplate.new(16)
    @@log.debug('L16') { template }
    assert_equal(5, template.graphs.size)
    assert_equal(LinearGraph.new([1,  2]), template.graphs[0])
    assert_equal(LinearGraph.new([4,  8]), template.graphs[1])
    assert_equal(LinearGraph.new([5, 10]), template.graphs[2])
    assert_equal(LinearGraph.new([6, 11]), template.graphs[3])
    assert_equal(LinearGraph.new([7,  9]), template.graphs[4])
    assert_equal([3, 3, 3, 3, 3], template.make_row(Array.new(15, 1)))
    assert_equal('L16: 4^5', template.summary)
    assert_equal('L16: [1, 2; 3], [4, 8; 12], [5, 10; 15], [6, 11; 13], [7, 9; 14]', template.to_s)
  end

  def test_L16_02
    template = LinearGraphTemplate.new(16, '2')
    @@log.debug('L16_02') { template }
    assert_equal(15, template.graphs.size)
    assert_equal(LinearGraph.new([ 1]), template.graphs[ 0])
    assert_equal(LinearGraph.new([ 2]), template.graphs[ 1])
    # ...
    assert_equal(LinearGraph.new([15]), template.graphs[14])
    assert_equal([1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
        template.make_row(Array.new(15, 1)))
    assert_equal('L16: 2^15', template.summary)
  end

  def test_L16_04
    template = LinearGraphTemplate.new(16, '4')
    @@log.debug('L16_04') { template }
    assert_equal(5, template.graphs.size)
    assert_equal(LinearGraph.new([1,  2]), template.graphs[0])
    assert_equal(LinearGraph.new([4,  8]), template.graphs[1])
    assert_equal(LinearGraph.new([5, 10]), template.graphs[2])
    assert_equal(LinearGraph.new([6, 11]), template.graphs[3])
    assert_equal(LinearGraph.new([7,  9]), template.graphs[4])
    assert_equal([3, 3, 3, 3, 3], template.make_row(Array.new(15, 1)))
    assert_equal('L16: 4^5', template.summary)
  end

  def test_L16_08
    template = LinearGraphTemplate.new(16, '8')
    @@log.debug('L16_08') { template }
    assert_equal(9, template.graphs.size)
    assert_equal(LinearGraph.new([1, 2, 4]), template.graphs[0])
    assert_equal(LinearGraph.new([ 8]), template.graphs[1])
    assert_equal(LinearGraph.new([ 9]), template.graphs[2])
    # ...
    assert_equal(LinearGraph.new([15]), template.graphs[8])
    assert_equal([7, 1, 1, 1, 1, 1, 1, 1, 1], template.make_row(Array.new(15, 1)))
    assert_equal('L16: 8^1, 2^8', template.summary)
  end

  def test_L16_16
    template = LinearGraphTemplate.new(16, '16')
    @@log.debug('L16_16') { template }
    assert_equal(1, template.graphs.size)
    assert_equal(LinearGraph.new([1, 2, 4, 8]), template.graphs[0])
    assert_equal([15], template.make_row(Array.new(15, 1)))
    assert_equal('L16: 16^1', template.summary)
  end

  def test_L32
    template = LinearGraphTemplate.new(32)
    @@log.warn('L32') { template }
    assert_equal(9, template.graphs.size)
    assert_equal(LinearGraph.new([1, 2, 4]), template.graphs[0])
    assert_equal(LinearGraph.new([ 8, 16]), template.graphs[1])
    # 「ソフトウェアテストHAYST法入門」では以下の組合せ。
    #assert_equal(LinearGraph.new([ 9, 19]), template.graphs[2])
    #assert_equal(LinearGraph.new([10, 20]), template.graphs[3])
    #assert_equal(LinearGraph.new([11, 23]), template.graphs[4])
    #assert_equal(LinearGraph.new([12, 17]), template.graphs[5])
    #assert_equal(LinearGraph.new([13, 18]), template.graphs[6])
    #assert_equal(LinearGraph.new([14, 21]), template.graphs[7])
    #assert_equal(LinearGraph.new([15, 22]), template.graphs[8])
    assert_equal(LinearGraph.new([ 9, 18]), template.graphs[2])
    assert_equal(LinearGraph.new([10, 20]), template.graphs[3])
    assert_equal(LinearGraph.new([11, 22]), template.graphs[4])
    assert_equal(LinearGraph.new([12, 19]), template.graphs[5])
    assert_equal(LinearGraph.new([13, 17]), template.graphs[6])
    assert_equal(LinearGraph.new([14, 23]), template.graphs[7])
    assert_equal(LinearGraph.new([15, 21]), template.graphs[8])
    assert_equal([7, 3, 3, 3, 3, 3, 3, 3, 3], template.make_row(Array.new(31, 1)))
    assert_equal('L32: 8^1, 4^8', template.summary)
  end

  def test_L64
    template = LinearGraphTemplate.new(64)
    @@log.debug('L64') { template }
    assert_equal(9, template.graphs.size)
    assert_equal(LinearGraph.new([ 1,  2,  4]), template.graphs[0])
    assert_equal(LinearGraph.new([ 8, 16, 32]), template.graphs[1])
    assert_equal(LinearGraph.new([ 9, 18, 36]), template.graphs[2])
    assert_equal(LinearGraph.new([10, 20, 35]), template.graphs[3])
    assert_equal(LinearGraph.new([11, 22, 39]), template.graphs[4])
    assert_equal(LinearGraph.new([12, 19, 38]), template.graphs[5])
    assert_equal(LinearGraph.new([13, 17, 34]), template.graphs[6])
    assert_equal(LinearGraph.new([14, 23, 37]), template.graphs[7])
    assert_equal(LinearGraph.new([15, 21, 33]), template.graphs[8])
    assert_equal([7, 7, 7, 7, 7, 7, 7, 7, 7], template.make_row(Array.new(63, 1)))
    assert_equal('L64: 8^9', template.summary)
  end

  def test_L128
    template = LinearGraphTemplate.new(128)
    @@log.debug('L128') { template }
    assert_equal([15, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7],
        template.make_row(Array.new(127, 1)))
    assert_equal('L128: 16^1, 8^16', template.summary)
  end

  def test_L256
    template = LinearGraphTemplate.new(256)
    @@log.debug('L256') { template }
    assert_equal([15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15],
        template.make_row(Array.new(255, 1)))
    assert_equal('L256: 16^17', template.summary)
  end

end
