// fnt_print.h

#ifndef __FNT_PRINT__
#define __FNT_PRINT__

// 標準では強制固定幅モードは利用しない
//#define USE_FIX

#include <psptypes.h>

/*
 * .fnt loadable font file format definition
 *
 * format               len         description
 * -------------------  ----------- -------------------------- ---------------------------------
 * u8 x4    version     4           'RB12'                     [0]
 * u16      maxwidth    2           最大幅                     [4]
 * u16      height      2           高さ                       [6]
 * u16      ascent      2           ベースライン               [8]
 * u8 x2    pad         2           パディング(32bit)          [10]
 * u32      firstchar   4           最初のフォント             [12]
 * u32      defaultchar 4           デフォルトのフォント       [16]
 * u32      size        4           フォント数                 [20]
 * u32      nbits       4           イメージデータサイズ       [24]
 * u32      noffset     4           オフセットデータサイズ *1  [28]
 * u32      nwidth      4           幅データサイズ*2           [32]
 * u8*      bits        nbits       イメージデータ             [36]
 * u8       pad         0～3        パディング(16/32bit)*3     [36 + nbits]
 * u16/32*  offset      noffsetx2/4 オフセットデータ*1*3       [36 + nbits + pad]
 * u8*      width       nwidth      幅データ*2                 [36 + nbits + pad + offset + pad]
 *
 * *1 noffsetが0の場合は固定幅となり、offsetデータは無し
 * *2 nwidthが0の場合は固定幅となり、widthデータは無し
 * *3 nbitsが0xFFDB未満の場合は16bit、0xFFDB以上の場合は32bitとなる
 *
 */

typedef struct {
    u16 maxwidth;       // 最大幅
    u16 height;         // 高さ
    u16 ascent;         // ベースライン(未使用)
    u32 firstchar;      // 最初のフォント
    u32 size;           // フォント数
    u32 defaultchar;    // デフォルトのフォント
    u32 nbits;          // イメージデータサイズ
    u32 noffset;        // オフセットデータサイズ
    u32 nwidth;         // 幅データサイズ
    const u8 *bits;     // イメージデータへのポインタ
    const u32 *offset;  // オフセットデータへのポインタ
    const u8 *width;    // 幅データへのポインタ
    u32 long_offset;    // オフセットデータ幅のフラグ
    u32 file_flag;      // ファイルからロードした場合のフラグ
    void *fnt_ptr;      // 確保したメモリへのポインタ
#ifdef USE_FIX
    u32 halfchar;       // 0～halfchar(UCS2)までのフォントは半角として扱う
    u32 fix_flag;       // 固定幅フラグ
    u32 fix_half;       // 固定幅時の半角幅
    u32 fix_wide;       // 固定幅時の全角幅
#endif
} fnt_t;

#define FNT_FONT_FILL	(0x01) // フォント部分をVRAMに書込む
#define FNT_BACK_FILL	(0x10) // BG部分をVRAMに書込む

/*---------------------------------------------------------------------------
  フォントをファイルからロード
    *path: フォントファイルのパス
    *font: font_tへのポインタ
    return: ファイルサイズ
            -1 指定ファイル無し / -2 メモリ確保失敗 / -3 読込失敗
---------------------------------------------------------------------------*/
s32 fnt_load_file(const void* path, fnt_t *font);

/*---------------------------------------------------------------------------
  メモリ上のフォントをロード
    *ptr: フォントデータへのポインタ
    *font: fnt_tへのポインタ
---------------------------------------------------------------------------*/
void fnt_load_mem(const void* ptr, fnt_t *font);

/*---------------------------------------------------------------------------
  フォントの解放
  ファイルからロードした場合は、メモリも解放する
    *font: fnt_tへのポインタ
    return: エラーの場合は負を返す
---------------------------------------------------------------------------*/
s32 fnt_free(fnt_t *font);

/*---------------------------------------------------------------------------
  文字列の幅を得る
    *font: font_tへのポインタ
    *str: 文字列へのポインタ(UTF-8N)
    return: 文字列の幅
---------------------------------------------------------------------------*/
u32 fnt_get_width(const fnt_t* font, const char *str, int mx);

/*---------------------------------------------------------------------------
  文字の幅を得る
    *font: font_tへのポインタ
    ucs2: 文字(UCS2)
    return: 文字の幅
---------------------------------------------------------------------------*/
u32 fnt_get_width_ucs2(const fnt_t* font, u32 ucs2, int mx);

/*---------------------------------------------------------------------------
  グリフデータを得る
    *font: font_tへのポインタ
    ucs2: 文字(UCS2)
    return: グリフデータへのポインタ
---------------------------------------------------------------------------*/
u8* fnt_get_bits(const fnt_t* font, u32 ucs2);

/*---------------------------------------------------------------------------
  一文字表示
    *font: font_tへのポインタ
    x: 横方向位置
    y: 縦方向位置
    *str: 文字列(UTF8-N)へのポインタ('\n'は改行を行う)
    color: 文字色
    back: 背景色
    fill: 書込フラグ FNT_FONT_FILL(0x01) 文字部, FNT_BACK_FILL(0x10) 背景部
    rate: 混合比(-100～100)
    mx: 横方向倍率
    my: 縦方向倍率
---------------------------------------------------------------------------*/
u32 fnt_print_xy(const fnt_t* font, u16 x, u16 y, void *str, u32 color,
    u32 back, u8 fill, s16 rate, u16 mx, u16 my, int x_length, int y_length);

/*---------------------------------------------------------------------------
  文字列表示(表示VRAMへの描画)
    *font: font_tへのポインタ
    x: 横方向位置
    y: 縦方向位置
    *str: 文字列(UTF8-N)へのポインタ('\n'は改行を行う)
    color: 文字色
    back: 背景色
    fill: 書込フラグ FNT_FONT_FILL(0x01) 文字部, FNT_BACK_FILL(0x10) 背景部
    rate: 混合比(-100～100)
    mx: 横方向倍率
    my: 縦方向倍率
---------------------------------------------------------------------------*/
u32 fnt_print_vram(const fnt_t* font, void *vram, u16 bufferwidth, u16 pixelformat,
    u16 x, u16 y, const void *str, u32 color, u32 back, u8 fill, s16 rate, u16 mx, u16 my, int x_length, int y_length);

/*---------------------------------------------------------------------------
  UFT-8Nの一文字をUTF-16に変換する
    *utf16: 変換後の文字(UTF-16)へのポインタ
    *utf8: UFT-8N文字へのポインタ
    return: UTF-8Nの次の文字へのポインタ
---------------------------------------------------------------------------*/
char* utf8_utf16(u16 *utf16, const char *utf8);

/*---------------------------------------------------------------------------
  UFT-8Nの文字列をUTF-16に変換する
    *utf16: 変換後の文字列(UTF-16)へのポインタ
    *utf8: UFT-8N文字列へのポインタ
---------------------------------------------------------------------------*/
void utf8_to_utf16(u16 *utf16, const char *utf8);

/*---------------------------------------------------------------------------
  UTF-16の文字列の長さを得る
    *utf16: 文字列(UTF-16)へのポインタ
    return: 文字数
---------------------------------------------------------------------------*/
u16 utf16len(const u16 *utf16);

#endif
