// /home/tarai/Projects/gsaw/gsaw/GenericSurface.cs created with MonoDevelop
// User: tarai at 17:43 2008/04/27
//
// To change standard headers go to Edit->Preferences->Coding->Standard Headers
//

using System;

namespace Holo.Image.Tiled {
	using Holo.Image;
	using Holo.Image.Generic;
	
	/**
	 * Tile-based surface implementation:
	 *  Surface has one-to-one mapping to Raster object.
	 * */
	public class TiledSurface : ISurface {
		public delegate void ReplaceTileHandler(int x, int y, IRaster oldTile, IRaster newTile);
		
		private const int DEFAULT_TILE_WIDTH = 64;
		private const int DEFAULT_TILE_HEIGHT = 64;
		
		private IRaster[] rasters;
		private IRaster defaultRaster;
		private int offsetX;
		private int offsetY;
		private int width;
		private int height;
		private ISurfaceUpdateAreaHandler onUpdateArea;
		private ReplaceTileHandler onReplaceTile;
		private IRasterFactory factory;
		
		public int TileMinX {
			get { return (offsetX / DEFAULT_TILE_WIDTH) * DEFAULT_TILE_WIDTH; }
		}
		
		public int TileMinY {
			get { return (offsetY / DEFAULT_TILE_HEIGHT) * DEFAULT_TILE_HEIGHT; }
		}
		
		public int TileMaxX {
			get { return ((offsetX + width + DEFAULT_TILE_WIDTH - 1) / DEFAULT_TILE_WIDTH) * DEFAULT_TILE_WIDTH; }
		}
		
		public int TileMaxY {
			get { return ((offsetY + height + DEFAULT_TILE_HEIGHT - 1) / DEFAULT_TILE_HEIGHT) * DEFAULT_TILE_HEIGHT; }
		}
		
		public int TileWidth {
			get { return DEFAULT_TILE_WIDTH; }
		}
		
		public int TileHeight {
			get { return DEFAULT_TILE_HEIGHT; }
		}
		
		protected int TileColumnCount {
			get { return (TileMaxX - TileMinX) / DEFAULT_TILE_WIDTH; }
		}
		
		protected int TileRowCount {
			get { return (TileMaxY - TileMinY) / DEFAULT_TILE_HEIGHT; }
		}
		
		public IRaster DefaultRaster {
			get { return defaultRaster; }
		}
		
		public int TileGridXForColumn(int c) {
			return TileMinX + DEFAULT_TILE_WIDTH * c;
		}
		
		public int TileGridYForRow(int r) {
			return TileMinY + DEFAULT_TILE_HEIGHT * r;
		}
		
		public int TileGridXAt(int x) {
			return TileGridXForColumn(TileColumnAt(x));
		}
		
		public int TileGridYAt(int y) {
			return TileGridYForRow(TileRowAt(y));
		}
		
		public int TileColumnAt(int x) {
			return (x - TileMinX) / DEFAULT_TILE_WIDTH;
		}
		
		public int TileRowAt(int y) {
			return (y - TileMinY) / DEFAULT_TILE_HEIGHT;
		}
		
		public int TileOffset(int c, int r) {
			return TileColumnCount * r + c;
		}
		
		public IRasterFactory RasterFactory {
			get { return factory; }
			set { 
				if (value != null) {
					factory = value; 
					defaultRaster = factory.CreateNewRaster(DEFAULT_TILE_WIDTH, DEFAULT_TILE_HEIGHT);
					defaultRaster.MustBeCopiedOnWrite = true;
				}
			}
		}
		
		public TiledSurface(int x, int y, int width, int height, IRasterFactory factory) {
			this.factory = factory;
			IRaster defaultRaster = factory.CreateNewRaster(DEFAULT_TILE_WIDTH, DEFAULT_TILE_HEIGHT);
			Initialize(x, y, width, height, defaultRaster);
		}
		
		public TiledSurface(TiledSurface src) {
			this.factory = src.factory;
			this.defaultRaster = src.defaultRaster;
			this.offsetX = src.offsetX;
			this.offsetY = src.offsetY;
			this.width = src.width;
			this.height = src.height;
			this.rasters = (IRaster[])src.rasters.Clone();
		}
		
		public void Initialize(int x, int y, int width, int height, IRaster defaultRaster) {
			offsetX = x;
			offsetY = y;
			this.width = width;
			this.height = height;
			rasters = new IRaster[TileRowCount * TileColumnCount];
			this.defaultRaster = defaultRaster;
			defaultRaster.MustBeCopiedOnWrite = true;
		}
				
		public int OffsetX {
			get { return offsetX; }
			set { /* TBD (or will be removed.) */ }
		}
		
		public int OffsetY {
			get { return offsetY; }
			set { /* TBD (or will be removed.) */ }
		}
		
		public int Width {
			get { return width; }
		}
		
		public int Height {
			get { return height; }
		}
		
		public void MakeRastersReadOnly(int x, int y, int width, int height) {
			int col1 = TileColumnAt(x);
			int row1 = TileRowAt(y);
			int col2 = TileColumnAt(x + width - 1);
			int row2 = TileRowAt(y + height - 1);
			for (int r = row1; r <= row2; r ++) {
				for (int c = col1; c <= col2; c ++) {
					IRaster raster = rasters[TileOffset(c, r)];
					if (raster == null)
						raster = defaultRaster;
					if (raster != defaultRaster)
						raster.MustBeCopiedOnWrite = true;
				}
			}
		}
		
		public void MakeRastersWriteable(int x, int y, int width, int height) {
			int col1 = TileColumnAt(x);
			int row1 = TileRowAt(y);
			int col2 = TileColumnAt(x + width - 1);
			int row2 = TileRowAt(y + height - 1);
			for (int r = row1; r <= row2; r ++) {
				for (int c = col1; c <= col2; c ++) {
					int offset = TileOffset(c, r);
					IRaster raster = rasters[offset];
					if (raster == null || raster.MustBeCopiedOnWrite) {
						// Swap surface with new one.
						if (raster == null)
							rasters[offset] = factory.CreateCopyOfRaster(defaultRaster);
						else
							rasters[offset] = factory.CreateCopyOfRaster(raster);
						
						rasters[offset].MustBeCopiedOnWrite = false;
						// Notify swap event here.
//						Console.WriteLine("Swap Tile:"+c+","+r+",reason="+(raster == null? "NULL":"CoW"));
						if (onReplaceTile != null)
							OnReplaceTile(TileGridXForColumn(c), TileGridYForRow(r), raster, rasters[offset]);
					}
				}
			}
		}
		
		public void MakeRastersReadable(int x, int y, int width, int height) {			
			int col1 = TileColumnAt(x);
			int row1 = TileRowAt(y);
			int col2 = TileColumnAt(x + width - 1);
			int row2 = TileRowAt(y + height - 1);
			for (int r = row1; r <= row2; r ++) {
				for (int c = col1; c <= col2; c ++) {
//					Console.WriteLine("Readable:(x,y)=("+x+","+y+"),(Txmin,Tymin)=("+TileMinX+","+TileMinY+"),(Txmax,Tymax)=("+TileMaxX+","+TileMaxY+"),(c,r)=("+c+","+r+"),(cmax,rmax)="+TileColumnCount+","+TileRowCount+",offset="+TileOffset(c, r));
					IRaster raster = rasters[TileOffset(c, r)];
					if (raster == null)
						rasters[TileOffset(c, r)] = defaultRaster;
				}
			}
		}
		
		public void ClearRasters() {
			ClearRasters(offsetX, offsetY, offsetX + width, offsetY + height);
		}
		
		public void ClearRasters(int x, int y, int width, int height) {			
			int col1 = TileColumnAt(x);
			int row1 = TileRowAt(y);
			int col2 = TileColumnAt(x + width - 1);
			int row2 = TileRowAt(y + height - 1);
			for (int r = row1; r <= row2; r ++) {
				for (int c = col1; c <= col2; c ++) {
					rasters[TileOffset(c, r)] = null;
				}
			}
		}
		
		public void FillRasters(int x, int y, int width, int height, byte[] color, byte[] alpha) {
			int col1 = TileColumnAt(x);
			int row1 = TileRowAt(y);
			int col2 = TileColumnAt(x + width - 1);
			int row2 = TileRowAt(y + height - 1);
			bool leftAligned = TileGridXAt(x) == x;
			bool rightAligned = TileGridXAt(x + width) == x + width;
			bool topAligned = TileGridYAt(y) == y;
			bool bottomAligned = TileGridYAt(y + height) == y + height;
			IRaster filledRaster = null;
			for (int r = row1; r <= row2; r ++) {
				bool wholeRowFilled = (r == row1 && !topAligned || r == row2 && !bottomAligned);
				for (int c = col1; c <= col2; c ++) {
					bool wholeColFilled = (c == col1 && !leftAligned || c == col2 && !rightAligned);
					
					int offset = TileOffset(c, r);
					IRaster raster = rasters[offset];
					if (wholeColFilled && wholeRowFilled) {
						if (filledRaster == null) {
							// Create new raster buffer that is filled with specified color and alpha value.
							filledRaster = factory.CreateNewRaster(DEFAULT_TILE_WIDTH, DEFAULT_TILE_HEIGHT);
							if (filledRaster.HasColorChannels && filledRaster.HasAlpha) {
								// TBD.
							} else if (filledRaster.HasAlpha) {
								// TBD.
							} else if (filledRaster.HasColorChannels) {
								// TBD.
							} else {
								Array.Copy(color, 0, filledRaster.PrimaryColor, 0, color.Length);
								Array.Copy(alpha, 0, filledRaster.PrimaryAlphaValue, 0, alpha.Length);
							}
							filledRaster.MustBeCopiedOnWrite = false;
						}
						rasters[offset] = filledRaster;
						if ((raster == null || raster.MustBeCopiedOnWrite) && onReplaceTile != null)
							OnReplaceTile(TileGridXForColumn(c), TileGridYForRow(r), raster, rasters[offset]);
					} else {
						if (raster == null || raster.MustBeCopiedOnWrite) {
							// Swap surface with new one.
							if (raster == null)
								rasters[offset] = factory.CreateCopyOfRaster(defaultRaster);
							else
								rasters[offset] = factory.CreateCopyOfRaster(raster);
							
							rasters[offset].MustBeCopiedOnWrite = false;
							// Notify swap event here.
	//						Console.WriteLine("Swap Tile:"+c+","+r+",reason="+(raster == null? "NULL":"CoW"));
							if (onReplaceTile != null)
								OnReplaceTile(TileGridXForColumn(c), TileGridYForRow(r), raster, rasters[offset]);
						}
						// TBD. area must be filled
					}
				}
			}
		}

		public ISurfaceIterator GetIteratorAt(int x, int y) {
			return new TiledSurfaceIterator(this, x, y);
		}
		
		public void SetRasterAt(ISurfaceIterator iter, IRaster raster) {
			if (iter.OffsetX >= TileMinX && iter.OffsetX < TileMaxX &&
			    iter.OffsetY >= TileMinY && iter.OffsetY < TileMaxY &&
			    raster != null && raster.Width == DEFAULT_TILE_WIDTH && raster.Height == DEFAULT_TILE_HEIGHT) {
				int c = TileColumnAt(iter.OffsetX);
				int r = TileRowAt(iter.OffsetY);
//				Console.WriteLine("SetRasterAt:set"+c+","+r);
				rasters[TileOffset(c, r)] = raster;
			}
		}

		public void Resize(int width, int height) {
			// TBD
		}
		
		public IRaster[] Rasters {
			get { return rasters; }
		}
		
		public ISurfaceUpdateAreaHandler OnUpdateArea {
			get { return onUpdateArea; }
			set { onUpdateArea = value; }
		}
		
		public ReplaceTileHandler OnReplaceTile {
			get { return onReplaceTile; }
			set { onReplaceTile = value; }
		}
	}
}