// /home/tarai/Projects/gsaw/gsaw/TiledSurfaceIterator.cs created with MonoDevelop
// User: tarai at 17:43 2008/04/27
//
// To change standard headers go to Edit->Preferences->Coding->Standard Headers
//

using System;

namespace Holo.Image.Tiled {
	using Holo.Image;
	
	public class TiledSurfaceIterator : ISurfaceIterator {
		protected int minX;
		protected int minY;
		protected int maxX;
		protected int maxY;
		protected int offsetX;
		protected int offsetY;
		protected int tileGridX;
		protected int tileGridY;
		protected int lastWidth;
		protected int lastHeight;
		protected TiledSurface surface;
		protected IRaster raster;
		
		public TiledSurfaceIterator(TiledSurface surface, int offsetX, int offsetY) {
			this.surface = surface;
			this.minX = surface.OffsetX;
			this.minY = surface.OffsetY;
			this.maxX = surface.OffsetX + surface.Width;
			this.maxY = surface.OffsetY + surface.Height;
			this.offsetX = offsetX;
			this.offsetY = offsetY;
			UpdateRaster();
		}
		
		public TiledSurfaceIterator(TiledSurface surface, int minX, int minY, int maxX, int maxY, int offsetX, int offsetY) {
			this.surface = surface;
			this.minX = minX;
			this.minY = minY;
			this.maxX = maxX;
			this.maxY = maxY;
			this.offsetX = offsetX;
			this.offsetY = offsetY;
			UpdateRaster();
		}
		
		public TiledSurfaceIterator(TiledSurfaceIterator src) {
			this.surface = src.surface;
			this.minX = src.minX;
			this.minY = src.minY;
			this.maxX = src.maxX;
			this.maxY = src.maxY;
			this.offsetX = src.offsetX;
			this.offsetY = src.offsetY;
			if (this.offsetX < this.minX)
				this.offsetX = this.minX;
			if (this.offsetY < this.minY)
				this.offsetY = this.minY;
			UpdateRaster();
		}
		
		public TiledSurfaceIterator(TiledSurfaceIterator src, int minX, int minY, int maxX, int maxY) {
			this.surface = src.surface;
			this.minX = minX;
			this.minY = minY;
			this.maxX = maxX;
			this.maxY = maxY;
			this.offsetX = src.offsetX;
			this.offsetY = src.offsetY;
			if (this.offsetX < this.minX)
				this.offsetX = this.minX;
			if (this.offsetY < this.minY)
				this.offsetY = this.minY;
			UpdateRaster();
		}
		
		public void SetRange(int minX, int minY, int maxX, int maxY) {
			this.minX = minX;
			this.minY = minY;
			this.maxX = maxX;
			this.maxY = maxY;
		}

		public int OffsetX {
			get { return offsetX; }
		}
		
		public int OffsetY {
			get { return offsetY; }
		}
		
		public IRaster Raster {
			get { return raster; }
		}
		
		public int OffsetXInRaster {
			get { return offsetX - tileGridX; }
		}
		
		public int OffsetYInRaster {
			get { return offsetY - tileGridY; }
		}
		
		public int WidthOfRaster {
			get { return (raster != null)? raster.Width - OffsetXInRaster : lastWidth; }
		}
		
		public int HeightOfRaster {
			get { return (raster != null)? raster.Height - OffsetYInRaster : lastHeight; }
		}
		
		public int OffsetInBuffer {
			get { return raster.RowStride * OffsetYInRaster + raster.PixelStride * OffsetXInRaster; }
		}
		
		public void AddX(int x) {
			offsetX += x;
			UpdateRaster();
		}
		
		public void SubX(int x) {
			offsetX -= x;
			UpdateRaster();
		}

		public void AddY(int y) {
			offsetY += y;
			UpdateRaster();
		}
		
		public void SubY(int y) {
			offsetY -= y;
			UpdateRaster();
		}

		public void ResetXAndAddY(int y) {
			offsetX = minX;
			offsetY += y;
			UpdateRaster();
		}

		public bool IsXEnded() {
			return offsetX >= maxX;
		}
		
		public bool IsXBelowMin() {
			return offsetX < minX;
		}

		public bool IsYEnded() {
			return offsetY >= maxY;			
		}
		
		public bool IsYBelowMin() {
			return offsetY < minY;
		}
		
		public void UpdateRaster() {
			if (Raster != null) {
				lastWidth = Raster.Width;
				lastHeight = Raster.Height;
			}
			
			if (offsetX >= surface.TileMinX && offsetX < surface.TileMaxX && offsetY >= surface.TileMinY && offsetY < surface.TileMaxY) {
				int c = surface.TileColumnAt(offsetX);
				int r = surface.TileRowAt(offsetY);
				raster = surface.Rasters[surface.TileOffset(c, r)];
				tileGridX = surface.TileGridXAt(offsetX);
				tileGridY = surface.TileGridYAt(offsetY);
//				Console.WriteLine("Tile({0}, {1})[Grid:{2},{3}]={4}", tileGridX, tileGridY, c, r, raster);
			} else {
//				Console.WriteLine("TiledSurfaceIterator:error: indexer is out of range at "+offsetX+","+offsetY+": range="+minX+","+minY+"-"+maxX+","+maxY);
				raster = null;
			}
		}
		
		public ISurfaceIterator Clone() {
			return new TiledSurfaceIterator(this);
		}
		
		public bool GetCurrentColor(out byte[] color, out int offset) {
			if (Raster == null || IsXBelowMin() || IsYBelowMin() || IsXEnded() || IsYEnded()) {
				color = null;
				offset = 0;
				return false;
			}
			if (Raster.HasColorChannels) {
				color = Raster.Buffer;
				offset = OffsetInBuffer;
			} else {
				color = Raster.PrimaryColor;
				offset = 0;
			}
			return true;
		}
		
		public bool GetCurrentAlpha(out byte[] alpha, out int offset) {
			if (Raster == null || IsXBelowMin() || IsYBelowMin() || IsXEnded() || IsYEnded()) {
				alpha = null;
				offset = 0;
				return false;
			}
			if (Raster.HasAlpha) {
				alpha = Raster.Buffer;
				offset = OffsetInBuffer + Raster.NumChannels - 1;
			} else {
				alpha = Raster.PrimaryAlphaValue;
				offset = 0;
			}
			return true;
		}
	}
}
