﻿Option Strict On
Option Explicit On

Imports System.IO
Imports System.Runtime.InteropServices.ComTypes
Imports System.Runtime.InteropServices

''' <summary>アイコン情報格納クラス。</summary>
''' <remarks>
''' アイコン情報の格納するクラス。
''' </remarks>
Public NotInheritable Class IconInfo

    ' 列番号
    Private mColumn As Integer

    ' 行番号
    Private mRow As Integer

    ' アイコン名
    Private mName As String

    ' ショートカットパス
    Private mPath As String

    Public Shared Function CreateNewIcon(column As Integer, row As Integer,
                                         path As String, linkFolder As String) As IconInfo
        Dim nm As String = String.Empty
        If File.Exists(path) Then
            nm = New FileInfo(path).Name
            nm = nm.Substring(0, nm.LastIndexOf("."))
        End If
        Return New IconInfo(column, row, nm, CopyShortCut(path, linkFolder))
    End Function

    ' ''' <summary>コンストラクタ。</summary>
    ' ''' <param name="column">列番号。</param>
    ' ''' <param name="row">行番号。</param>
    ' ''' <param name="path">ショートカットパス。</param>
    'Public Sub New(column As Integer,
    '               row As Integer, path As String)
    '    ' 各情報を格納する
    '    Me.mColumn = column
    '    Me.mRow = row
    '    Me.mPath = CopyShortCut(path)
    '    Me.mName = String.Empty

    '    ' ファイル情報から名前を取得する
    '    If File.Exists(path) Then
    '        Me.mName = New FileInfo(path).Name
    '        Me.mName = Me.mName.Substring(0, Me.mName.LastIndexOf("."))
    '    End If
    'End Sub

    ''' <summary>コンストラクタ。</summary>
    ''' <param name="column">列番号。</param>
    ''' <param name="row">行番号。</param>
    ''' <param name="name">アイコン名。</param>
    ''' <param name="path">ショートカットパス。</param>
    Public Sub New(column As Integer, row As Integer,
                   name As String, path As String)
        Me.mColumn = column
        Me.mRow = row
        Me.mPath = path
        Me.mName = name
    End Sub

    ''' <summary>引数のアイコン情報と情報の入れ替えを行う。</summary>
    ''' <param name="other">入れ替えるアイコン情報。</param>
    ''' <remarks>
    ''' アイコン情報の入れ替えを行う。
    ''' </remarks>
    Public Sub Swap(other As IconInfo)
        Dim tmp As String

        ' 名前情報を入れ替える
        tmp = Me.mName
        Me.mName = other.mName
        other.mName = tmp

        ' ショートカット情報を入れ替える
        tmp = Me.mPath
        Me.mPath = other.mPath
        other.mPath = tmp
    End Sub

    ''' <summary>このアイコン情報の移動を行う。</summary>
    ''' <param name="newColumn">新しい列位置。</param>
    ''' <param name="newRow">新しい行位置。</param>
    ''' <remarks>
    ''' アイコン情報の移動を行う。
    ''' </remarks>
    Public Sub Move(newColumn As Integer, newRow As Integer)
        Me.mColumn = newColumn
        Me.mRow = newRow
    End Sub

    ''' <summary>情報のショートカットパスを更新する。</summary>
    ''' <param name="dropPath">新しいショートカットパス。</param>
    Public Sub UpdatePath(dropPath As String, linkFolder As String)
        Me.mPath = CopyShortCut(dropPath, linkFolder)
        If File.Exists(dropPath) Then
            Me.mName = New FileInfo(dropPath).Name
            Me.mName = Me.mName.Substring(0, Me.mName.LastIndexOf("."))
        End If
    End Sub

    Private Shared Function CopyShortCut(dropPath As String, linkFolder As String) As String
        Dim srcFile As New FileInfo(dropPath)
        Dim res As String = dropPath
        Dim nam As String = srcFile.Name

        If System.Text.RegularExpressions.Regex.IsMatch(dropPath, "exe$") Then
            Dim shLink = New ShellLinkObject()
            Dim ipFile As IPersistFile = CType(shLink, IPersistFile)

            Try
                With CType(shLink, IShellLink)
                    .SetPath(dropPath)
                End With

                res = String.Format("{0}\{1}.lnk", linkFolder, nam.Substring(0, nam.LastIndexOf(".")))
                With CType(shLink, IPersistFile)
                    .Save(res, True)
                End With
            Catch ex As Exception
                MessageBox.Show("リンク作成失敗、やり直してください", "エラー", MessageBoxButton.OK, MessageBoxImage.Error)

            Finally
                Marshal.ReleaseComObject(ipFile)
                Marshal.ReleaseComObject(shLink)
            End Try

        ElseIf System.Text.RegularExpressions.Regex.IsMatch(dropPath, "lnk$") Then
            res = String.Format("{0}\{1}", linkFolder, nam)
            srcFile.CopyTo(res, True)
        End If

        Return res
    End Function

#Region "properties"

    ''' <summary>列番号。</summary>
    ''' <value>整数値。</value>
    Public ReadOnly Property Column() As Integer
        Get
            Return Me.mColumn
        End Get
    End Property

    ''' <summary>行番号。</summary>
    ''' <value>整数値。</value>
    Public ReadOnly Property Row() As Integer
        Get
            Return Me.mRow
        End Get
    End Property

    ''' <summary>アイコン名。</summary>
    ''' <value>文字列型。</value>
    Public Property Name() As String
        Get
            Return Me.mName
        End Get
        Set(value As String)
            Me.mName = value
        End Set
    End Property

    ''' <summary>ショートカットパス。</summary>
    ''' <value>文字列型。</value>
    Public ReadOnly Property Path() As String
        Get
            Return Me.mPath
        End Get
    End Property

#End Region

End Class
