/*
 
Copyright (C) 2009 NTT DATA INTELLILINK CORPORATION
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package jp.co.intellilink.hinemos.export.conf.monitor;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;

import jp.co.intellilink.hinemos.util.Config;
import jp.co.intellilink.hinemos.util.EjbConnectionManager;
import jp.co.intellilink.hinemos.util.Messages;
import jp.co.intellilink.hinemos.util.StringListComparator;
import jp.co.intellilink.hinemos.util.WriteCsvFile;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PortRunCountConstant;
import com.clustercontrol.bean.PortRunIntervalConstant;
import com.clustercontrol.bean.ProtocolConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.port.bean.MonitorPortInfo;
import com.clustercontrol.port.ejb.session.MonitorPortController;

/**
 * サービス・ポート監視情報を取得するクラス<br>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class ExportMonitorPort extends ExportMonitorNumeric {

	// ログ出力
	private static Log log = LogFactory.getLog(ExportMonitorPort.class);
	
    public ExportMonitorPort() {
		super(ExportMonitorNumeric.INPUT_VERIFICATION_POSITIVE_INTEGER);
	}

	/**
     * サービス・ポート監視情報をExportする。<br>
     */
    public void exportPort() {
    	log.info(Messages.getMsg("Port.Start", new String[]{Messages.getMsg("ExportTool.Export")}));
    	
    	//CSV出力配列を用意
    	ArrayList<ArrayList<String>> csvOutput = new ArrayList<ArrayList<String>>();
    	
    	//監視項目リスト取得
    	ArrayList idList = getPortList();
    	
    	if(idList instanceof ArrayList){
			Iterator itr = idList.iterator();
			while(itr.hasNext()){
				ArrayList line = (ArrayList)itr.next();
				
				Object[] args = {(String)line.get(0)};
				log.info(Messages.getMsg("MonitorID", args));
				
				MonitorInfo info = getPort((String)line.get(0));
				csvOutput.add(createCsvFormat(info));
			}
			
			//監視項目IDでソート
			Collections.sort(csvOutput, new StringListComparator(0));
    	}
    	
		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd_HHmmss");
		
		WriteCsvFile.writeCsvFile(
				Config.getConfig("Export.Dir") + "/PORT_" + format.format(new Date()) + ".csv", 
				csvOutput);
		
		log.info(Messages.getMsg("Port.End", new String[]{Messages.getMsg("ExportTool.Export")}));
    }
    
    /**
	 * サービス・ポート監視情報をExportする。<br>
	 * 
	 * @param monitorInfo サービス・ポート監視情報
	 * @return CSVフォーマット文字列
	 */
    @Override
	public ArrayList<String> createMonitorInfo(MonitorInfo monitorInfo) {
		ArrayList<String> csvFormat = new ArrayList<String>();
		
        MonitorPortInfo portInfo = (MonitorPortInfo)monitorInfo.getCheckInfo();
        if(portInfo == null){
        	portInfo = new MonitorPortInfo();
        }
        //TCP・サービスプロトコル
        if(portInfo.getServiceId() == null || portInfo.getServiceId().equals(ProtocolConstant.TYPE_PROTOCOL_TCP))
        	csvFormat.add(com.clustercontrol.util.Messages.getString("tcp.connect.only"));
        else
        	csvFormat.add("");
        //ポート番号
        if (portInfo.getPortNo() != null)
        	csvFormat.add(portInfo.getPortNo().toString());
        else
        	csvFormat.add("");
        //試行回数
        if (portInfo.getRunCount() != null)
        	csvFormat.add(PortRunCountConstant.typeToString(portInfo.getRunCount().intValue()));
        else
        	csvFormat.add("");
        //試行間隔
        if (portInfo.getRunInterval() != null)
        	csvFormat.add(PortRunIntervalConstant.typeToString(portInfo.getRunInterval().intValue()));
        else
        	csvFormat.add("");
        //タイムアウト
        if (portInfo.getTimeout() != null)
        	csvFormat.add(portInfo.getTimeout().toString());
        else
        	csvFormat.add("");
        
		return csvFormat;
	}

    /**
     * サービス・ポート監視情報を取得する。<br>
     * 
     * @param monitorId 取得対象の監視項目ID
     * @return サービス・ポート監視情報
     */
    protected MonitorInfo getPort(String monitorId) {

    	MonitorPortController port = EjbConnectionManager.getConnectionManager().getMonitorPortController();

        MonitorInfo info = null;
        try {
        	info = port.getPort(monitorId);
        } catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
        }
        return info;
    }
    
    /**
     * 監視項目リストを取得する。<br>
     * 
     * @return 監視項目一覧
     */
    protected ArrayList getPortList() {

    	MonitorPortController port = EjbConnectionManager.getConnectionManager().getMonitorPortController();

    	ArrayList records = null;
    	try {
    		records = port.getPortList();
        } catch (Exception e) {
			log.error(Messages.getMsg("ConnectManagerFailed"), e);
			System.exit(14);
        }
        return records;
    }
}